package com.avaya.generic.channel.testclient.utils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Pattern;

public class Utils {

    public static boolean isNullOrEmpty(String s) {
        return s == null || s.isEmpty();
    }

    public static boolean isNotEmpty(String s) {
        return s != null && !s.isEmpty();
    }

    /**
     * For correctness of the URLs.
     * This is required mainly because we have host names and IP addresses instead of URLs in the properties file.
     *
     * Joins the parts on slashes.
     * Adds "http://" at the beginning if the first part does not start from "http://" or "https://".
     *
     * @param parts
     * @return the url
     */
    public static String buildUrl(String... parts) {

        final String url = joinUrlParts(parts);

        // add the protocol if necessary
        if (httpPattern.matcher(url).lookingAt() || httpSecurePattern.matcher(url).lookingAt()) {
            return url;
        } else {
            return "http://" + url;
        }
    }

    /**
     * The difference from the previous method is that the returned string always starts with "https://"
     * @param parts
     * @return
     */
    public static String buildHttpSecureUrl(String... parts) {

        final String url = joinUrlParts(parts);

        // add the protocol if necessary
        if (httpPattern.matcher(url).lookingAt()) {
            return httpPattern.matcher(url).replaceFirst("https://");
        } else if (httpSecurePattern.matcher(url).lookingAt()) {
            return url;
        } else {
            return "https://" + url;
        }
    }

    // console utils

    public static String readLine() throws IOException {

        BufferedReader reader = new BufferedReader(new InputStreamReader(System.in));
        lastCall.set(LastCall.READ_LINE);
        String line = reader.readLine();
        if (line != null && !line.trim().isEmpty()) {
            return line.trim();
        } else {
            lastCall.set(LastCall.READ_EMPTY_LINE);
            return "";
        }
    }

    public static synchronized void print(String format, Object ... args) {
        if (format != null) {
            System.out.printf(format, args);
        } else {
            System.out.print("null");
        }
        lastCall.set(LastCall.PRINT);
    }

    public static synchronized void print(String s) {
        if (s != null) {
            System.out.print(s);
        } else {
            System.out.print("null");
        }
        lastCall.set(LastCall.PRINT);
    }

    // println() is print() plus \n

    public static synchronized void println(String format, Object ... args) {
        print(format, args);
        System.out.println();
        lastCall.set(LastCall.PRINT_LN);
    }

    public static synchronized void println(String s) {
        print(s);
        System.out.println();
        lastCall.set(LastCall.PRINT_LN);
    }

    public static synchronized void println() {
        System.out.println();
        lastCall.set(LastCall.PRINT_LN);
    }

    /**
     * Guarantees to print one and only one empty line. Consequent calls do nothing.
     * It is recommanded to call this before and after you print something big or important.
     *
     * Currently, requests and exceptions are wrapped with separators,
     * and also there are separators between commands.
     */
    public static synchronized void printSeparator() {
        final LastCall localLastCall = lastCall.get();
        if (localLastCall != null) {
            switch (localLastCall) {
                case PRINT:
                    System.out.println();
                    System.out.println();
                    break;
                case PRINT_LN:
                case READ_LINE:
                    System.out.println();
                    break;
                case READ_EMPTY_LINE:
                case NONE:
                default:
                    break;
            }
        }
        lastCall.set(LastCall.NONE);
    }

    /**
     * convenience method for error printing
     */
    public static synchronized void printException(Exception e) {
        printSeparator();
        if (isNotEmpty(e.getMessage())) {
            if (e instanceof ExternalException) {
                println(e.getMessage());
            } else {
                final String className = e.getClass().getSimpleName();
                println("%s: %s", className, e.getMessage());
            }
        } else {
            lastCall.set(LastCall.PRINT_LN);
        }
        printSeparator();
    }

    /**
     * The version of the above method for asynchronous output
     */
    public static synchronized void printException(String producer, Exception e) {
        printSeparator();
        if (isNotEmpty(e.getMessage())) {
            if (e instanceof ExternalException) {
                println("%s: %s", producer, e.getMessage());
            } else {
                final String className = e.getClass().getSimpleName();
                println("%s: %s: %s", producer, className, e.getMessage());
            }
        } else {
            print("%s: ", producer);
            lastCall.set(LastCall.PRINT_LN);
        }
        printSeparator();
    }


    /**
     * Used to cut the ssoToken when printing
     */
    public static String trimTo16chars(String s) {
        if (s == null) {
            return null;
        } else if (s.length() > 16) {
            return s.substring(0, 16) + "...";
        } else {
            return s;
        }
    }

    //=================================== private ===============================================

    private static String joinUrlParts(String[] parts) {

        if (parts.length == 0) {
            return "";
        }

        StringBuilder sb = new StringBuilder(256);
        for (String part : parts) {
            if (!Utils.isNullOrEmpty(sb.toString())) {
                sb.append("/");
            }
            sb.append(part.trim().replaceAll("(\\\\|/)+$", "").replaceAll("^(\\\\|/)+", ""));
        }

        return sb.toString();
    }

    private static final Pattern httpPattern = Pattern.compile("^http://", Pattern.CASE_INSENSITIVE);
    private static final Pattern httpSecurePattern = Pattern.compile("^https://", Pattern.CASE_INSENSITIVE);

    private static enum LastCall { PRINT, PRINT_LN, READ_LINE, READ_EMPTY_LINE, NONE }
    private static final AtomicReference<LastCall> lastCall = new AtomicReference(LastCall.NONE);

    private Utils() {
    }
}
