 //
 //  Logging.m
 //  Oceana™ReferenceClient
 //
 //  Copyright 2016 Avaya Inc
 //  All rights reserved. Usage of this source is bound to the terms described the file
 //  Avaya SDK EULA.txt, included in this SDK.
 //  Avaya – Confidential & Proprietary. Use pursuant to your signed agreement or Avaya Policy.
 //
 //

#import <Foundation/Foundation.h>
#import "Logging.h"
#import <CocoaLumberjack/CocoaLumberjack.h>
#import <CocoaLumberjack/DDLog.h>
#import <CocoaLumberjack/DDLogMacros.h>
#import "LogFormatter.h"
#import "asl.h"
#import "AppSettings.h"
#import <OceanaCustomerWebVoiceVideo/CustomLog.h>

#define SYSTEM_VERSION_GREATER_THAN_OR_EQUAL_TO(v)  ([[[UIDevice currentDevice] systemVersion] compare:v options:NSNumericSearch] != NSOrderedAscending)

@implementation Logging 
int static LOG_LEVEL;

int const LOG_LEVEL_ERROR = 0;
int const LOG_LEVEL_WARN = 1;
int const LOG_LEVEL_INFO = 2;
int const LOG_LEVEL_DEBUG = 3;

char* const APP_NAME = "OceanaReferenceClient";
NSString* const APP_BUNDLE = @"com.avaya.OceanaReferenceClient";

char* const ASL_TIME_KEY = "CFLog Local Time";
char* const ASL_SENDER_KEY = "Sender";
char* const ASL_LEVEL_KEY = "Level";
char* const ASL_MESSAGE_KEY = "Message";
char* const ASL_MESSAGE_ID_KEY = "ASLMessageID";

//This setting sets the level at which redirected log messages are logged at,
//as NSLog does not provide a log level parameter.
NSUInteger const SDK_LOG_LEVEL = DDLogLevelAll;

//NSUInteger ddLogLevel = DDLogLevelAll;
#ifdef DEBUG
static const DDLogLevel ddLogLevel = DDLogLevelAll;
#else
static const DDLogLevel ddLogLevel = DDLogLevelAll;
#endif


DDFileLogger* fileLogger;

DDLogLevel logLevelForPref(int pref){
    DDLogLevel level = DDLogLevelInfo;
    if(pref == LOG_LEVEL_ERROR){
        level = DDLogLevelError;
    } else if(pref == LOG_LEVEL_WARN){
        level = DDLogLevelWarning;
    } else if(pref == LOG_LEVEL_INFO){
        level = DDLogLevelInfo;
    } else if(pref == LOG_LEVEL_DEBUG){
        level = DDLogLevelDebug;
    }
    return level;
}

+(void) addLoggerWithNewLevel:(int) level{
    if([[DDLog allLoggers] containsObject:fileLogger]){
        [DDLog removeLogger:fileLogger];
    }
    long ddLogLevel = logLevelForPref(level);
    [DDLog addLogger:fileLogger withLevel:ddLogLevel];
}

+(void) updateValuesFromPrefs
{
    BOOL logEnabled = [AppSettings isFileLoggingEnabled];
    
    int logMaxNumber = [AppSettings getMaxLogNumber];
    int logFileSize = [AppSettings getLogFileSize];
    
    int oldLogLevel = LOG_LEVEL;
    LOG_LEVEL = [AppSettings getLogLevel];
    if(logEnabled && ![[DDLog allLoggers] containsObject:fileLogger]){
        DDLogLevel fileLoggerLevel = DDLogLevelAll;//logLevelForPref(LOG_LEVEL);
        [DDLog addLogger:fileLogger withLevel:fileLoggerLevel];
        [Logging logInfo:@"Adding file logger"];
    } else if(!logEnabled && [[DDLog allLoggers] containsObject:fileLogger]){
        [DDLog removeLogger:fileLogger];
        [Logging logInfo:@"Removing file logger"];
    } else if(oldLogLevel != LOG_LEVEL){
        [Logging addLoggerWithNewLevel:LOG_LEVEL];
    }
    
    fileLogger.rollingFrequency = -1; //disable time based rolling
    fileLogger.logFileManager.maximumNumberOfLogFiles = logMaxNumber;
    fileLogger.maximumFileSize = 1024 * logFileSize;//Convert kB to B
    [Logging logInfo:@"Loaded Logger values, max number of logs is %ld, max file size is %ld kb",
     logMaxNumber, logFileSize ];
}

void append(NSString *msg)
{
    //Log out a redirected message if the sdk log level is enabled or the log level has not been set.
    //If the log level has not been set, this means the NSLog statement was made before the logger
    //was initialised.
    if(LOG_LEVEL >= SDK_LOG_LEVEL){
        //NOTE: Change below command if you modify SDK_LOG_LEVEL
        //Use a format string to avoid a compiler warning
        DDLogInfo(@"%@",msg);
    }
}

void _Log(NSString *prefix, const char *file, int lineNumber, const char *funcName, NSString *format,...)
{
    va_list ap;
    va_start (ap, format);
    NSString *msg = [[NSString alloc] initWithFormat:format arguments:ap];
    va_end (ap);
    append(msg);
}


+(void) initialize
{
    //Set up standard IDE loggers
    if(![[DDLog allLoggers] containsObject:[DDOSLogger sharedInstance]]){
        [DDLog addLogger:[DDOSLogger sharedInstance]];
    }
    if(![[DDLog allLoggers] containsObject:[DDTTYLogger sharedInstance]]){
        [DDLog addLogger:[DDTTYLogger sharedInstance]];
    }
    
    //Set up file Logger
    //Logger may not be nil after suspend/resume
    if(fileLogger == nil){
        fileLogger = [[DDFileLogger alloc] init];
        fileLogger.logFormatter = [[LogFormatter alloc] init];
    }
    
    //Read user preferences
    [Logging updateValuesFromPrefs];
    
    //print log header
    [Logging printLogHeader];
}

+(NSMutableArray *)getErrorLogs
{
    NSMutableArray* errorLogFiles = [[NSMutableArray alloc] init];
    NSArray *sortedLogFileInfos = [fileLogger.logFileManager sortedLogFileInfos];
    for (int i = 0; i < sortedLogFileInfos.count; i++) {
        DDLogFileInfo *logFileInfo = [sortedLogFileInfos objectAtIndex:i];
        NSData *fileData = [NSData dataWithContentsOfFile:logFileInfo.filePath];
        [errorLogFiles addObject:fileData];
    }
    
//    NSString *documentsDirectory = [NSSearchPathForDirectoriesInDomains (NSDocumentDirectory, NSUserDomainMask, YES) objectAtIndex:0];
//    NSString *fileName = [documentsDirectory stringByAppendingPathComponent:@"call_tarversal.log"];
//    //create file if it doesn't exist
//    if(![[NSFileManager defaultManager] fileExistsAtPath:fileName]) {
//        NSData *fileData2 = [NSData dataWithContentsOfFile:fileName];
//            [errorLogFiles addObject:fileData2];
//        }
    return errorLogFiles;
}

+(void) printLogHeader
{
    NSString* appVersion = [[NSBundle mainBundle] bundleIdentifier];
    
    NSString* model = [[UIDevice currentDevice] model];
    NSString* sysVer = [[UIDevice currentDevice] systemVersion];
    NSString* sysName = [[UIDevice currentDevice] systemName];
    
    float batteryLevel = [[UIDevice currentDevice] batteryLevel];
    UIDeviceBatteryState batteryState = [[UIDevice currentDevice] batteryState];
    NSString* batteryStateText = @"Unknown";
    if(batteryState == UIDeviceBatteryStateCharging){
        batteryStateText = @"Charging";
    } else if(batteryState == UIDeviceBatteryStateFull){
        batteryStateText = @"Full";
    } else if(batteryState == UIDeviceBatteryStateUnplugged){
        batteryStateText = @"Discharging";
    }
    
    [Logging logInfo:@"*********************** Avaya Oceana™ReferenceClient ***********************"];
    [Logging logInfo:@"Oceana™ReferenceClient %@", appVersion];
    [Logging logInfo:@"Device Details: %@ %@ %@", model, sysVer, sysName];
    [Logging logInfo:@"Battery details: %f %@", batteryLevel, batteryStateText];
}

+(NSData*) readASLLogs
{
    //Create the ASL log level lookup array
    NSArray* ASL_LEVELS = @[@"EMER", @"ALRT", @"CRIT", @"ERR ", @"WARN", @"NOTE", @"INFO", @"DBUG"];
    
    NSNumberFormatter *formatter = [[NSNumberFormatter alloc] init];
    formatter.numberStyle = NSNumberFormatterDecimalStyle;
    
    //Get ASL logs to capture the Reference Client SDK Logs
    /* Query the ASL database for all messages sent by "Oceana™ReferenceClient" */
    
    aslmsg query = asl_new (ASL_TYPE_QUERY);
    asl_set_query (query, ASL_KEY_SENDER, APP_NAME, ASL_QUERY_OP_EQUAL);
    
    aslresponse response = asl_search (NULL, query);
    
    /* Iterate all messages in found matching the query */
    //NOTE: This Data object is manually assembled from the log messages returned by ASL.
    //Theoretically over time this could cause attachment size issues with this data object
    //However, ASL queries on iOS are capped at 256 results so the size of this data object
    //should never grow too large.
    NSMutableData* data = [[NSMutableData alloc] init];
    aslmsg msg;
    msg = asl_next(response);
    while (msg)
    {
        
        /* Iterate message keys, extract value*/
        NSString* time = @"";
        NSString* level = @"";
        NSString* sender = @"";
        NSString* message = @"";
        NSString* messageId = @"";
        for (int keyIndex = 0; ; keyIndex++)
        {
            const char *key = asl_key (msg, keyIndex);
            if (key == NULL) break;
            
            const char *value = asl_get (msg, key);
            
            if(value != nil){
                if(strcmp(ASL_TIME_KEY,key) == 0){
                    time = [NSString stringWithUTF8String:value];
                } else if(strcmp(ASL_LEVEL_KEY,key) == 0){
                    //Convert char* to NSString
                    NSString* temp =  [NSString stringWithUTF8String:value];
                    //Convert NSString to NSNumber
                    NSNumber* levelNum = [formatter numberFromString:temp];
                    //Check for not null and that the numeric value does not exceed the ASL_LEVELS array capacity
                    if(levelNum == nil || [ASL_LEVELS count] <= [levelNum unsignedIntValue]){
                        level = @"NONE";
                    } else {
                        //Lookup pretty print version of Log Level from ASL_LEVELS
                        level = [ASL_LEVELS objectAtIndex:[levelNum unsignedIntValue]];
                    }
                } else if(strcmp(ASL_SENDER_KEY,key) == 0){
                    sender = [NSString stringWithUTF8String:value];
                } else if(strcmp(ASL_MESSAGE_KEY,key) == 0){
                    message = [NSString stringWithUTF8String:value];
                } else if(strcmp(ASL_MESSAGE_ID_KEY,key) == 0){
                    messageId = [NSString stringWithUTF8String:value];
                }
            }
        }
        
        if([time length] ==0){
            //Set a placeholder time value to make logs easier to read
            time = @"0000-00-00 00:00:00.000";
        }
        
        NSString* logMessage = [NSString stringWithFormat:@"%@ | [%@::%@] | %@\n", time,level,messageId,message];
        NSData* logMessageAsData = [logMessage dataUsingEncoding:NSUTF8StringEncoding];
        
        
        [data appendData:logMessageAsData];
        msg = asl_next (response);
    }
    asl_free (response);
    asl_free(query);
    
    return [NSData dataWithData:data];
}

+(void) logException: (NSString*) name withReason: (NSString*) reason andTrace: (NSArray*) trace
{
    [Logging logError:@"Got Exception of type %@", name];
    [Logging logError:@"Reason: %@", reason];
    for(int i = 0; i < [trace count]; i++){
        [Logging logError: [trace objectAtIndex:i]];
    }
}

+(void) logError: (NSString*) format, ...
{
    va_list ap;
    va_start (ap, format);
    NSString *msg = [[NSString alloc] initWithFormat:format arguments:ap];
    va_end (ap);
    //Use a format string to avoid a compiler warning
    DDLogError(@"%@",msg);
}

+(void) logWarn: (NSString *) format, ...
{
    va_list ap;
    va_start (ap, format);
    NSString *msg = [[NSString alloc] initWithFormat:format arguments:ap];
    va_end (ap);
    //Use a format string to avoid a compiler warning
    DDLogWarn(@"%@",msg);
}

+(void) logInfo: (NSString *) format, ...
{
    va_list ap;
    va_start (ap, format);
    NSString *msg = [[NSString alloc] initWithFormat:format arguments:ap];
    va_end (ap);
    //Use a format string to avoid a compiler warning
    DDLogInfo(@"%@",msg);
}

+(void) logDebug: (NSString *) format, ...
{
    va_list ap;
    va_start (ap, format);
    NSString *msg = [[NSString alloc] initWithFormat:format arguments:ap];
    va_end (ap);
    //Use a format string to avoid a compiler warning
    DDLogDebug(@"%@",msg);
}

@end
