/* Copyright Avaya Inc. */

#import <Foundation/Foundation.h>
#import "CSMessagingEnums.h"
#import "CSMessagingCompletionHandler.h"
#import "CSCapability.h"
#import "CSMessagingLimits.h"
#import "CSMessagingAsynchDataRetrieval.h"

@protocol CSMessagingAttachmentDelegate;

/**
 * Attachment class is representing a message optional parameter.
 * It can contain any file uploaded by sending participant.
 *
 * If user is sending a picture, the client application is responsible for creating a thumbnail picture
 * and setting proper object parameters. In the case of any other attachment type, a preview is not needed.
 * An attachment can be removed, if parent message hasn't been sent yet.
 */
@interface CSMessagingAttachment : NSObject

/**
 * The delegate responsible for handling messaging attachment events.
 */
@property (nonatomic, weak) id<CSMessagingAttachmentDelegate> delegate;

/**
 * The unique ID of the attachment.
 */
@property (nonatomic, readonly) NSString *attachmentId;

/**
 * The ID of the message that the attachment is associated with.
 *
 * @see ::CSMessage::messageId
 */
@property (nonatomic, readonly) NSString *messageId;

/**
 * The name of attachment.
 *
 * @see #setName:completionHandler:
 * @see #updateNameCapability
 */
@property (nonatomic, readonly) NSString *name;

/**
 * The MIME type of attachment.
 *
 * @see #setMimeType:completionHandler:
 * @see #updateMimeTypeCapability
 */
@property (nonatomic, readonly) NSString *mimeType;

/**
 * A Boolean value indicating whether the attachment is a thumbnail.
 *
 * @see #setIsThumbnail:completionHandler:
 * @see #updateIsThumbnailCapability
 */
@property (nonatomic, readonly, getter=isThumbnail) BOOL thumbnail;

/**
 * Duration of video or audio attachment represented in seconds.
 */
@property (nonatomic, readonly) NSUInteger duration;

/**
 * The size of the attachment in bytes.
 */
@property (nonatomic, readonly) CSLimitsSizeType size;

/**
 * The location of the attachment.
 *
 * @see #setLocation:completionHandler:
 * @see #updateLocationCapability
 */
@property (nonatomic, readonly) NSString *location;

/**
 * The status of the attachment.
 */
@property (nonatomic, readonly) CSMessagingAttachmentStatus status;

/**
 * Indicates whether the attachment was generated by client application or provided by user.
 *
 * @see #setIsGeneratedContent:completionHandler:
 * @see #updateIsGeneratedContentCapability
 */
@property (nonatomic, readonly, getter=isGeneratedContent) BOOL generatedContent;

/**
 * Returns the location of the attachment so that it can be opened by the application.
 * This property has the additional effect of changing the status of the attachment to Opened.
 *
 * @return Location of attachment or nil if location is not avaliable.
 * @deprecated THIS SELECTOR WILL BE REMOVED IN RELEASE VERSION 3.3.
 * Use the {@link consume} method instead.
 */
@property (nonatomic, readonly) NSString *locationForConsumption DEPRECATED_ATTRIBUTE;

/**
 * Returns the location of the attachment so that it can be opened by the application.
 * This method has the additional effect of changing the status of the attachment to Opened.
 *
 * @return Location of attachment or nil if location is not avaliable.
 */
- (NSString *)consume;

/**
 * Sets the attachment name.
 *
 * @param name attachment name.
 * @param handler The completion handler for this operation.
 */
- (void)setName:(NSString *)name completionHandler:(CSMessagingCompletionHandler)handler;

/**
 * Sets the thumbnail property.
 *
 * @param isThumbnail new value of isThumbnail property.
 * @param handler The completion handler for this operation.
 */
- (void)setIsThumbnail:(BOOL)isThumbnail completionHandler:(CSMessagingCompletionHandler)handler;

/**
 * Sets the generatedContent property.
 *
 * @param isGeneratedContent new value of isGeneratedContent property.
 * @param handler The completion handler for this operation.
 */
- (void)setIsGeneratedContent:(BOOL)isGeneratedContent completionHandler:(CSMessagingCompletionHandler)handler;

/**
 * Sets the attachment location.
 *
 * @param location attachment location.
 * @param handler The completion handler for this operation.
 */
- (void)setLocation:(NSString *)location completionHandler:(CSMessagingCompletionHandler)handler;

/**
 * Sets the attachment MIME type.
 *
 * @param mimeType attachment MIME type.
 * @param handler The completion handler for this operation.
 */
- (void)setMimeType:(NSString *)mimeType completionHandler:(CSMessagingCompletionHandler)handler;

/**
 * Starts the download of the attachment asynchronously.
 *
 * @param path path to download attachment.
 * @param handler The completion handler for this operation.
 *
 * @return A ::CSMessagingAsynchDataRetrieval object.
 *
 * @see #downloadCapability
 */
- (CSMessagingAsynchDataRetrieval *)download:(NSString *)path completionHandler:(CSMessagingCompletionHandler)handler;

/**
 * A ::CSCapability object that indicates if the attachment's name property can be changed.
 *
 * This capability is @b Allowed when the Attachment is in a draft or error state.
 *
 * @see #setName:completionHandler:
 */
@property (nonatomic, readonly) CSCapability *updateNameCapability;

/**
 * A ::CSCapability object that indicates if the attachment's thumbnail property can be changed.
 *
 * This capability is @b Allowed when the Attachment is in a draft or error state.
 *
 * @see #setIsThumbnail:completionHandler:
 */
@property (nonatomic, readonly) CSCapability *updateIsThumbnailCapability;

/**
 * A ::CSCapability object that indicates if the attachment's "generated content" property can be changed.
 *
 * This capability is @b Allowed when the attachment is in a draft or error state.
 *
 * @see #setIsGeneratedContent:completionHandler:
 */
@property (nonatomic, readonly) CSCapability *updateIsGeneratedContentCapability;

/**
 * A ::CSCapability object that indicates if the attachment's location property can be changed.
 *
 * This capability is @b Allowed when the attachment is in a draft or error state.
 *
 * @see #setLocation:completionHandler:
 */
@property (nonatomic, readonly) CSCapability *updateLocationCapability;

/**
 * A ::CSCapability object that indicates if the attachment's MIME type property can be changed.
 *
 * This capability is @b Allowed when the attachment is in a draft or error state.
 *
 * @see #setMimeType:completionHandler:
 */
@property (nonatomic, readonly) CSCapability *updateMimeTypeCapability;

/**
 * A ::CSCapability object that indicates if the attachment can be downloaded.
 *
 * This capability is @b Allowed when all of the following conditions are true:
 *
 * @li The messaging service is online (has an active connection to the server).
 * @li The attachment is in the "ready to download" state.
 *
 * @see #download:completionHandler:
 */
@property (nonatomic, readonly) CSCapability *downloadCapability;

/**
 * A ::CSCapability object that indicates if the attachment can be consumed.
 *
 * This capability is @b Allowed when the attachment is in either the downloaded or consumed state.
 *
 * @see #download:completionHandler:
 */
@property (nonatomic, readonly) CSCapability *consumeCapability;

/**
 * Returns a CSCapability object that indicates if the attachment can be removed.
 *
 * This capability is @b Allowed when the attachment is in a draft or error state.
 *
 * @see CSMessage#removeAttachment:completionHandler:
 */
@property (nonatomic, readonly) CSCapability *removeCapability;

@end

/*
 * Protocol of delegate that can be used to retrieve status updates for a messaging attachment.
 */
@protocol CSMessagingAttachmentDelegate <NSObject>

/**
 * Called to report that the name property has been changed.
 *
 * @param messagingAttachment the {@link CSMessagingAttachment} for which the attribute has been changed.
 * @param name the new value for the name attribute.
 * @see -[CSMessagingAttachment name]
 */
- (void)messagingAttachment:(CSMessagingAttachment *)messagingAttachment didChangeName:(NSString *)name;

/**
 * Called to report that the isThumbnail property has been changed.
 *
 * @param messagingAttachment the {@link CSMessagingAttachment} for which the attribute has been changed.
 * @param IsThumbnail The new value for the isThumbnail property
 * @see -[CSMessagingAttachment thumbnail]
 */
- (void)messagingAttachment:(CSMessagingAttachment *)messagingAttachment didChangeIsThumbnail:(BOOL)IsThumbnail;

/**
 * Called to report that the isGeneratedContent property of the attachment has changed.
 *
 * @param messagingAttachment the {@link CSMessagingAttachment} for which the attribute has been changed.
 * @param isGeneratedContent The new value for the isGeneratedContent property
 * @see -[CSMessagingAttachment generatedContent]
 */
- (void)messagingAttachment:(CSMessagingAttachment *)messagingAttachment didChangeIsGeneratedContent:(BOOL)isGeneratedContent;

/**
 * Called to report that the location property of the attachment has changed.
 *
 * @param messagingAttachment the {@link CSMessagingAttachment} for which the attribute has been changed.
 * @param location The new value for the location property
 * @see -[CSMessagingAttachment location]
 */
- (void)messagingAttachment:(CSMessagingAttachment *)messagingAttachment didChangeLocation:(NSString *)location;

/**
 * Called to report that the mimeType property of the attachment has changed.
 *
 * @param messagingAttachment the {@link CSMessagingAttachment} for which the attribute has been changed.
 * @param mimeType The new value for the mimeType property
 * @see -[CSMessagingAttachment mimeType]
 */
- (void)messagingAttachment:(CSMessagingAttachment *)messagingAttachment didChangeMimeType:(NSString *)mimeType;

/**
 * Called to report that the attachment status changed.
 *
 * @param messagingAttachment the {@link CSMessagingAttachment} for which the attribute has been changed.
 * @param status The new attachment status
 * @see -[CSMessagingAttachment status]
 */
- (void)messagingAttachment:(CSMessagingAttachment *)messagingAttachment didChangeStatus:(CSMessagingAttachmentStatus)status;

/**
 * Called to report that messaging attachment capabilities have changed.
 *
 * @param messagingAttachment the {@link CSMessagingAttachment} for which the attribute has been changed.
 * @see ::CSMessagingAttachment::consumeCapability
 * @see ::CSMessagingAttachment::downloadCapability
 * @see ::CSMessagingAttachment::removeCapability
 * @see ::CSMessagingAttachment::updateIsGeneratedContentCapability
 * @see ::CSMessagingAttachment::updateLocationCapability
 * @see ::CSMessagingAttachment::updateMimeTypeCapability
 * @see ::CSMessagingAttachment::updateNameCapability
 * @see ::CSMessagingAttachment::updateIsThumbnailCapability
 */
- (void)messagingAttachmentDidChangeCapabilities:(CSMessagingAttachment *)messagingAttachment;

@end
