//
//  OceanaVideoClient.h
//  Oceana™ReferenceClient
//
//  Copyright 2016 Avaya Inc
//  All rights reserved. Usage of this source is bound to the terms described the file
//  Avaya SDK EULA.txt, included in this SDK.
//  Avaya – Confidential & Proprietary. Use pursuant to your signed agreement or Avaya Policy.
//

#import <Foundation/Foundation.h>
#import <AVFoundation/AVFoundation.h>
#import "OceanaVideoClient.h"
#import "AppSettings.h"
#import "Logging.h"
#import "WorkConfig.h"
#import "AppDelegate.h"

@implementation OceanaVideoClient

AOOceanaCustomerWebVoiceVideo *videoCustomer;
AOWork *videoWork;
AOAttributeList *attributes;
AOVideoInteraction *video;
WorkConfig *workConfiguration;
AOClientConfiguration* clientConfiguration;
AOVideoDevice *device;
NSObject<AOVideoInteractionDelegate>* videoDelegate;
NSObject<AOConnectionListenerDelegate>* videoConnectionDelegate;

-(id)init
{
    self = [super init];
    if (self) {
        workConfiguration = [[WorkConfig alloc]init];
        
        [self configureCPProvider];
        videoCustomer = [[AOOceanaCustomerWebVoiceVideo alloc]initWithClientConfiguration:clientConfiguration];
        //    videoCustomer = [[AOOceanaCustomerWebVoiceVideo alloc] initWithIsSecure:YES andPort:@"443" andRestFQDN:@"linpubp018.gl.avaya.com"];
        videoWork = videoCustomer.createWork;
        
        videoWork.context = @""; //Here will be context
        
        //     OLD WORK API
        //     attributes = videoWork.createAttributeList;
        //     attributes.attributes = [workConfiguration attributeBuilder];
        
        video = videoWork.createVideoInteraction;
#ifdef DEBUG_1
        dispatch_async(dispatch_get_main_queue(), ^(void){
            AppDelegate *appDelegate = (AppDelegate *)[[UIApplication sharedApplication] delegate];
            video.useThisCodec = appDelegate.selectedCodec;
            
        });
#endif
        device = video.videoDevice;
    }
    return self;
}

- (void)configureCPProvider
{
    AOConfiguration* config = [[AOConfiguration alloc]init];
    config.restUrl = [AppSettings getServer];
    config.port = [AppSettings getPort];
    config.isSecure = [AppSettings useSecureLogin];
    config.urlPath = [AppSettings getAMCRestUrl];
    
    AOWebGatewayConfiguration* webGatewayConfig = [[AOWebGatewayConfiguration alloc]init];
    webGatewayConfig.webGatewayAddress = [AppSettings getAAWGServerAddress];
    webGatewayConfig.port = [AppSettings getAAWGServerPort];
    webGatewayConfig.isSecure = [AppSettings isAawgSecure];
    webGatewayConfig.webGatewayUrlPath = [AppSettings getAAWGRestUrl];
    
    
    clientConfiguration = [[AOClientConfiguration alloc]init];
    clientConfiguration.configuration = config;
    clientConfiguration.webGatewayConfiguration = webGatewayConfig;
}

- (void)setAuthorizationToken:(NSString*)token;
{
    video.authorizationToken = token;
}

- (void)configureInteraction
{
    [video muteAudio:FALSE];
    [video muteVideo:FALSE];
    [video enableVideo:TRUE];
}

- (void)startInteraction
{
    dispatch_async(dispatch_get_main_queue(), ^{
      AppDelegate *appdelegate = (AppDelegate*) [[UIApplication sharedApplication] delegate];
      [video setPlatformType:appdelegate.selectedPlatformType];
    });
    
    [self configureAudioSessionWithVideo: YES];
    [self buildAMCRequestPayload];
    video.delegate = videoDelegate;
    video.connectionListenerDelegate = videoConnectionDelegate;
    video.destinationAddress = [AppSettings getDestinationAddress];
    [video start];
}

- (void)endInteraction
{
    [video end];
}

- (void)discard
{
    [video discard];
}

- (BOOL) isHeld {
    return video.isHeld;
}

- (void)sendDTMF:(AODTMFTone)tone
{
    [video sendDTMF:tone];
}

- (void)toggleMuteAudio
{
    [video muteAudio: ![video isAudioMuted]];
    [Logging logDebug:@"%s Setting Mute audio state to %hhd", __PRETTY_FUNCTION__, [video isAudioMuted]];
}

- (void)toggleMuteVideo
{
    [video muteVideo:![video isVideoMuted]];
    [Logging logDebug:@"%s Setting Mute video state to %hhd", __PRETTY_FUNCTION__, [video isVideoMuted]];
}

-(void) toggleVideoEnabled
{
    [video enableVideo: ![video isVideoEnabled]];
    [Logging logInfo:@"%s Setting video enabled state to %hhd",__PRETTY_FUNCTION__, [video isVideoEnabled]];
}

-(BOOL) isVideoEnabled {
    return [video isVideoEnabled];
}

- (void) toggleHold
{
    if (video.isHeld)
    {
        [video unholdWithCompletionHandler:^(NSError *error){}];
    } else {
        [video holdWithCompletionHandler:^(NSError *error){}];
    }
    [Logging logDebug:@"%s Setting Hold video state to %hhd",__PRETTY_FUNCTION__, video.isHeld];
}

-(long) getCallTimeElapsed
{
    return [video getInteractionTimeElapsed];
}

-(void) setVideoDelegate:(NSObject<AOVideoInteractionDelegate>*)delegate
{
    videoDelegate = delegate;
}

- (void) setConnectionDelegate:(NSObject<AOConnectionListenerDelegate> *)delegate
{
    videoConnectionDelegate = delegate;
}

- (AOInteractionState)getInteractionState
{
    return [video getInteractionState];
}

- (void)buildAMCRequestPayload
{
    videoWork.context = [AppSettings getCustID];
    videoWork.topic = [workConfiguration topicBuilder];
    videoWork.locale = [AppSettings getLocale];
    videoWork.routingStrategy = [AppSettings getStrategy];
    videoWork.services = [workConfiguration serviceMapBuilder];
    
    if ([[AppSettings getSourceName] isEqual: @"" ] || [[AppSettings getResourceId]isEqualToString:@""]) {
        [Logging logInfo:@"%s ResourceMap is empty", __PRETTY_FUNCTION__];
    } else {
        videoWork.resources = [workConfiguration resourceMapBuilder];
    }
}

- (void)readVideoDetailsWithCompletionHandler:(void (^)(AOVideoDetails *videoDetails))handler
{
    [video readVideoDetailsWithCompletionHandler:^(AOVideoDetails *videoDetails) {
        [Logging logInfo:@"%s %@", __PRETTY_FUNCTION__, videoDetails];
        handler(videoDetails);
    }];
}

- (void)readAudioDetailsWithCompletionHandler:(void (^)(AOAudioDetails *audioDetails))handler {
    [video readAudioDetailsWithCompletionHandler:^(AOAudioDetails *audioDetails) {
        [Logging logInfo:@"%s %@", __PRETTY_FUNCTION__, audioDetails];
        handler(audioDetails);
    }];
}

- (void)readCallTypeWithCompletionHandler:(void (^)(NSString *csCallTypeString))handler {
    [video readCallTypeWithCompletionHandler:^(NSString *csCallTypeString) {
        handler(csCallTypeString);
    }];
}


- (void)setRemoteView:(UIView*) remote
{
    [device setRemoteView:remote];
}

- (void)setLocalView:(UIView*) local
{
    dispatch_async(dispatch_get_main_queue(), ^(void){
        AppDelegate *appDelegate = (AppDelegate *)[[UIApplication sharedApplication] delegate];
        [device setVideoCaptureResolutionWithCaptureOrientation:appDelegate.currentSelectedVideoRecordPref orientationPreference:appDelegate.currentSelectedVideoRecordOrientation];
    });
//    [device setCameraCaptureResolution:AOVideoResolution_480x272];    
    [device setLocalView:local];
}

- (void)selectCamera:(AOCameraType)cameraType
{
    [device selectCamera:cameraType];
}

- (void)switchCamera
{
    [device switchCamera];
}

- (void)configureAudioSessionWithVideo:(BOOL)withVideo {
    [Logging logInfo:@"%s", __PRETTY_FUNCTION__];
    AVAudioSession *audioSession = [AVAudioSession sharedInstance];
    NSError* err;
    
    [audioSession setCategory:AVAudioSessionCategoryPlayAndRecord
                         mode:(withVideo ? AVAudioSessionModeVideoChat : AVAudioSessionModeVoiceChat)
                      options:(AVAudioSessionCategoryOptionAllowBluetooth | AVAudioSessionCategoryOptionMixWithOthers)
                        error:&err];
    if (err) {
        [Logging logError:@"%s Error setting audio category %@", __PRETTY_FUNCTION__, err];
    }
    
}
@end
