/* Copyright Avaya Inc. */

#import <Foundation/Foundation.h>

#import "CSCredentialProvider.h"

/**
 * Enumeration representing mode for push notification activation.
 */
typedef NS_ENUM(NSUInteger, CSPushNotificationActivationMode) {
	/**
	* Push Notifications are disabled
	*/
	CSPushNotificationActivationModeDisabled,
	/**
	* Push Notification activation will be done using HTTP
	*/
	CSPushNotificationActivationModeHTTP,
	/**
	* Push Notification activation will be done using SIP
	*/
	CSPushNotificationActivationModeSIP,
};

/**
 * Configuration information for Push Notifications.
 */
@interface CSPushNotificationConfiguration : NSObject

/**
 * A Boolean value indicating whether Push Notifications are enabled.
 * @deprecated Use the {@link #activationMode} property instead.
 */
@property (nonatomic, getter=isEnabled, setter=setEnabled:) BOOL enabled DEPRECATED_ATTRIBUTE;

/**
 * Push Notification mode to be used for activation.
 */
@property (nonatomic) CSPushNotificationActivationMode activationMode;

/**
 * The address information for the telephony event push notification service
 *
 * Example: https://server_address:server_port/
 */
@property (nonatomic, copy) NSURL *telephonyPushNotificationServiceURL;

/**
 * A globally unique app-specific device token generated by the 
 * Apple Push Notification Service and assigned to this application
 * instance when the iOS application registers with the service to receive
 * incoming call push notifications. This token is required for  receiving
 * VoIP priority push notification messages through  iOS PushKit
 * APIs. Avaya telephony servers (e.g., Avaya Aura or Avaya IP Office
 * Server) use Apple's VoIP priority push notification mechanism
 * to deliver incoming call notifications to iOS clients.
 * Valid voipPushToken and pushToken fields must be
 * provided in order to get push notification service from Avaya telephony
 * servers.
 */
@property (nonatomic, strong) NSString *voipPushToken;

/**
 * A globally unique app-specific device token generated by the
 * Apple Push Notification Service and assigned to this application
 * instance when the iOS application registers with the service to receive
 * telephony push notifications other than incoming calls (e.g., voice mail
 * status updates). See Apple's "Local and Remote Notification
 * Programming Guide" for more information on how this token
 * is generated and used.
 * Valid voipPushToken and pushToken fields must be
 * provided in order to get push notification service from Avaya telephony
 * servers.
 */
@property (nonatomic, strong) NSString *pushToken;

/**
 * Application identifier in the form com.domain.application
 */
@property (nonatomic, strong) NSString *applicationBundleId;

/**
 * The credential provider
 */
@property (nonatomic, weak) id <CSCredentialProvider> credentialProvider;

/**
 * A Boolean flag indicating whether the client expects the telephony server to send
 * incoming call push notification events over Apple's VoIP push notification network or
 * remote (non-VoIP) push notification network. By default, the flag is set to YES, indicating that
 * the VoIP push notification channel will be used for incoming calls. However, in market
 * environments where use of CallKit APIs is not allowed (e.g., due to governmental regulations),
 * the client application should set this parameter to NO, thus controlling the delivery of incoming
 * call push notification messages over Apple's  remote (non-VoIP) push notification network.
 */
@property (nonatomic) BOOL useVoIPPushNotificationChannelForIncomingCalls;

/**
 * A Boolean flag indicating whether incoming push notification events should
 * be sent over Apple's development push notification network
 * (api.development.push.apple.com) or the production push notification network
 * (api.push.apple.com). The Boolean flag is set to YES  if the application is built
 * using a development provisioning profile. By default, the property is
 * set to NO, and Apple's production network is used to route push notification
 * messages.
 */
@property (nonatomic) BOOL useDevelopmentPushNotificationNetwork;

@end
