//
//  CrashHandler.m
//  Oceana™ReferenceClient
//
//  Copyright 2016 Avaya Inc
//  All rights reserved. Usage of this source is bound to the terms described the file
//  Avaya SDK EULA.txt, included in this SDK.
//  Avaya – Confidential & Proprietary. Use pursuant to your signed agreement or Avaya Policy.
//
//

#import <Foundation/Foundation.h>
#import "CrashHandler.h"
#import "Logging.h"
#include <execinfo.h>

@implementation CrashHandler

const NSInteger UncaughtExceptionHandlerSkipAddressCount = 4;
const NSInteger UncaughtExceptionHandlerReportAddressCount = 5;

+ (id) instance
{
    static CrashHandler *sharedCrashHandler = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        sharedCrashHandler = [[self alloc] init];
    });
    return sharedCrashHandler;
}

+ (NSArray *)backtrace
{
    void *backtraceFrames[128];
    int i, frameCount = backtrace(backtraceFrames, 128);
    
    char** strs = backtrace_symbols(backtraceFrames, frameCount);
    NSMutableArray *backtrace = [NSMutableArray arrayWithCapacity:frameCount];
    for (i = 0; i < frameCount; ++i) {
        NSString* frame = [NSString stringWithUTF8String:strs[i]];
        [backtrace addObject:frame];
    }
    free(strs);
    
    return backtrace;
}

void UncaughtExceptionHandler(NSException *exception) {
    NSArray* backtrace = [CrashHandler backtrace];
    [Logging logError:@"Crash Occurred"];
    NSString* name = [exception name];
    NSString* reason = [exception reason];
    
    // report the exception
    [Logging logException:name withReason: reason andTrace: backtrace];
}

-(void) initialise{
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        NSSetUncaughtExceptionHandler(&UncaughtExceptionHandler);
    });
}
@end