//
//  OceanaClient.h
//  Oceana™ReferenceClient
//
//  Copyright 2016 Avaya Inc
//  All rights reserved. Usage of this source is bound to the terms described the file
//  Avaya SDK EULA.txt, included in this SDK.
//  Avaya – Confidential & Proprietary. Use pursuant to your signed agreement or Avaya Policy.
//

#import <Foundation/Foundation.h>
#import <AVFoundation/AVFoundation.h>
#import "OceanaClient.h"
#import "AppSettings.h"
#import "Logging.h"
#import "WorkConfig.h"
#import "AppDelegate.h"


@implementation OceanaClient

AOOceanaCustomerWebVoiceVideo *customerWebVoice;
AOWork *work;
AOAttributeList *listOfAttributes;
AOAudioInteraction *audio;
WorkConfig *workConfig;
AOClientConfiguration* clientConfig;
NSObject<AOAudioInteractionDelegate>* audioDelegate;
NSObject<AOConnectionListenerDelegate>* connectionDelegate;

-(id)init
{
    self = [super init];
    if(self){
        workConfig = [[WorkConfig alloc]init];
        
        [self configureCPProvider];
        customerWebVoice = [[AOOceanaCustomerWebVoiceVideo alloc]initWithClientConfiguration:clientConfig];
        work = customerWebVoice.createWork;
        /*
         * Alternate method of creating routing attributes.
         * The below mechanism creates a simple work request with the attributes provided
         
         listOfAttributes = work.createAttributeList;
         listOfAttributes.attributes = [workConfig attributeBuilder];
         */
        audio = work.createAudioInteraction;
#ifdef DEBUG_1
        dispatch_async(dispatch_get_main_queue(), ^(void){
            AppDelegate *appDelegate = (AppDelegate *)[[UIApplication sharedApplication] delegate];
            audio.useThisCodec = appDelegate.selectedCodec;
        });
#endif
    }
    return self;
}

- (void)configureCPProvider
{
    //    dispatch_async(dispatch_get_main_queue(), ^{
    clientConfig = [[AOClientConfiguration alloc]init];
    
    AOWebGatewayConfiguration* webGatewayConfig = [[AOWebGatewayConfiguration alloc]init];
    webGatewayConfig.webGatewayAddress = [AppSettings getAAWGServerAddress];
    webGatewayConfig.port = [AppSettings getAAWGServerPort];
    webGatewayConfig.isSecure = [AppSettings isAawgSecure];
    webGatewayConfig.webGatewayUrlPath = [AppSettings getAAWGRestUrl];
    
    clientConfig.webGatewayConfiguration = webGatewayConfig;
    NSInteger selectedPlatform = [AppSettings selectedPlatform];
    if (selectedPlatform == AOPlatFormType_OCEANA) {
        AOConfiguration* config = [[AOConfiguration alloc]init];
        config.restUrl = [AppSettings getServer];
        config.port = [AppSettings getPort];
        config.isSecure = [AppSettings useSecureLogin];
        config.urlPath = [AppSettings getAMCRestUrl];
        clientConfig.configuration = config;
    }
    //    });
}

- (void)setAuthorizationToken:(NSString*)token;
{
    audio.authorizationToken = token;
}

- (void)configureAudio
{
    [audio muteAudio:FALSE];
}

- (void)startInteraction
{
    dispatch_async(dispatch_get_main_queue(), ^{
      AppDelegate *appdelegate = (AppDelegate*) [[UIApplication sharedApplication] delegate];
      [audio setPlatformType:appdelegate.selectedPlatformType];
    });
    
    [self configureAudioSessionWithVideo: NO];
    [self buildAMCRequestPayload];
    audio.delegate = audioDelegate;
    audio.connectionListenerDelegate = connectionDelegate;
    audio.destinationAddress = [AppSettings getDestinationAddress];
    
    [audio start];
}

- (void)endInteraction
{
    [audio end];
}

- (void)discard
{
    [audio discard];
}

-(BOOL) isHeld {
    return audio.isHeld;
}

- (void)sendDTMF:(AODTMFTone)tone
{
    [audio sendDTMF:tone];
}

- (void)toggleMuteAudio
{
    [audio muteAudio: ![audio isAudioMuted]];
    [Logging logDebug:@"%s Setting Mute audio state to %hhd",__PRETTY_FUNCTION__, [audio isAudioMuted]];
}

- (void) toggleHold
{
    if (audio.isHeld)
    {
        [audio unholdWithCompletionHandler:^(NSError *error){}];
    } else {
        [audio holdWithCompletionHandler:^(NSError *error){}];
    }
    [Logging logDebug:@"%s Setting Hold audio state to %hhd",__PRETTY_FUNCTION__, audio.isHeld];
}

-(long) getCallTimeElapsed
{
    return [audio getInteractionTimeElapsed];
}

-(void) setAudioDelegate:(NSObject<AOAudioInteractionDelegate>*)delegate
{
    audioDelegate = delegate;
}

-(void) setConnectionlistenerDelegate:(NSObject<AOConnectionListenerDelegate> *)delegate
{
    connectionDelegate = delegate;
}

- (AOInteractionState)getInteractionState
{
    return [audio getInteractionState];
}

- (NSString*)versionNumber
{
    return customerWebVoice.versionNumber;
}

- (void)buildAMCRequestPayload
{
    work.context = [AppSettings getCustID];
    work.topic = [workConfig topicBuilder];
    work.locale = [AppSettings getLocale];
    work.routingStrategy = [AppSettings getStrategy];
    work.services = [workConfig serviceMapBuilder];
    
    if ([[AppSettings getSourceName] isEqual: @"" ] || [[AppSettings getResourceId]isEqualToString:@""]) {
        [Logging logInfo:@"%s ResourceMap is empty", __PRETTY_FUNCTION__];
    } else {
        work.resources = [workConfig resourceMapBuilder];
    }
}

- (void)readAudioDetailsWithCompletionHandler:(void (^)(AOAudioDetails *audioDetails))handler
{
    [audio readAudioDetailsWithCompletionHandler:^(AOAudioDetails *audioDetails) {
        [Logging logInfo:@"%s %@", __PRETTY_FUNCTION__, audioDetails];
        handler(audioDetails);
    }];
}
- (void)readCallTypeWithCompletionHandler:(void (^)(NSString *csCallTypeString))handler {
    [audio readCallTypeWithCompletionHandler:^(NSString *csCallTypeString) {
        handler(csCallTypeString);
    }];
}

- (void)configureAudioSessionWithVideo:(BOOL)withVideo {
    [Logging logInfo:@"%s", __PRETTY_FUNCTION__];
    AVAudioSession *audioSession = [AVAudioSession sharedInstance];
    NSError* err;
    
    [audioSession setCategory:AVAudioSessionCategoryPlayAndRecord
                         mode:(withVideo ? AVAudioSessionModeVideoChat : AVAudioSessionModeVoiceChat)
                      options:(AVAudioSessionCategoryOptionAllowBluetooth | AVAudioSessionCategoryOptionMixWithOthers)
                        error:&err];
    if (err) {
        [Logging logError:@"%s Error setting audio category %@", __PRETTY_FUNCTION__, err];
    }
}

-(BOOL) isBlootuthIsConnected {
    NSArray *arrayInputs = [[AVAudioSession sharedInstance] availableInputs];
    for (AVAudioSessionPortDescription *port in arrayInputs)
    {
        if ([port.portType isEqualToString:AVAudioSessionPortBluetoothHFP] || [port.portType isEqualToString:AVAudioSessionPortBluetoothA2DP]|| [port.portType isEqualToString:AVAudioSessionPortBluetoothLE])
        {
            return YES;
        }
    }
    return NO;
}

@end
