/* Copyright Avaya LLC. */

#import <Foundation/Foundation.h>
#import "CSContact.h"
#import "CSContactService.h"
/**
 * Describes the source type of the user's address.
 * e.g. the remote address was provided by the client or
 * by the network or if the remote address represents a
 * conference participant.
 */
typedef NS_ENUM(NSInteger, CSCallLogAddressSourceType) {
	/** The address source is not defined. */
	CSCallLogAddressSourceTypeUndefined = 0,
	/** The call log entry or event is generated after user actions. */
	CSCallLogAddressSourceTypeUser,
	/** The call log entry or event is generated by messages from the network (from a server). */
	CSCallLogAddressSourceTypeNetwork,
	/** The call log entry or event is generated by messages from a conference. */
	CSCallLogAddressSourceTypeConference
};

/** 
 * Interface object representing information about remote party for a call log entry.
 * A CSCallLogParticipant object contains remote address, remote number, display name and other information about the called or calling user.
 */
@interface CSCallLogParticipant : NSObject

/** Constructor with parameters. */
- (instancetype)initWithDisplayName:(NSString*)displayName remoteAddress:(NSString*)remoteAddress remoteNumber:(NSString*)remoteNumber enterTime:(NSDate *)enterTime leaveTime:(NSDate *)leaveTime isModerator:(bool)isModerator sourceType:(CSCallLogAddressSourceType)sourceType isRestrictedParticipant:(bool)isRestrictedParticipant;

/** The full remote address (i.e., full uri) to the remote endpoint. */
@property (nonatomic, readonly) NSString *remoteAddress;

/** The remote number associated with this participant on this call. */
@property (nonatomic, readonly) NSString *remoteNumber;

/** The remote user's display name as reported by the network. */
@property (nonatomic, readonly) NSString *displayName;

/**
 * Determines the source of the remote endpoint address. The address
 * associated with this remote user may be provided by the local client,
 * the network, or may have been obtained from a conference roster.
 */
@property (nonatomic, readonly) CSCallLogAddressSourceType sourceType;

/**
 * For an outgoing point to point call, this time represents the time (UTC) the call
 * was dialed out. For an incoming call, this time represents the time
 * when local phone device received incoming call. For a conference call where
 * the participant list is available, this time stamp reflects when the remote
 * participant joined the conference.
 */
@property (nonatomic, readonly) NSDate *enterDate;

/**
 * Represents the time (UTC) of when this participant has left the session.
 * For p2p session, this represents when the session ended.
 * The default value is 0, and in the case of conference supporting participant list, if the
 * participant did not leave the conference prior to the conference ending, the
 * LeaveTime will continue to report nil to indicate that the participant did not leave the
 * call when the conference was ended by local user.
 */
@property (nonatomic, readonly) NSDate *leaveDate;

/** 
 * Indicates if this remote participant represents a moderator
 * on this conference call. If the remote participant is
 * not a moderator for the conference call or if the call
 * is a point to point call, this property is set to NO.
 */
@property (nonatomic, readonly) BOOL isModerator;

/**
 * A Boolean value indicating whether the participant has limited capabilities on the call.
 * Restricted participant cannot be lecturer or presenter and cannot become a normal participant.
 * Restricted participant has no moderator controls and can only have a single receive-only video stream.
 * Restricted participant cannot unblock camera and send video, or cannot perform remote camera control operations (e.g., control camera of a remote room system).
 * Restricted participant is muted by the conference server by default, and can only be unmuted by the moderator, e.g., by getting moderator's attention to be unmuted through raise hand mechanism.
 */
@property (nonatomic, readonly, getter=isRestrictedParticipant) BOOL restrictedParticipant;

/** Matched contact for this participant. If there was no matched contact, this property will remain nil.
 * @deprecated It is recommended to use the ::CSContactService::resolveContacts: method instead.
 */
@property (nonatomic, readonly) CSContact *matchingContact DEPRECATED_ATTRIBUTE;
@end
