/* Copyright Avaya LLC. */

#import <Foundation/Foundation.h>

/**
 * Enumeration representing messaging polling refresh modes.
 */
typedef NS_ENUM(NSUInteger, CSMessagingRefreshMode) {
	CSMessagingRefreshModePush		= 0, /**< Push mode means continuous notifications in real time. */
	CSMessagingRefreshMode1Minute	= 1, /**< Polling for changes will be done every one minute. */
	CSMessagingRefreshMode2Minutes	= 2, /**< Polling for changes will be done every two minutes. */
	CSMessagingRefreshMode5Minutes	= 5,  /**< Polling for changes will be done every five minutes. */
	CSMessagingRefreshMode10Minutes = 10, /**< Polling for changes will be done every ten minutes. */
	CSMessagingRefreshMode15Minutes	= 15, /**< Polling for changes will be done every fifteen minutes. */
	CSMessagingRefreshMode30Minutes = 30, /**< Polling for changes will be done every thirty minutes. */
	CSMessagingRefreshMode60Minutes	= 60,  /**< Polling for changes will be done every one hour. */
	CSMessagingRefreshModeManual	= 1000 /**< Polling for updates is disabled. */
};

/**
 * Enumeration representing messaging sources.
 */
typedef NS_ENUM(NSUInteger, CSMessagingProviderType) {
	CSMessagingProviderTypeAvayaMultimediaMessaging,	/**< Avaya Multimedia Messaging provider (AMM) */
	CSMessagingProviderTypeAvayaEquinoxCloud,	/**< Avaya Equinox Cloud provider (Zang) */
	CSMessagingProviderTypeAvayaOneXPortal      /**< Avaya OneX Portal provider */
};

/**
 * Describes the possible types of messages that can be included in a conversation.
 */
typedef NS_ENUM(NSInteger, CSMessagingType) {
	CSMessagingTypeMessage, /**< Messages of this type contain either body text, attachments, or both. Most messages in a conversation are of this type. */
	CSMessagingTypeJoined, /**< Joined type is used to notify that new participant has joined the conversation. */
	CSMessagingTypeLeft, /**< Left type is used to notify that existing participant has left the conversation. */
	CSMessagingTypeSession, /**< Session type is used to notify that conversation was escalated to audio/video. */
	CSMessagingTypeSubjectChanged, /**< SubjectChanged type is used to notify that Subject of conversation has changed. */
	CSMessagingTypeClosed, /**< Closed type is used to notify that the conversation has been closed. */
	CSMessagingTypeStart /**< A message of this type is the first message in every conversation. There is no content associated with this message. */
};

/**
 * Provides additional reason information for the Left and Closed message types.
 */
typedef NS_ENUM(NSInteger, CSMessagingReasonType) {
	CSMessagingReasonTypeNone = 0, /** The message reason is not specified. */
	CSMessagingReasonTypeManual,   /** The message is a result of a manual intervention. */
	CSMessagingReasonTypeAuto,     /** The message is generated by the system. */
	CSMessagingReasonTypeForced    /** The message is a result of power user's enforcement action.
	                                *  E.g. participant left the conversation due to he was removed by other participant who has appropriate permissions to drop conversation participants.
	                                */
};

/**
 * Provides additional reason information for the Left and Closed message types.
 */
typedef NS_ENUM(NSUInteger, CSMessagingImportance) {
	CSMessagingImportanceNormal = 0, /**< The message reason is not specified. */
	CSMessagingImportanceLow, /**< The message is a result of a manual intervention. */
	CSMessagingImportanceHigh /**< The message is generated by the system. */
};

/**
 * Represents the sensitivity of the conversation.
 */
typedef NS_ENUM(NSUInteger, CSMessagingSensitivityLevel) {
	CSMessagingSensitivityLevelNormal = 0, /**< Normal sensitivity level. */
	CSMessagingSensitivityLevelPersonal, /**< Personal sensitivity level. */
	CSMessagingSensitivityLevelPrivate, /**< Private sensitivity level. */
	CSMessagingSensitivityLevelConfidential /**< Confidential sensitivity level. */
};

/**
 ** Represents the participant type.
 */
typedef NS_ENUM(NSUInteger, CSMessagingParticipantType) {
	CSMessagingParticipantTypeUndefined = 0, /**< Undefined participant type. */
	CSMessagingParticipantTypeContact, /**< Contact participant type. */
	CSMessagingParticipantTypeExternal, /**< External participant type. */
	CSMessagingParticipantTypeTopic, /**< Topic participant type. */
	CSMessagingParticipantTypeTeam, /**< Team participant type. */
	CSMessagingParticipantTypeDistributionList, /**< DistributionList participant type. */
	CSMessagingParticipantTypeSysAdmin /**< SystemAdmin participant type. */
};

/**
 * Represents the messaging source that the participant comes from.
 */
typedef NS_ENUM(NSUInteger, CSMessagingParticipantSourceType) {
	CSMessagingParticipantSourceTypeEnterprise, /**< Participant comes from enterprise source. */
	CSMessagingParticipantSourceTypeCloud, /**< Participant comes from cloud source. */
	CSMessagingParticipantSourceTypeUndefined /**< Source type could not be determined. */
};

/**
 * Represents type of phone number.
 */
typedef NS_ENUM(NSUInteger, CSMessagingParticipantPhoneNumberType) {
	CSMessagingParticipantPhoneNumberTypeUndefined, /**< Type could not be determined. */
	CSMessagingParticipantPhoneNumberTypeWork, /**< Work phone number type. */
	CSMessagingParticipantPhoneNumberTypeHome, /**< Home phone number type. */
	CSMessagingParticipantPhoneNumberTypeMobile, /**< Mobile phone number type. */
	CSMessagingParticipantPhoneNumberTypePager, /**< Pager phone number type. */
	CSMessagingParticipantPhoneNumberTypeFax, /**< Fax phone number type. */
	CSMessagingParticipantPhoneNumberTypeCustom /**< Custom phone number type. Actual value can be read from CSMessagingParticipantPhoneNumber.customType. */
};

/**
 * Messaging display order
 */
typedef NS_ENUM(NSInteger, CSMessagingDisplayOrder) {
	CSMessagingDisplayOrderFirst,
	CSMessagingDisplayOrderLast
};

/**
 * Messaging sort order
 */
typedef NS_ENUM(NSInteger, CSMessagingSortOrder) {
	CSMessagingSortOrderFirst,
	CSMessagingSortOrderLast
};

/**
 * Represents the conversation type.
 */
typedef NS_ENUM(NSInteger, CSMessagingConversationType) {
	/**
     * A published conversation to which any user may subscribe. Topics are based on a
	 * social-networking interaction model. Users elect to subscribe to a conversation of interest
	 * to them, and can join, leave, and rejoin at their discretion. All content of a topic is
	 * available to all who observe it, very much like an internet forum or message board.
	 *
	 * This conversation type is not currenlty supported.
	 */
	CSMessagingConversationTypeTopic,

	/**
	 * A private conversation between participants. The participants in a thread must each be
	 * invited to the conversation (either at the outset, or over the course of the conversation).
	 * Participants can elect to leave a thread, but cannot rejoin unless re-invited by an active
	 * participant. The content of a thread is not the same for all participants; it is directly
	 * related to when a user has joined and left the conversation. You are only aware of
	 * conversation activity that occurs while you are an active participant.
	 */
	CSMessagingConversationTypeThread,
    
    /**
     * A peer to peer conversation. Adding and removing additional participants is not allowed.
     * Participants cannot leave thread.
     */
    CSMessagingConversationTypeP2P
};

/**
 * Messaging participant modality
 */
typedef NS_ENUM (NSUInteger, CSMessagingParticipantModality) {
	CSMessagingParticipantModalityUndefined = 0,
	CSMessagingParticipantModalityText,
	CSMessagingParticipantModalityAudio,
	CSMessagingParticipantModalityVideo,
	CSMessagingParticipantModalityImage,
	CSMessagingParticipantModalityGenericAttachment
};

/**
 * Represents the message attachment status.
 */
typedef NS_ENUM(NSInteger, CSMessagingAttachmentStatus) {
	CSMessagingAttachmentStatusDraft,		/**< The attachment hasn't been sent yet. */
	CSMessagingAttachmentStatusSending,		/**< The attachment is currently being sent. */
	CSMessagingAttachmentStatusReadyToDownload, /**< Attachment is ready to be downloaded. */
	CSMessagingAttachmentStatusDownloading, /**< Attachment is being downloaded. */
	CSMessagingAttachmentStatusReadyToOpen, /**< Attachment is ready to consume. */
	CSMessagingAttachmentStatusOpened,		/**< Attachment was already opened by user. */
	CSMessagingAttachmentStatusDraftError,	/**< Failed to send the attachment. The messaging limits for attachments may have been exceeded. */
	CSMessagingAttachmentStatusDraftRemoving /**< Attachment is being removed from the message. */
};

/**
 * Represents the server reconciled status of the conversation.
 */
typedef NS_ENUM(NSInteger, CSMessagingConversationStatus) {
	CSMessagingConversationStatusUnsent,		/**< Represents a conversation that has been created by the local application but has not yet been sent. */
	CSMessagingConversationStatusSending,		/**< Used for conversations which are currently being sent. */
	CSMessagingConversationStatusRemoving,		/**< Used for conversations which are currently being removed. */
	CSMessagingConversationStatusError,			/**< State indicating that conversation couldn't be sent. Additional error information is provided within the operation result. */
	CSMessagingConversationStatusSent,			/**< A conversation in Published state has been successfully sent to the server and is now able to be viewed by other participants. */
	CSMessagingConversationStatusLeaving,		/**< Used for conversations which are currently being left. */
	CSMessagingConversationStatusLeft,			/**< Used for conversations which were left. */
};

/**
 * Message status
 */
typedef NS_ENUM(NSInteger, CSMessagingMessageStatus) {
	CSMessagingMessageStatusUnsent,		/**< Represents a message that has been created by the local application but has not yet been sent. */
	CSMessagingMessageStatusSending,	/**< Used for messages which are currently being sent. */
	CSMessagingMessageStatusRemoving,	/**< Used for messages which are currently being removed. */
	CSMessagingMessageStatusError,		/**< State indicating that message couldn't be sent. Additional error information is provided within the operation result. */
	CSMessagingMessageStatusSent		/**< A message in Published state has been successfully sent to the server and is now able to be viewed by other participants in the conversation. */
};

/**
 * Messaging session type
 */
typedef NS_ENUM(NSUInteger, CSMessagingSessionType) {
	CSMessagingSessionTypeUndefined = 0,
	CSMessagingSessionTypeAudio,
	CSMessagingSessionTypeVideo
};

/**
 * Represents read state of a message.
 */
typedef NS_ENUM(NSInteger, CSMessagingMessageReadState) {
	CSMessagingMessageReadStateUnknown,			/**< Unknown status. Default value for outgoing messages. */
	CSMessagingMessageReadStateNotRead,			/**< Not read by any recipient (outgoing message) or not read by me (incoming messages). */
	CSMessagingMessageReadStateRead,			/**< Read by all recipients (outgoing message) or marked as read by me (incoming message). */
	CSMessagingMessageReadStatePartiallyRead	/**< Outgoing message read by certain recipients. */
};

/**
 * Represents delivery state of a message.
 */
typedef NS_ENUM(NSInteger, CSMessagingMessageDeliveryState) {
	CSMessagingMessageDeliveryStateUnsent,				/**< Default value for draft outgoing message. */
	CSMessagingMessageDeliveryStateSent,				/**< Delivered to messaging server (outgoing message sent successfully). */
	CSMessagingMessageDeliveryStateDelivered			/**< Delivered to one or more recipients (both incoming and outgoing messages). */
};
