/* Copyright Avaya LLC. */

#import <Foundation/Foundation.h>

/**
 * The button type values.
 */
typedef NS_ENUM(NSInteger, CSButtonType) {
	/** Not specified. */
	CSButtonTypeUnspecified = 0,
	/** Button being customized is own / bridged line appearance. */
	CSButtonTypeLineAppearance,
	/** Button being customized is an AST feature or Agent feature. */
	CSButtonTypeFeature,
	/** Button being customized is an autodial feature. */
	CSButtonTypeAutodial,
	/** Button being customized is a contact. */
	CSButtonTypeContact,
	/** Button being customized is an application. */
	CSButtonTypeApplication
};

/**
 * The parameters that can be present in a customized button.
 */
@interface CSCustomizedButton : NSObject

/**
 * Returns the physical button index as configured on SMGR.
 */
@property (nonatomic, readonly) NSInteger configuredLocation;

/**
 * Returns the button index as customized on application's button module by the user.
 */
@property (nonatomic, readonly) NSInteger customizedLocation;

/**
 * Returns the button type for the button being customized.
 */
@property (nonatomic, readonly) CSButtonType buttonType;

/**
 * Indicates whether the button being customized is an agent feature when
 * the button being customized is a feature.
 */
@property (nonatomic, readonly) BOOL isAgentFeature;

/**
 * Returns the feature type of the feature when the button being customized is a feature.
 * Returns CSFeatureType (for AST feature) or CSAgentFeatureType (for Agent feature)
 * casted to an NSInteger.
 */
@property (nonatomic, readonly) NSInteger featureType;

/**
 * Returns the application name of the application when the button being customized is an application.
 */
@property (nonatomic, readonly) NSString *applicationName;

/**
 * Returns the line owner of the line-appearance when the button being customized is a line-appearance.
 */
@property (nonatomic, readonly) NSString *lineOwner;

/**
 * Returns the line ID of the line-appearance when the button being customized is a line-appearance.
 */
@property (nonatomic, readonly) NSString *lineID;

/**
 * Returns any additional parameter of the feature when the button being customized is a feature.
 */
@property (nonatomic, readonly) NSString *featureParameter;

/**
 * Returns the unique address of a contact when the button being customized is a contact.
 */
@property (nonatomic, readonly) NSString *uniqueAddressForMatching;

/**
 * Returns the contact number of a contact when the button being customized is a contact.
 */
@property (nonatomic, readonly) NSString *contactNumber;

/**
 * Returns the personalized label for the customized button.
 */
@property (nonatomic, readonly) NSString *personalizedLabel;

/**
 * Indicates whether the set personalized label operation is allowed.
 *
 * This property is used only for the contact buttons.
 * The set personalized label operation is always allowed for the other buttons.
 */
@property (nonatomic, readonly) BOOL canSetPersonalizedLabel;

/**
 * Constructs a CSCustomizedButton object from the specified parameters.
 *
 * @param configuredLocation The physical button index as configured on SMGR.
 * @param customizedLocation The button index as customized on application's button module by the user.
 * @param buttonType The button type.
 * @param isAgentFeature Indicates whether the button is an agent feature.
 * @param featureType The feature type of the feature.
 * @param applicationName The application name of the application.
 * @param lineOwner The line owner of the line-appearance.
 * @param lineID The line ID of the line-appearance.
 * @param featureParameter Additional parameter of the feature.
 * @param uniqueAddressForMatching The unique address of a contact.
 * @param contactNumber The contact number of a contact.
 * @param personalizedLabel The personalized label for the customized button.
 * @param canSetPersonalizedLabel Indicates whether the set personalized label operation is allowed.
 */
-(instancetype)initWithConfiguredLocation:(NSInteger)configuredLocation
                       customizedLocation:(NSInteger)customizedLocation
                               buttonType:(CSButtonType)buttonType
                           isAgentFeature:(BOOL)isAgentFeature
                              featureType:(NSInteger)featureType
                          applicationName:(NSString*)applicationName
                                lineOwner:(NSString*)lineOwner
                                   lineID:(NSString*)lineID
                         featureParameter:(NSString*)featureParameter
                 uniqueAddressForMatching:(NSString*)uniqueAddressForMatching
                            contactNumber:(NSString*)contactNumber
                        personalizedLabel:(NSString*)personalizedLabel
                  canSetPersonalizedLabel:(BOOL)canSetPersonalizedLabel;

- (instancetype)initWithButtonType:(CSButtonType)buttonType;

- (instancetype)initWithUniqueAddressForMatching:(NSString*)uniqueAddressForMatching
                                   contactNumber:(NSString*)contactNumber;

@end
