/* Copyright Avaya LLC. */

#import <Foundation/Foundation.h>

#import "CSHIDButtonType.h"
#import "CSHIDDeviceConnectionStatus.h"

@class CSHIDService;
/**
 *  Describes methods that are implemented by the delegate of a @c CSHIDService.
 */
@protocol CSHIDServiceDelegate <NSObject>

/**
 * Sent when the HID service has been activated.
 *
 * @param hidService The HID service instance sending the message.
 */
- (void)hidServiceDidActivate:(CSHIDService *)hidService;

/**
 * Sent when the HID service has been deactivated.
 *
 * @param hidService The HID service instance sending the message.
 */
- (void)hidServiceDidDeactivate:(CSHIDService *)hidService;

/**
 * Sent when a button pressed on the HID device was not handled
 * automatically by the Client SDK.
 *
 * @param hidService The HID service instance sending the message.
 * @param button The button that was not handled.
 */
- (void)hidService:(CSHIDService*)hidService buttonPressUnhandled:(CSHIDButtonType)button;

@optional

/**
 * Sent when the connection status to the HID device was changed.
 *
 * @param hidService The HID service instance sending the message.
 * @param deviceName The name of the HID device.
 * @param status The status of the HID device connection
 */
- (void)hidService:(CSHIDService*)hidService deviceName:(NSString*)deviceName didChangeConnectionStatus:(CSHIDDeviceConnectionStatus)status;

@end

/**
 * This service provides information about the Human Interface Device (HID) 
 * devices connected to the machine, along with associated button presses that
 * are NOT automatically handled by the SDK.
 *
 * In computing, the USB human interface device class (USB HID class) is a part
 * of the USB specification for computer peripherals: it specifies a device
 * class (a type of computer hardware) for human interface devices such as
 * keyboards, mice, game controllers and alphanumeric display devices.
 * In the context of the SDK, these devices are typically USB-connected
 * telephony headsets or speakerphones.
 *
 * Note that any button presses that are automatically handled by the SDK are
 * not reported.
 *
 * Currently, the only button reported is CSHIDButtonTypeRedial.
 */
@interface CSHIDService : NSObject

/**
 * Updates a list of devices for which call control should be deactivated or disallowed
 *
 * @param devicesToExclude An array containing GUIDs of call controllable devices for which call control
 *        should be deactivated (if already selected & activated) or disallowed (if selected at a later time).
 *        GUIDs of call controllable devices are as received from audioInterface
 */
- (void)updateDevicesForCallControlExclusion:(NSArray *)devicesToExclude;

/**
 * Notifications about service state change.
 */
@property (nonatomic, weak) id<CSHIDServiceDelegate> delegate;

@end

