/******************************************************************************/
/*                                                                            */
/* Copyright Avaya LLC.                                                       */
/*                                                                            */
/******************************************************************************/

#import "CallLogViewController.h"
#import "SDKManager.h"

@interface CallLogViewController ()

@end

@implementation CallLogViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    
    // Change window background to white
    self.view.wantsLayer = YES;
    self.view.layer.backgroundColor = [NSColor whiteColor].CGColor;
    
    self.tableView.delegate = self;
    
    NSColor *color = [NSColor blueColor];
    NSMutableAttributedString *formattedTitle = [[NSMutableAttributedString alloc] initWithAttributedString:self.backBtnLabel.attributedTitle];
    NSRange titleRange = NSMakeRange(0,formattedTitle.length);
    [formattedTitle addAttribute:NSForegroundColorAttributeName value:color range:titleRange];
    self.backBtnLabel.attributedTitle = formattedTitle;
}

- (IBAction)backBtn:(id)sender {
    
    [self dismissController:self];
}

- (NSInteger)numberOfRowsInTableView:(NSTableView *)tableView {
    
    CSCallLogService *callLogService = nil;
    for (CSUser *user in [SDKManager getInstance].users) {
        
        if (user.callLogService) {
            
            callLogService = user.callLogService;
            break;
        }
    }
    
    NSLog(@"%s log count: [%lu]", __PRETTY_FUNCTION__, callLogService.callLogs.count);
    return callLogService.callLogs.count;
}

- (NSView*)tableView:(NSTableView *)tableView viewForTableColumn:(NSTableColumn *)tableColumn row:(NSInteger)row {
    
    CSCallLogService *callLogService = nil;
    for (CSUser *user in [SDKManager getInstance].users) {
        
        if (user.callLogService) {
            
            callLogService = user.callLogService;
            break;
        }
    }
    
    NSArray *logItems = callLogService.callLogs;
    
    // Populate call log in reverse order i.e. newest entries first
    CSCallLogItem *item = logItems[logItems.count - (row + 1)];
    
    NSTableCellView *cellView = [tableView makeViewWithIdentifier:tableColumn.identifier owner:self];
    
    if ([tableColumn.identifier isEqualToString:@"callTime"]) {
        
        // Populate first column (Call Time)
        
        NSString *callTime = @"";
        
        NSCalendar *calendar = [[NSCalendar alloc] initWithCalendarIdentifier:NSCalendarIdentifierGregorian];
        NSDateComponents *components = [calendar components:NSCalendarUnitDay fromDate:[NSDate date] toDate:item.startTime options:0];
        
        NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
        dateFormatter.locale = [NSLocale currentLocale];
        
        if ([components day] == 0) { // Call happened today
            
            dateFormatter.dateStyle = NSDateFormatterNoStyle;
            dateFormatter.timeStyle = NSDateFormatterShortStyle;
            callTime = [dateFormatter stringFromDate:item.startTime];
        } else if ([components day] == -1) { // Call happened yesterday
            
            dateFormatter.timeStyle = NSDateFormatterShortStyle;
            callTime = [NSString stringWithFormat:@"Yesterday %@", [dateFormatter stringFromDate:item.startTime]];
        } else { // Call happened in past
            
            dateFormatter.dateStyle = NSDateFormatterShortStyle;
            dateFormatter.timeStyle = NSDateFormatterShortStyle;
            callTime = [dateFormatter stringFromDate:item.startTime];
        }
        cellView.textField.stringValue = [NSString stringWithFormat:@"%@", callTime];
    } else if ([tableColumn.identifier isEqualToString:@"callLogActionType"]) {
        
        // Populate second column (Call Log Action Type)
        NSString *mode = nil;
        
        switch (item.callLogActionType) {
            case CSCallLogActionTypeUndefined:
                mode = @"Undefined";
                break;
            case CSCallLogActionTypeMissed:
                mode = @"Missed";
                break;
            case CSCallLogActionTypeRedirected:
                mode = @"Redirected";
                break;
            case CSCallLogActionTypeAnswered:
                mode = @"Answered";
                break;
            case CSCallLogActionTypeOutgoing:
                mode = @"Outgoing";
                break;
            case CSCallLogActionTypeTransferred:
                mode = @"Transferred";
                break;
            case CSCallLogActionTypeIncoming:
                mode = @"Incoming";
                break;
            default:
                break;
        }
        cellView.textField.stringValue = mode;
        
    } else if ([tableColumn.identifier isEqualToString:@"callerName"]) {
        
        // Populate third column (Remote Party Name)
        NSString *callerName = @"Unknown";
        
        if (item.isCallerPrivate) {
            
            callerName = @"Restricted";
        } else if (item.isConference) {
            
            callerName = @"Conference";
        } else {
            
            callerName = [NSString stringWithFormat:@"%@", [item.remoteParticipants.firstObject displayName]];
        }
        cellView.textField.stringValue = callerName;
        
    } else if ([tableColumn.identifier isEqualToString:@"callerNumber"]) {
        
        // Poplate forth column (Remote Party Number)
        cellView.textField.stringValue =  item.remoteNumber;
        
    } else if ([tableColumn.identifier isEqualToString:@"callDuration"]) {
        
        // Poplate fifth column (Call Duration)
        NSInteger timeInterval = (NSInteger)item.durationInSeconds;
        NSInteger ss = timeInterval % 60;
        NSInteger mm = (timeInterval / 60) % 60;
        NSInteger hh = timeInterval / 3600;
        
        cellView.textField.stringValue = [NSString stringWithFormat:@"%@", [NSString stringWithFormat:@"%0.2ld:%0.2ld:%0.2ld", (long)hh, (long)mm, (long)ss]];
    }
    
    return cellView;
}

- (void)tableViewSelectionDidChange:(NSNotification *)notification {
    
    NSTableView *tb = notification.object;
    
    NSLog(@"%s selected row: [%lu]", __PRETTY_FUNCTION__, (long)tb.selectedRow + 1);
}
@end
