/* Copyright Avaya LLC. */

#import <Cocoa/Cocoa.h>
#import "CSScreenSharingListener.h"
#import "CSContentSharing.h"

/**
 * CSScreenSharingView object provides screen sharing canvas on which
 * received frames are drawn on. Should be set as rendering canvas of CSOSXScreenSharingView.
 * Client should hold strong reference to object of this class.
 *
 * @see CSOSXScreenSharingView
 */
@interface CSScreenSharingView : NSView <CSContentSharingDelegate>

/**
 * CSScreenSharingListener object which implements CSContentSharingDelegate
 * and is responsible for drawing incoming frames on CALayer object 
 * backing CSScreenSharingView. CSScreenSharingListener has to be created and set
 * by client.
 *
 * @see CSContentSharingDelegate
 */
@property (strong, nonatomic) CSScreenSharingListener *contentSharingDelegate;

/**
 * CSContentSharing object which is used by CSScreenSharingView to send remote
 * control related events.
 */
@property (weak, nonatomic) CSContentSharing *contentSharing;

/**
 * Image of cursor which is rendered on sharing canvas. By default image with name
 * cursor.png from application bundle is taken. In case image
 * is not provided neither in  application bundle nor by
 * this property, cursor isn't rendered on sharing canvas.
 */
@property (strong, nonatomic) NSImage *cursorImage;

/**
 * An image to be rendered in the center of sharing canvas when sharing
 * is paused. There's no default image for pause icon. In case image
 * is not provided by client using this property, pause image isn't rendered on
 * sharing canvas.
 */
@property (strong, nonatomic) NSImage *pauseImage;

/**
 * Requests remote control. Accesible also via {@link CSContentSharing::requestRemoteControlWithCompletionHandler:}.
 */
- (void)requestRemoteControlWithCompletionHandler:(void (^)(CSParticipant *participant, NSError *error))handler;

/**
 * Declines remote control. Accesible also via {@link CSContentSharing::declineRemoteControlWithCompletionHandler:}.
 */
- (void)declineRemoteControlWithCompletionHandler: (void (^)(CSParticipant *participant, NSError *error))handler;

/**
 * Starts remote control. Accesible also via {@link CSContentSharing::startRemoteControlWithCompletionHandler:}.
 */
- (void)startRemoteControlWithCompletionHandler: (void (^)(CSParticipant *participant, NSError *error))handler;

/**
 * Starts remote control. Accesible also via {@link CSContentSharing::endRemoteControlWithCompletionHandler:}.
 */
- (void)endRemoteControlWithCompletionHandler: (void (^)(CSParticipant *participant, NSError *error))handler;

/**
 * Send text to remote clipboard.
 *
 * @param text Text to be sent to remote clipboard.
 */
- (void)sendToClipboard:(NSString *)text;

/**
 * When called, updates view's tracking area used by remote control
 */
- (void)updateRemoteControlTrackingArea;

@end
