﻿using Avaya.ClientServices;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace CommunicationSampleApp.Classes
{
    class CollaborationManager
    {
        public static CollaborationManager CollaborationManagerInstance;
        public static CollaborationService CollaborationServiceInstance;
        public static Collaboration CollaborationInstance;
        public static ContentSharing ContentSharingInstance;
        public static bool IsDesktopSharingActive = false;
        public static bool IsContentSharingActive = false;
        public event EventHandler UpdateSharingControlUIEvent;

        public static CollaborationManager GetInstance()
        {
            if (CollaborationManagerInstance == null)
            {
                CollaborationManagerInstance = new CollaborationManager();
            }
            return CollaborationManagerInstance;
        }

        /// <summary>
        /// Add collaboration service events
        /// </summary>
        public void RegisterWebCollaborationServiceEvents()
        {
            CollaborationServiceInstance.CollaborationCreated += CollaborationCreated;
            CollaborationServiceInstance.CollaborationCreationFailed += CollaborationCreationFailed;
            CollaborationServiceInstance.CollaborationRemoved += CollaborationRemoved;
        }

        /// <summary>
        /// Remove collaboration service events
        /// </summary>
        public void UnregisterWebCollaborationServiceEvents()
        {
            CollaborationServiceInstance.CollaborationCreated -= CollaborationCreated;
            CollaborationServiceInstance.CollaborationCreationFailed -= CollaborationCreationFailed;
            CollaborationServiceInstance.CollaborationRemoved -= CollaborationRemoved;
        }

        public void CollaborationCreated(object sender, CollaborationEventArgs e)
        {
            Console.Write("CollaborationCreated");
            CollaborationInstance = e.Collaboration;
            RegisterCollaborationEvents(e.Collaboration);
        }

        public void CollaborationCreationFailed(object sender, CollaborationFailureEventArgs e)
        {
            Console.Write("CollaborationCreationFailed");
            NotificationHelper.ShowNotification("Collaboration Creation Failed.");
        }

        public void CollaborationRemoved(object sender, CollaborationEventArgs e)
        {
            Console.Write("CollaborationRemoved");
        }

        /// <summary>
        /// Add collaboration events
        /// </summary>
        /// <param name="collaboration"></param>
        public void RegisterCollaborationEvents(Collaboration collaboration)
        {
            collaboration.Started += collaborationStarted;
            collaboration.Ended += collaborationEnded;
        }

        private void collaborationStarted(object sender, EventArgs e)
        {
            Console.Write("collaborationStarted");
            RegisterContentSharingEvents(CollaborationInstance.ContentSharing);
            ContentSharingInstance = CollaborationInstance.ContentSharing;
        }

        private void collaborationEnded(object sender, EventArgs e)
        {
            Console.Write("collaborationEnded");
        }

        /// <summary>
        /// Register content sharing events
        /// </summary>
        /// <param name="contentSharing"></param>
        public void RegisterContentSharingEvents(ContentSharing contentSharing)
        {
            UpdateSharingControlUIEvent(this, EventArgs.Empty);
            contentSharing.Started += ContentSharingStarted;
            contentSharing.Ended += ContentSharingEnded;
            contentSharing.SharingFrameReceived += ContentSharingFrameReceived;
            contentSharing.SharingFrameChanged += ContentSharingFrameChanged;

        }

        public void ContentSharingStarted(Object sender, ContentSharingEventArgs e)
        {
            CollaborationWindow.collaborationWindow = new CollaborationWindow(ContentSharingInstance);
            CollaborationWindow.collaborationWindow.Show();
        }

        public void ContentSharingEnded(Object sender, EventArgs e)
        {
            IsDesktopSharingActive = false;
            IsContentSharingActive = false;
            if (CollaborationWindow.collaborationWindow != null)
            {
                CollaborationWindow.collaborationWindow.Close();
            }
            UpdateSharingControlUIEvent(this, EventArgs.Empty);
        }

        public void ContentSharingFrameReceived(Object sender, ContentSharingFrameEventArgs e)
        {
        }

        public void ContentSharingFrameChanged(Object sender, ContentSharingRectEventArgs e)
        {
            Console.Write("ContentSharingFrameChanged");
        }

        /// <summary>
        /// Starts sharing full screen
        /// </summary>
        public void StartSharingFullScreen()
        {
            if ((CollaborationInstance.ContentSharing != null)
                && (CollaborationInstance.ContentSharing.ShareFullScreenCapability.Allowed)
                && (CollaborationInstance.ContentSharing.Displays.Count > 0))
            {
                // Selects first display
                uint displayID = (uint)CollaborationInstance.ContentSharing.Displays.ElementAt(0).Key;
                CollaborationInstance.ContentSharing.StartSharingFullScreen(displayID, (sender, failureEventArgs) =>
                {
                    if (failureEventArgs == null)
                    {
                        Console.Write("Start sharing success");
                        IsDesktopSharingActive = true;
                        UpdateSharingControlUIEvent(this, EventArgs.Empty);
                    }
                    else
                    {
                        Console.Write("Start sharing failed : " + failureEventArgs.ProtocolReason);
                        NotificationHelper.ShowNotification("Start sharing failed : " + failureEventArgs.ProtocolReason);
                    }
                });
            }
            else
            {
                NotificationHelper.ShowNotification("Problem with ShareFullScreen");
            }

        }

        /// <summary>
        /// Stops sharing full screen
        /// </summary>
        public void StopSharingFullScreen()
        {
            CollaborationInstance.ContentSharing.End((sender, failureEventArgs) =>
            {
                if (failureEventArgs == null)
                {
                    Console.Write("Stop sharing success");
                    IsDesktopSharingActive = false;
                    UpdateSharingControlUIEvent(this, EventArgs.Empty);
                }
                else
                {
                    Console.Write("Stop sharing failed : " + failureEventArgs.ProtocolReason);
                    NotificationHelper.ShowNotification("Stop sharing failed : " + failureEventArgs.ProtocolReason);
                }
            });
        }
    }
}
