(function () {

    'use strict';

    function CollaborationController (user, call, template) {
        this.user = user;
        this.call = call;
        this.template = template;

        this.collaborations = undefined;
        this.collaboration = undefined;

        this.contentSharingController = undefined;
        this.whiteboardSharingController = undefined;

        this.view = new CollaborationView(this.template, {
            startContentSharing: this.startContentSharing.bind(this),
            endContentSharing: this.endContentSharing.bind(this),
            startWhiteboardSharing: this.startWhiteboardSharing.bind(this),
            endWhiteboardSharing: this.endWhiteboardSharing.bind(this),
            setZoom: this.setZoom.bind(this),
            setAutoFit: this.setAutoFit.bind(this),
        });

        this.init();
    };

    /**
     * Starts collaboration session if allowed by WCS configuration.
     */
    CollaborationController.prototype.init = function () {
        this.collaborations = this.user.getCollaborations();

        if (this.collaborations === undefined) {
            this.view.printError('Collaborations disabled by configuration');

            return;
        }

        this.collaboration = this.collaborations.getCollaborationForCall(this.call.getCallId());
        this.collaboration.start().then(function () {
            this.collaboration.addOnCollaborationEndedCallback(function () {
                this.view.printError('Collaboration session has ended');
            }.bind(this));

            this.collaboration.addOnCollaborationServiceUnavailableCallback(function () {
                this.view.printError('Disconnected from collaboration service');
            }.bind(this));

            this.setupSharing();
        }.bind(this)).catch(function (error) {
            console.error(error);

            this.view.printError('Failed to start collaboration');
        }.bind(this));
    };

    /**
     * Creates services to handle events from/to content and whiteboard sharing.
     */
    CollaborationController.prototype.setupSharing = function () {
        var collaborationCanvasId = 'collaborationCanvas' + this.call.getCallId();

        this.view.setCollaborationCanvasId(collaborationCanvasId);

        this.setupContentSharingController(collaborationCanvasId);
        this.setupWhiteboardSharingController(collaborationCanvasId);

        this.view.show();
    };
    
    /**
     * Create service for Content Sharing.
     */
    CollaborationController.prototype.setupContentSharingController = function (collaborationCanvasId) {
        this.contentSharingController = new ContentSharingController(this.collaboration,
                                                                     this.template,
                                                                     collaborationCanvasId);
        this.contentSharingController
            .getContentSharing()
            .addOnContentSharingStartedCallback(this.view.onContentSharingStarted.bind(this.view)); 
        this.contentSharingController
            .getContentSharing()
            .addOnContentSharingEndedCallback(this.view.onContentSharingEnded.bind(this.view));
    };

    /**
     * Create service for Whiteboard Sharing.
     */
    CollaborationController.prototype.setupWhiteboardSharingController = function (collaborationCanvasId) {
        this.whiteboardSharingController = new WhiteboardSharingController(this.collaboration,
                                                                           this.template,
                                                                           collaborationCanvasId);
        this.whiteboardSharingController
            .getWhiteboard()
            .addOnWhiteboardStartedCallback(this.view.onWhiteboardSharingStarted.bind(this.view));
        this.whiteboardSharingController
            .getWhiteboard()
            .addOnWhiteboardEndedCallback(this.view.onWhiteboardSharingEnded.bind(this.view));
    };

    /**
     * Try to start content sharing.
     */
    CollaborationController.prototype.startContentSharing = function () {
        this.contentSharingController.start();
    };

    /**
     * Try to end content sharing.
     */
    CollaborationController.prototype.endContentSharing = function () {
        this.contentSharingController.end();
    };

    /**
     * Try to start whiteboard sharing.
     */
    CollaborationController.prototype.startWhiteboardSharing = function () {
        this.whiteboardSharingController.start();
    };

    /**
     * Try to end whiteboard sharing.
     */
    CollaborationController.prototype.endWhiteboardSharing = function () {
        this.whiteboardSharingController.end();
    };

    /**
     * Sets zoom level to given value.
     *
     * @param {number} level
     */
    CollaborationController.prototype.setZoom = function (level) {
        if (this.contentSharingController.getContentSharing().isContentSharingActive()) {
            this.contentSharingController.setZoom(level);
        }
        else if (this.whiteboardSharingController.getWhiteboard().getActiveSurface()) {
            this.whiteboardSharingController.setZoom(level);
        }
        else {
            this.view.printError("Collaboration session is inactive");
        }
    };

    /**
     * Automatically fits the content within container.
     *
     * @param {number} width
     * @param {number} height
     */
    CollaborationController.prototype.setAutoFit = function (width, height) {
        if (this.contentSharingController.getContentSharing().isContentSharingActive()) {
            this.contentSharingController.setAutoFit(width, height);
        }
        else if (this.whiteboardSharingController.getWhiteboard().getActiveSurface()) {
            this.whiteboardSharingController.setAutoFit(width, height);
        }
        else {
            this.view.printError("Collaboration session is inactive");
        }
    };

    window.CollaborationController = CollaborationController;

})();
