(function () {

    'use strict';

    function ContentSharingController (collaboration, template, collaborationCanvasId) {
        this.collaboration = collaboration;
        this.template = template;
        this.collaborationCanvasId = collaborationCanvasId;

        this.contentSharing = undefined;
        this.contentSharingRenderer = undefined;

        this.view = new ContentSharingView(this.template);

        this.init();
    }

    /**
     * Creates renderer for Content Sharing to display received content;
     */
    ContentSharingController.prototype.init = function () {
        this.contentSharing = this.collaboration.getContentSharing();

        this.contentSharingRenderer = new AvayaClientServices.Renderer.Konva.KonvaContentSharingRenderer();
        this.contentSharingRenderer.init(this.contentSharing, this.collaborationCanvasId);

        this.contentSharing.addOnContentSharingStartedCallback(
            this.maybeSetOutgoingStream.bind(this)
        );

        this.contentSharing.addOnContentSharingEndedCallback(
            this.view.onEnded.bind(this.view)
        );
    };

    /**
     * Returns current instance.
     *
     * @returns {AvayaClientServices.Services.Collaboration.ContentSharing}
     */
    ContentSharingController.prototype.getContentSharing = function () {
        return this.contentSharing;
    };

    /**
     * Sets zoom level to given value.
     *
     * @param {number} level
     */
    ContentSharingController.prototype.setZoom = function (level) {
        if (this.contentSharing.isContentSharingActive()) {
            this.contentSharingRenderer.zoom(level);
        }
        else {
            this.view.printError("Content sharing is not running");
        }
    };

    /**
     * Automatically fits the content within container.
     *
     * @param {number} width
     * @param {number} height
     */
    ContentSharingController.prototype.setAutoFit = function (width, height) {
        if (this.contentSharing.isContentSharingActive()) {
            this.contentSharingRenderer.autoFit(width, height);
        }
        else {
            this.view.printError("Content sharing is not running");
        }
    };

    /**
     * Starts screen sharing session if it is allowed and not already active.
     */
    ContentSharingController.prototype.start = function () {
        if (this.collaboration.getContentSharingCapability().isAllowed === true &&
            this.contentSharing.getStartScreenSharingCapability().isAllowed === true) {
            if (this.contentSharing.isContentSharingActive()) {
                this.view.printError('Content sharing is already started');
            }
            else {
                this.contentSharing.startScreenSharing().catch(function () {
                    this.view.printError('Content sharing is failed to start');
                }.bind(this));
            }
        }
        else {
            this.view.printError('Content sharing is not allowed');
        }
    };

    /**
     * Ends active screen sharing session if present.
     */
    ContentSharingController.prototype.end = function () {
        if (this.collaboration.getContentSharingCapability().isAllowed === true) {
            if (this.contentSharing.isContentSharingActive()) {
                this.contentSharing.end().catch(function () {
                    this.view.printError('Content sharing is failed to stop');
                }.bind(this));
            }
            else {
                this.view.printError('Content sharing is already stopped');
            }
        }
        else {
            this.view.printError('Content sharing is not allowed');
        }
    };
    
    /**
     * Provides outgoing screen sharing stream to the view.
     */
    ContentSharingController.prototype.maybeSetOutgoingStream = function () {
        if (this.contentSharing.isPresenting()) {
            this.view.setOutgoingStream(this.contentSharing.getOutgoingScreenSharingStream());
        }
    };

    window.ContentSharingController = ContentSharingController;

})();