(function (AvayaClientServices, window, $) {
    'use strict';

    function SettingsController () {
        this._client = new AvayaClientServices();
        this._client.registerLogger(console);

        this._config = {};
        this._user = undefined;

        this._settingsView = new SettingsView({
            clientVersion: this._client.getVersion(),
            save: this.save.bind(this),
        });

        this._callsController = undefined;
        this._contactsController = undefined;
    }

    SettingsController.prototype = {
        /**
         * Save settings to access Web Gateway and Device Services, configure
         * an instance of Communication Services Package based on input.
         *
         * @param {Object} settings
         * @param {string} settings.hostname
         * @param {number} settings.port
         * @param {string} [settings.login]
         * @param {string} [settings.password]
         */
        save: function (settings) {
            if (settings.hostname) {
                this.createUser(settings.hostname,
                                settings.port,
                                settings.login,
                                settings.password);
            }
            else {
                this._settingsView.clientRegistrationFailed('missing server info');
            }
        },

        /**
         * Creates a user.
         *
         * @param {string} hostname
         * @param {number} port
         * @param {string} [login]
         * @param {string} [password]
         */
        createUser: function (hostname, port, login, password) {
            this._config.callUserConfiguration = {
                videoEnabled: true,
            };

            var useSecureConnection = true;
            var serverInfo = new AvayaClientServices.Config.ServerInfo(hostname,
                                                                       port,
                                                                       useSecureConnection);
            var connection = new AvayaClientServices.Config.NetworkProviderConfiguration(serverInfo);

            this._config.sgConfiguration = {
                enabled: true,
                credentialProvider: new AvayaClientServices.Config.GuestCredentialProvider(),
                networkProviderConfiguration: connection,
            };

            this._config.uccpConfiguration = {
                enabled: true,
                credentialProvider: null,
                networkProviderConfiguration: {
                    webSocketConfig: {}
                }
            };

            this._config.wcsConfiguration = {
                enabled: true,
            };

            if (login && password) {
                var credentials = new AvayaClientServices.Config.CredentialProvider(login,
                                                                                    password);
                this._config.acsConfiguration = {
                    enabled: true,
                    credentialProvider: credentials,
                    networkProviderConfiguration: connection,
                }
            }

            this._user = this._client.createUser(this._config);

            this._initServices(this._user, login, password);

            this._user.start().then(function () {
                this._settingsView.clientRegistrationSuccessful();
            }.bind(this), function () {
                this._settingsView.clientRegistrationFailed("unable to start services");
            }.bind(this));
        },

        /**
         * Services initialization.
         *
         * @param {Object} user
         * @param {string} login
         * @param {string} password
         */
         _initServices: function (user, login, password) {
            user.getCalls().addOnCallsServiceAvailableCallback(function () {
                this._callsController = new CallsController(user,
                                                            this._client,
                                                            login,
                                                            password);
            }.bind(this));

            if (user.getContacts()) {
                user.getContacts().addOnContactsServiceAvailableCallback(function () {
                    this._contactsController = new ContactsController(user);
                    this._settingsView.showContactsTab();
                }.bind(this));
            }
        }
    };

    window.SettingsController = SettingsController;

})(AvayaClientServices, window, jQuery);
