(function () {

    'use strict';

    function CollaborationView (template, handlers) {
        BaseView.apply(this);

        this.template = template;
        this.handlers = handlers;

        this.collaborationPanel = this.template.find('.conferenceCollaborationPanel');
        this.collaborationCanvas = this.template.find('.collaborationCanvas');
        
        this.startContentSharing = this.template.find('#startContentSharing');
        this.endContentSharing = this.template.find('#endContentSharing');

        this.startWhiteboardSharing = this.template.find('#startWhiteboardSharing');
        this.endWhiteboardSharing = this.template.find('#endWhiteboardSharing');

        this.isContentSharingActive = false;
        this.isWhiteboardSharingActive = false;

        this.zoom = this.template.find('#collaborationZoom');
        this.zoomValue = this.template.find('#collaborationZoomValue');
        this.zoomAutoFit = this.template.find('#collaborationAutoFit');

        this.zoomInitialValue = '1';
        this.lastContentSharingZoomValue = undefined;
        this.lastWhiteboardSharingZoomValue = undefined;

        this.init();
    }

    CollaborationView.prototype = Object.create(BaseView.prototype);

    /**
     * Assigns handlers to start/end button clicks.
     */
    CollaborationView.prototype.init = function () {
        this.startContentSharing.on('click', this.handlers.startContentSharing);
        this.endContentSharing.on('click', this.handlers.endContentSharing);

        this.startWhiteboardSharing.on('click', this.handlers.startWhiteboardSharing);
        this.endWhiteboardSharing.on('click', this.handlers.endWhiteboardSharing);

        this.initZoom();
    };

    /**
     * Sets up zoom controls.
     */
    CollaborationView.prototype.initZoom = function () {
        var menu = this.zoom.next('.dropdown-menu');

        for (var level = 0.1, max = 2; level < max; level = level + 0.1) {
            menu.append(this.createZoomMenuItem(level.toFixed(1)));
        }

        this.zoom.parent().find('.zoomLevel').each(function (index, item) {
            $(item).on('click', function (event) {
                event.preventDefault();

                var value = Number(event.target.innerHTML);

                if (this.isContentSharingActive) {
                    this.lastContentSharingZoomValue = value;
                }
                else if (this.isWhiteboardSharingActive) {
                    this.lastWhiteboardSharingZoomValue = value;
                }

                this.zoomValue.html(value);

                this.handlers.setZoom(value);
            }.bind(this));
        }.bind(this));

        this.zoomAutoFit.on('click', function () {
            this.zoomValue.html('Auto-Fit');

            this.handlers.setAutoFit(
                this.collaborationCanvas.get(0).clientWidth,
                this.collaborationCanvas.get(0).clientHeight,
            );
        }.bind(this));
    };

    /**
     * Helps to create a link for zoom level menu.
     *
     * @param {string} level
     * @returns {Object}
     */
    CollaborationView.prototype.createZoomMenuItem = function (level) {
        return $('<li><a href="#" class="zoomLevel">' + level + '</a></li>');
    };

    /**
     * Reveals collaboration page.
     */
    CollaborationView.prototype.show = function () {
        this.collaborationPanel.show();
    };

    /**
     * Updates ID of the container where canvas will be rendered. It is crucial
     * for Content Sharing and Whiteboard Sharing renderers to use an ID as they
     * won't accept other kind of selector.
     * 
     * @param {string} id
     */
    CollaborationView.prototype.setCollaborationCanvasId = function (id) {
        this.collaborationCanvas.attr('id', id);
    };

    /**
     * Rearranges controls when content sharing has started.
     */
    CollaborationView.prototype.onContentSharingStarted = function () {
        this.isContentSharingActive = true;

        this.startWhiteboardSharing.prop('disabled', true);
        this.endWhiteboardSharing.prop('disabled', true);
        
        this.startContentSharing.hide();
        this.endContentSharing.show();

        if (this.lastContentSharingZoomValue) {
            this.zoomValue.html(this.lastContentSharingZoomValue);
        }

        this.zoom.prop('disabled', false);
        this.zoomAutoFit.prop('disabled', false);
    };

    /**
     * Rearranges controls when content sharing has ended.
     */
    CollaborationView.prototype.onContentSharingEnded = function () {
        this.isContentSharingActive = false;

        this.startWhiteboardSharing.prop('disabled', false);
        this.endWhiteboardSharing.prop('disabled', false);

        this.startContentSharing.show();
        this.endContentSharing.hide();

        this.zoom.prop('disabled', true);
        this.zoomAutoFit.prop('disabled', true);
    };

    /**
     * Rearranges controls when whiteboard sharing has started.
     */
    CollaborationView.prototype.onWhiteboardSharingStarted = function () {
        this.isWhiteboardSharingActive = true;

        this.startContentSharing.prop('disabled', true);
        this.endContentSharing.prop('disabled', true);

        this.startWhiteboardSharing.hide();
        this.endWhiteboardSharing.show();

        if (this.lastWhiteboardSharingZoomValue) {
            this.zoomValue.html(this.lastWhiteboardSharingZoomValue);
        }

        this.zoom.prop('disabled', false);
        this.zoomAutoFit.prop('disabled', false);

        this.zoomValue.html(this.zoomInitialValue);
    };

    /**
     * Rearranges controls when whiteboard sharing has ended.
     */
    CollaborationView.prototype.onWhiteboardSharingEnded = function () {
        this.isWhiteboardSharingActive = false;

        this.startContentSharing.prop('disabled', false);
        this.endContentSharing.prop('disabled', false);

        this.startWhiteboardSharing.show();
        this.endWhiteboardSharing.hide();

        this.zoom.prop('disabled', true);
        this.zoomAutoFit.prop('disabled', true);

        this.zoomValue.html(this.zoomInitialValue);
    };

    window.CollaborationView = CollaborationView;

})();
