(function () {

    'use strict';

    function WhiteboardSharingController (collaboration, template, collaborationCanvasId) {
        this.collaboration = collaboration;
        this.template = template;
        this.collaborationCanvasId = collaborationCanvasId;

        this.whiteboard = undefined;
        this.whiteboardRenderer = undefined;

        this.drawingTools = new AvayaClientServices.Services.Collaboration.WhiteboardTools();
        this.currentToolType = AvayaClientServices.Services.Collaboration.WhiteboardToolTypes.RECTANGLE;
        this.lineWidth = 'SMALL';
        this.textSize = 'MEDIUM';
        this.toolMode = 'STROKE';

        this.view = new WhiteboardSharingView(this.template, {
            addSurface: this.addSurface.bind(this),
            setActiveSurface: this.setActiveSurface.bind(this),
            removeSurface: this.removeSurface.bind(this),
            setLineWidth: this.setLineWidth.bind(this),
            setTextSize: this.setTextSize.bind(this),
            setToolMode: this.setToolMode.bind(this),
            setToolType: this.setToolType.bind(this),
            setColor: this.setColor.bind(this),
        });

        this.init();
    }

    WhiteboardSharingController.prototype.init = function () {
        this.whiteboard = this.collaboration.getWhiteboard();

        this.whiteboardRenderer = new AvayaClientServices.Renderer.Konva.KonvaWhiteboardRenderer();
        this.whiteboardRenderer.init(this.whiteboard, this.collaborationCanvasId);

        this.whiteboard.addOnWhiteboardStartedCallback(this.view.onWhiteboardStarted.bind(this.view));
        this.whiteboard.addOnWhiteboardEndedCallback(this.view.onWhiteboardEnded.bind(this.view));

        this.whiteboard.addOnWhiteboardSurfaceAddedCallback(this.view.onSurfaceAdded.bind(this.view));
        this.whiteboard.addOnWhiteboardSurfaceRemovedCallback(this.view.onSurfaceRemoved.bind(this.view));
        this.whiteboard.addOnWhiteboardSurfaceSetActiveCallback(this.view.onSurfaceActivated.bind(this.view));

        this.initDrawingTools();
    };

    /**
     * Creates tool buttons for drawing on the whiteboard.
     */
    WhiteboardSharingController.prototype.initDrawingTools = function () {
        this.view.initToolbox(
            AvayaClientServices.Services.Collaboration.WhiteboardLineWidths,
            AvayaClientServices.Services.Collaboration.WhiteboardTextSizes,
            AvayaClientServices.Services.Collaboration.WhiteboardToolModes,
            AvayaClientServices.Services.Collaboration.WhiteboardToolTypes,
            this.whiteboardRenderer.possibleColors,
        );

        this.whiteboard.addOnWhiteboardStartedCallback(function (whiteboard) {
            if (whiteboard.getDrawingCapability().isAllowed === true) {
                this.view.unlockToolbox();
            }
            else {
                this.view.printError('Drawing on the whiteboard is not allowed');
            }
        }.bind(this));

        this.whiteboard.addOnWhiteboardEndedCallback(this.view.lockToolbox.bind(this.view));
    };

    /**
     * Add callback to the time when Whiteboard Sharing has started.
     *
     * @returns {AvayaClientServices.Services.Collaboration.Whiteboard}
     */
    WhiteboardSharingController.prototype.getWhiteboard = function () {
        return this.whiteboard;
    };

    /**
     * Sets zoom level to given value.
     *
     * @param {number} level
     */
    WhiteboardSharingController.prototype.setZoom = function (level) {
        if (this.whiteboard.getActiveSurface()) {
            this.whiteboardRenderer.zoom(level);
        }
        else {
            this.view.printError("Content sharing is not running");
        }
    };

    /**
     * Automatically fits the content within container.
     *
     * @param {number} width
     * @param {number} height
     */
    WhiteboardSharingController.prototype.setAutoFit = function (width, height) {
        if (this.whiteboard.getActiveSurface()) {
            this.whiteboardRenderer.autoFit(width, height);
        }
        else {
            this.view.printError("Content sharing is not running");
        }
    };

    /**
     * Starts whiteboard sharing session if it is allowed and not already active.
     */
    WhiteboardSharingController.prototype.start = function () {
        if (this.collaboration.getWhiteboardCapability().isAllowed === true) {
            if (this.whiteboard.getActiveSurface() === undefined) {
                this.whiteboard.start().catch(function (error) {
                    console.error(error);

                    this.view.printError('Whiteboard sharing is failed to start');
                }.bind(this));
            }
            else {
                this.view.printError('Whiteboard sharing is already started');
            }
        }
        else {
            this.view.printError('Whiteboard sharing is not allowed');
        }
    };

    /**
     * Ends whiteboard sharing session if active.
     */
    WhiteboardSharingController.prototype.end = function () {
        if (this.collaboration.getWhiteboardCapability().isAllowed === true) {
            if (this.whiteboard.getActiveSurface() === undefined) {
                this.view.printError('Whiteboard sharing is already stopped');
            }
            else {
                this.whiteboard.end().catch(function (error) {
                    console.error(error);

                    this.view.printError('Whiteboard sharing is failed to stop');
                }.bind(this));
            }
        }
        else {
            this.view.printError('Whiteboard sharing is not allowed');
        }
    };

    /**
     * Request a new surface to be added to the whiteboard.
     */
    WhiteboardSharingController.prototype.addSurface = function () {
        if (this.whiteboard.getAddWhiteboardCapability().isAllowed === true) {
            this.whiteboard.requestNewSurface().catch(function (error) {
                console.error(error);

                this.view.printError('Failed to request new surface');
            }.bind(this));
        }
        else {
            this.view.printError('Adding new surfaces is not allowed');
        }
    };

    /**
     * Requests to set surface with ID as an active one.
     *
     * @param {number} id
     */
    WhiteboardSharingController.prototype.setActiveSurface = function (id) {
        var surface = this.whiteboard.getSurfaceById(id);

        surface.setAsActive().catch(function (error) {
            console.error(error);

            this.view.printError('Failed to set surface as active');
        }.bind(this));
    };

    /**
     * Requests to remove surface with given ID.
     *
     * @param {number} id
     */
    WhiteboardSharingController.prototype.removeSurface = function (id) {
        var surface = this.whiteboard.getSurfaceById(id);

        if (this.whiteboard.getRemoveWhiteboardCapability().isAllowed === true) {
            this.whiteboard.removeSurface(surface).catch(function (error) {
                console.error(error);

                this.view.printError('Failed to remove a surface');
            });
        }
        else {
            this.view.printError('Removing surfaces is not allowed');
        }
    };

    /**
     * Sets line width for tools that supports that.
     *
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardLineWidths}
     */
    WhiteboardSharingController.prototype.setLineWidth = function (width) {
        this.lineWidth = width;

        this.setToolType(this.currentToolType);
    };

    /**
     * Sets text site for tools that supports that.
     *
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardToolTypes} type
     */
    WhiteboardSharingController.prototype.setTextSize = function (size) {
        this.textSize = size;

        this.setToolType(this.currentToolType);
    };

    /**
     * Sets if figured should be filled with color or not.
     *
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardToolModes}
     */
    WhiteboardSharingController.prototype.setToolMode = function (mode) {
        this.toolMode = mode;

        this.setToolType(this.currentToolType);
    };
    
    /**
     * Sets active tool for drawing/writing.
     *
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardToolTypes} type
     */
    WhiteboardSharingController.prototype.setToolType = function (type) {
        var tool = this.drawingTools[type];

        if (tool === undefined) {
            return this.view.printError('Unknown tool type: ' + type);
        }

        this.currentToolType = type;

        if (tool.modes !== undefined) {
            if (tool.modes === AvayaClientServices.Services.Collaboration.WhiteboardLineWidths) {
                return this.whiteboardRenderer.setTool(type, this.lineWidth);
            }

            if (tool.modes === AvayaClientServices.Services.Collaboration.WhiteboardTextSizes) {
                return this.whiteboardRenderer.setTool(type, this.textSize);
            }

            if (tool.modes === AvayaClientServices.Services.Collaboration.WhiteboardToolModes) {
                return this.whiteboardRenderer.setTool(type, this.toolMode);
            }
        }

        this.whiteboardRenderer.setTool(type);
    };
    
    /**
     * Sets color to be used for active tool.
     *
     * @param {string} color
     */
    WhiteboardSharingController.prototype.setColor = function (color) {
        this.whiteboardRenderer.setColor(color);
    };
    
    window.WhiteboardSharingController = WhiteboardSharingController;

})();