(function (window, $) {
    'use strict';

    function ContactController(contactsService) {
        this._contactsService = contactsService;
        this._contact = undefined;
        this._editableContact = undefined;
        this._contactView = new window.ContactView();
    }


    ContactController.prototype = {
        /**
         * Init function for Contact service.
         * Registration callbacks releated with Contact service.
         *
         * @param {Object} contact
         */
        init: function (contact) {
            this._contact = contact;

            this._editableContact = this._contactsService.createEditableContactFromContact(this._contact);
            this._contactView.resetContactForm();
            this._contactView.setContactModalTitle(this._editableContact.getDisplayName().getValue());

            var contactData = {
                contactFirstName: this._editableContact.getFirstName().getValue(),
                contactFirstNameASCII: this._editableContact.getASCIIFirstName().getValue(),
                contactAlias: this._editableContact.getAlias().getValue(),
                contactCompany: this._editableContact.getCompany().getValue(),

                contactLastName: this._editableContact.getLastName().getValue(),
                contactLastNameASCII: this._editableContact.getASCIILastName().getValue(),
                contactLocation: this._editableContact.getLocation().getValue(),
                contactManager: this._editableContact.getManager().getValue(),

                contactDisplayName: this._editableContact.getDisplayName().getValue(),
                contactDisplayNameASCII: this._editableContact.getASCIIDisplayName().getValue(),
                contactWorkRoomNumber: this._editableContact.getWorkRoomNumber().getValue(),
                contactDepartment: this._editableContact.getDepartment().getValue(),

                contactStreetAddress: this._editableContact.getStreetAddress().getValue(),
                contactCity: this._editableContact.getCity().getValue(),

                contactState: this._editableContact.getState().getValue(),
                contactNotes: this._editableContact.getNotes().getValue(),

                contactCountry: this._editableContact.getCountry().getValue(),
                contactScopiaUserId: this._editableContact.getScopiaUserId().getValue(),

                contactPostalCode: this._editableContact.getPostalCode().getValue(),

                phoneNumbers: this._editableContact.getPhoneNumbers().getValues(),
                emailAddresses: this._editableContact.getEmailAddresses().getValues(),
                IMAddresses: this._editableContact.getIMAddresses().getValues(),

                custom1: this._editableContact.getCustom1().getValues(),
                custom2: this._editableContact.getCustom2().getValues(),
                custom3: this._editableContact.getCustom3().getValues(),
            };
            if (this._contact.getTerminal()) {
                contactData.contactTerminalName = this._contact.getTerminal().getTerminalName();
                contactData.contactTerminalId = this._contact.getTerminal().getTerminalId();
            }
            this._contactView.setAllContactData(contactData);

            this._addExistedPhoneNumbers(
                this._editableContact.getPhoneNumbers().getValues()
            );

            this._addExistedEmailAddresses(
                this._editableContact.getEmailAddresses().getValues()
            );

            this._addExistedIMAddresses(
                this._editableContact.getIMAddresses().getValues()
            );

            this._contactView.showContactModal();
        },

        /**
         * Function add phone numbers to contact details.
         *
         * @param {Array} phoneNumbers
         * @private
         */
        _addExistedPhoneNumbers: function (phoneNumbers) {
            phoneNumbers.forEach(function (phoneNumber, index) {
                var phoneNumberData = {
                    index: index,
                    default: phoneNumber.isDefault(),
                    label1: phoneNumber.getLabel1(),
                    label2: phoneNumber.getLabel2(),
                    phoneNumber: phoneNumber.getPhoneNumber(),
                    speedDialEnabled: phoneNumber.isSpeedDialEnabled(),
                };
                switch (phoneNumber.getPhoneNumberType()) {
                    case AvayaClientServices.Services.Contacts.ContactPhoneNumberType.PHONE_NUMBER_WORK:
                        phoneNumberData.phoneNumberType = 'Work';
                        break;
                    case AvayaClientServices.Services.Contacts.ContactPhoneNumberType.PHONE_NUMBER_HANDLE:
                        phoneNumberData.phoneNumberType = 'SIP Handle';
                        break;
                    case AvayaClientServices.Services.Contacts.ContactPhoneNumberType.PHONE_NUMBER_MOBILE:
                        phoneNumberData.phoneNumberType = 'Mobile';
                        break;
                    case AvayaClientServices.Services.Contacts.ContactPhoneNumberType.PHONE_NUMBER_HOME:
                        phoneNumberData.phoneNumberType = 'Home';
                        break;
                    case AvayaClientServices.Services.Contacts.ContactPhoneNumberType.PHONE_NUMBER_FAX:
                        phoneNumberData.phoneNumberType = 'Fax';
                        break;
                    case AvayaClientServices.Services.Contacts.ContactPhoneNumberType.PHONE_NUMBER_PAGER:
                        phoneNumberData.phoneNumberType = 'Pager';
                        break;
                    default:
                        phoneNumberData.phoneNumberType = 'Work';
                }
                this._contactView.addPhoneNumber(phoneNumberData);
            }.bind(this));
        },

        /**
         * Function add email addresses to contact details.
         *
         * @param {Object} emailAddresses
         * @private
         */
        _addExistedEmailAddresses: function (emailAddresses) {
            emailAddresses.forEach(function (emailAddress, index) {
                var emailAddressData = {
                    index: index,
                    default: emailAddress.isDefault(),
                    emailAddress: emailAddress.getAddress(),
                };
                switch (emailAddress.getEmailAddressType()) {
                    case AvayaClientServices.Services.Contacts.ContactEmailAddressType.EMAIL_ADDRESS_WORK:
                        emailAddressData.emailAddressType = 'Work';
                        break;
                    case AvayaClientServices.Services.Contacts.ContactEmailAddressType.EMAIL_ADDRESS_OTHER:
                        emailAddressData.emailAddressType = 'Other';
                        break;
                    default:
                        emailAddressData.emailAddressType = 'Work';
                }
                this._contactView.addEmailAddress(emailAddressData);
            }.bind(this));
        },

        /**
         * Function add IM addresses to contact details.
         *
         * @param {Object} IMAddresses
         * @private
         */
        _addExistedIMAddresses: function (IMAddresses) {
            IMAddresses.forEach(function (IMAddress, index) {
                var IMAddressData = {
                    index: index,
                    default: IMAddress.isDefault(),
                    IMAddress: IMAddress.getAddress(),
                };
                switch (IMAddress.getIMAddressType()) {
                    case AvayaClientServices.Services.Contacts.ContactIMAddressType.IM_ADDRESS_WORK:
                        IMAddressData.IMAddressType = 'Work';
                        break;
                    case AvayaClientServices.Services.Contacts.ContactIMAddressType.IM_ADDRESS_OTHER:
                        IMAddressData.IMAddressType = 'Other';
                        break;
                    default:
                        IMAddressData.IMAddressType = 'Work';
                }
                this._contactView.addIMAddress(IMAddressData);
            }.bind(this));
        },
    };

    window.ContactController = ContactController;

})(window, jQuery);
