(function (window, $) {
    'use strict';

    function ConferenceView(template, handlers, hasVideo) {
        BaseView.apply(this);

        this._template = template;

        this._init();
        this._addHandlers(handlers);

        this._hasVideo = hasVideo;
    }

    ConferenceView.prototype = Object.create(BaseView.prototype);

    ConferenceView.prototype._init = function(){
        this.conferenceControlBtns = this._template.find(".conferenceControlBtns");
        this.conferencePanelTitle = this._template.find(".conferencePanelTitle");
        this.rosterPanel = this._template.find(".rosterPanel");
        this.participantRoster = this._template.find(".participantRoster");
        this.lockConferenceBtn = this._template.find(".lockConferenceBtn");
        this.unlockConferenceBtn = this._template.find(".unlockConferenceBtn");
        this.muteAllBtn = this._template.find(".muteAllBtn");
        this.unmuteAllBtn = this._template.find(".unmuteAllBtn");
        this.leaveConferenceBtn = this._template.find(".leaveConferenceBtn");
    };

    ConferenceView.prototype._addHandlers = function (handlers) {
        this.lockConferenceBtn.on('click', function () {
            handlers.handleLockUnlockConference();
        });

        this.unlockConferenceBtn.on('click', function () {
            handlers.handleLockUnlockConference();
        });

        this.muteAllBtn.on('click', function () {
            handlers.handleMuteUnmuteAll();
        });

        this.unmuteAllBtn.on('click', function () {
            handlers.handleMuteUnmuteAll();
        });

        this.leaveConferenceBtn.on('click', function () {
            handlers.handleLeaveConference();
        });
    };

    /**
     * Shows all of the conference panels.
     */
    ConferenceView.prototype.showConferencePanels = function () {
        this.conferencePanelTitle.show();
        this.conferenceControlBtns.show();
        this.rosterPanel.show();
    };

    /**
     * Hides all of the conference panels.
     */
    ConferenceView.prototype.hideConferencePanels = function () {
        this.conferenceControlBtns.hide();
        this.conferencePanelTitle.hide();
        this.rosterPanel.hide();
        this.participantRoster.empty();
    };

    /**
     * Allows user to Lock/Unlock the conference.
     */
    ConferenceView.prototype.unlockConferenceLockControls = function (allowed) {
        this.lockConferenceBtn.prop('disabled', !allowed);
        this.unlockConferenceBtn.prop('disabled', !allowed);
    };

    /**
     * Allows user to Mute All participants.
     */
    ConferenceView.prototype.unlockMuteAllControls = function (allowed) {
        this.muteAllBtn.prop('disabled', !allowed);
    };

    /**
     * Allows user to Unmute All participants.
     */
    ConferenceView.prototype.unlockUnmuteAllControls = function (allowed) {
        this.unmuteAllBtn.prop('disabled', !allowed);
    };

    /**
     * Adds participant label to roster.
     *
     * @param {String} id
     * @param {String} displayName
     * @param {Boolean} isVideoBlocked
     * @param {Boolean} isAudioMuted
     */
    ConferenceView.prototype.addParticipantToRoster = function (id, displayName, isVideoBlocked, isAudioMuted) {
        var participant = '<li class="list-group-item participant" id="roster_' + id + '"><span class="displayName">' + displayName + '</span>';
        if (this._hasVideo && ( ! isVideoBlocked)) {
            participant = participant + '<span class="pull-right glyphicon glyphicon-facetime-video"></span>';
        }

        if ( ! isAudioMuted) {
            participant = participant + '<span class="pull-right glyphicon glyphicon-volume-up"></span>';
        } else {
            participant = participant + '<span class="pull-right glyphicon glyphicon-volume-off"></span>';
        }

        participant = participant + '</li>';

        this.participantRoster.append(participant);
    };

    /**
     * Adds pending participant label to roster.
     *
     * @param {Object} participant
     */
    ConferenceView.prototype.addPendingParticipantToRoster = function (participant) {
        var id = participant.getParticipantId();
        var displayName = participant.getDisplayName();

        var participantHTML = '<li class="list-group-item" id="roster_' + id + '"><span class="displayName">' + displayName + '</span>';
        participantHTML = participantHTML + '<span class="pull-right glyphicon glyphicon-remove"></span>';
        participantHTML = participantHTML + '<span class="pull-right glyphicon glyphicon-ok"></span>';
        participantHTML = participantHTML + '</li>';
        this.participantRoster.append(participantHTML);

        var participantElemId = '#roster_' + id;
        var acceptBtn = $(participantElemId + ' > .glyphicon-ok');
        var denyBtn = $(participantElemId+ ' > .glyphicon-remove');

        // Accept handler
        acceptBtn.off('click').on('click', function () {
            participant.accept().then(function(){
                console.log('Participant ' +  id + ' was accepted successfully');
            }, function(error) {
                console.log('Attempt to accept participant ' +  id + ' was failed');
                if(error){
                    console.error(error);
                }
            });
        });

        // Deny handler
        denyBtn.off('click').on('click', function () {
            participant.deny().then(function(){
                console.log('Participant ' +  id + ' was denied successfully');
            }, function(error) {
                console.log('Attempt to deny participant ' +  id + ' was failed');
                if(error) {
                    console.error(error);
                }
            });
        });
    };

    /**
     * Removes participant label from roster.
     *
     * @param {String} id
     */
    ConferenceView.prototype.removeParticipantFromRoster = function (id) {
        var participant = $('#roster_' + id);
        if(!!participant) {
            participant.remove();
        }
    };

    /**
     * Updates participant label in roster.
     *
     * @param {String} id
     * @param {String} displayName
     * @param {Boolean} isVideoBlocked
     * @param {Boolean}  isAudioMuted
     */
    ConferenceView.prototype.updateParticipantInRoster = function (id, displayName, isVideoBlocked, isAudioMuted) {
        var participant = $('#roster_' + id);
        var html = '<span class="displayName">' + displayName + '</span>';
        if (this._hasVideo && ( ! isVideoBlocked)) {
            html = html + '<span class="pull-right glyphicon glyphicon-facetime-video"></span>';
        }

        if ( ! isAudioMuted) {
            html = html + '<span class="pull-right glyphicon glyphicon-volume-up"></span>';
        } else {
            html = html + '<span class="pull-right glyphicon glyphicon-volume-off"></span>';
        }

        participant.html(html);
    };

    /**
     * Toggle muteAll/unmuteAll buttons.
     *
     * @param {Boolean} isConferenceMuted
     */
    ConferenceView.prototype.changeMuteUnmuteAllBtn = function (isConferenceMuted) {
        if (isConferenceMuted) {
            this.muteAllBtn.hide();
            this.unmuteAllBtn.show();
        } else {
            this.muteAllBtn.show();
            this.unmuteAllBtn.hide();
        }
    };

    /**
     * Toggle lock/unlock buttons.
     *
     * @param {Boolean} isLocked
     */
    ConferenceView.prototype.changeLockUnlockBtn = function (isLocked) {
        if (isLocked) {
            this.lockConferenceBtn.hide();
            this.unlockConferenceBtn.show();
        } else {
            this.lockConferenceBtn.show();
            this.unlockConferenceBtn.hide();
        }
    };

    window.ConferenceView = ConferenceView;

})
(window, jQuery);
