(function () {
    
    'use strict';

    function WhiteboardSharingView (template, handlers) {
        BaseView.apply(this);

        this.template = template;
        this.handlers = handlers;

        this.controls = this.template.find('#whiteboardSharingControls');
        this.surfaces = this.template.find('#whiteboardSurfaces');
        this.addSurface = this.template.find('#whiteboardAddSurface');

        this.addSurface.on('click', this.handlers.addSurface);

        this.toolbox = this.template.find('#toolbox');
        this.toolboxLineWidths = this.template.find('#toolboxLineWidths');
        this.toolboxToolModes = this.template.find('#toolboxToolModes');
        this.toolboxToolTypes = this.template.find('#toolboxToolTypes');
        this.toolboxTextSizes = this.template.find('#toolboxTextSizes');
        this.toolboxColors = this.template.find('#toolboxColors');
    }

    WhiteboardSharingView.prototype = Object.create(BaseView.prototype);

    /**
     * Reveals controls for whiteboard when it has started.
     */
    WhiteboardSharingView.prototype.onWhiteboardStarted = function () {
        this.controls.show();
    };

    /**
     * Hides controls for whiteboard when it has ended.
     */
    WhiteboardSharingView.prototype.onWhiteboardEnded = function () {
        this.controls.hide();
    };

    /**
     * Handles event when a new surface has been added to the whiteboard.
     *
     * @param {AvayaClientServices.Services.Collaboration.Whiteboard} whiteboard
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardSurface} surface
     * @param {AvayaClientServices.Services.Collaboration.Participant} participant
     */
    WhiteboardSharingView.prototype.onSurfaceAdded = function (whiteboard, surface, participant) {
        var newSurface = $(`
            <div id="surface${surface.getSurfaceId()}" class="panel panel-default">
                <div class="panel-body whiteboardSurface">
                    <span>${surface.getSurfaceId()}</span>
                    <button class="btn btn-default setActive" data-surface="${surface.getSurfaceId()}">Activate</button>
                    <button class="btn btn-default removeSurface" data-surface="${surface.getSurfaceId()}">Remove</button>
                </div>
            </div>
        `);

        this.surfaces.append(newSurface);

        this.surfaces.on('click', '#surface' + surface.getSurfaceId() + ' .setActive', function (event) {
            this.handlers.setActiveSurface(Number(event.target.dataset.surface));
        }.bind(this));
        
        this.surfaces.on('click', '#surface' + surface.getSurfaceId() + ' .removeSurface', function (event) {
            this.handlers.removeSurface(Number(event.target.dataset.surface));
        }.bind(this));
    };

    /**
     * Handles event when a curtain surface become active.
     *
     * @param {AvayaClientServices.Services.Collaboration.Whiteboard} whiteboard
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardSurface} surface
     * @param {AvayaClientServices.Services.Collaboration.Participant} participant
     */
    WhiteboardSharingView.prototype.onSurfaceActivated = function (whiteboard, surface, participant) {
        this.surfaces.find('.setActive').each(function (index, button) {
            button.disabled = Number(button.dataset.surface) === surface.getSurfaceId();
        });
    };

    /**
     * Handles event when a surface has been removed from the whiteboard.
     *
     * @param {AvayaClientServices.Services.Collaboration.Whiteboard} whiteboard
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardSurface} surface
     * @param {AvayaClientServices.Services.Collaboration.Participant} participant
     */
    WhiteboardSharingView.prototype.onSurfaceRemoved = function (whiteboard, surface, participant) {
        this.surfaces.children('#surface' + surface.getSurfaceId()).remove();
    };

    /**
     * Creates buttons to manipulate drawing tools.
     *
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardLineWidths} lineWidths
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardTextSizes} textSizes
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardToolModes} toolModes
     * @param {AvayaClientServices.Services.Collaboration.WhiteboardToolTypes} toolTypes
     * @param {Array} colors
     */
    WhiteboardSharingView.prototype.initToolbox = function (lineWidths, textSizes, toolModes, toolTypes, colors) {
        this.createToolset(this.toolboxLineWidths, lineWidths, this.handlers.setLineWidth);
        this.createToolset(this.toolboxTextSizes, textSizes, this.handlers.setTextSize);
        this.createToolset(this.toolboxToolModes, toolModes, this.handlers.setToolMode);
        this.createToolset(this.toolboxToolTypes, toolTypes, this.handlers.setToolType);
        this.createToolset(this.toolboxColors, colors, this.handlers.setColor, function (button, mode) {
            button.css('background', mode);

            return button;
        });
    };

    /**
     * Helps to create a set of buttons and assign a handler to each one.
     *
     * @param {Object} parent
     * @param {Object} toolset
     * @param {Function} handler
     * @param {Function} [alterButton]
     */
    WhiteboardSharingView.prototype.createToolset = function (parent, toolset, handler, alterButton) {
        (Array.isArray(toolset) ? toolset : Object.keys(toolset)).map(function (tool) {
            parent.append(this.createButton(tool, alterButton));
        }.bind(this));

        parent.find('button').each(function (index, button) {
            $(button).on('click', function (event) {
                handler(event.target.dataset.mode);
            }.bind(this));
        }.bind(this));
    };

    /**
     * Helps to create a button.
     *
     * @param {string} mode
     * @param {Function} [alterButton]
     * @returns {Object}
     */
    WhiteboardSharingView.prototype.createButton = function (mode, alterButton) {
        var button = $('<button class="btn btn-default btn-sm" data-mode="' + mode + '">' + mode + '</button>');

        if (alterButton === undefined) {
            return button;
        }

        return alterButton(button, mode);
    };

    /**
     * Locks toolbox buttons when whiteboard is inactive.
     */
    WhiteboardSharingView.prototype.lockToolbox = function () {
        this.toolbox.find('button').each(function (index, button) {
            button.disabled = true;
        });
    };

    /**
     * Allows user to use drawing tools. This is intended to be called if the
     * DrawingCapability explicitly allows that.
     */
    WhiteboardSharingView.prototype.unlockToolbox = function () {
        this.toolbox.find('button').each(function (index, button) {
            button.disabled = false;
        });
    };

    window.WhiteboardSharingView = WhiteboardSharingView;

})();