﻿//////////////////////////////////////////////////////////////////////////////
// CCTEntities.cs
//
// Copyright 2015 Avaya Inc. All Rights Reserved.
//
// Usage of this source is bound to the terms described in
// Licenses/License.txt
//
// Avaya - Confidential & Proprietary. Use pursuant to your signed agreement or
// Avaya Policy
//////////////////////////////////////////////////////////////////////////////

using EncoderDecoder.Notifications;
using Nortel.CCT;
using System;
using System.Collections.Generic;

namespace CustomDesktop
{
    class CCTEntities
    {
        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);

        public delegate void pushSupervisorTabUpdate(Dictionary<string, SupervisedAgentNew> listOfMyAgents);

        public static event pushSupervisorTabUpdate SupervisorTabUpdateEvent;

        public static Dictionary<string, SupervisedAgentNew> listOfMyAgents = new Dictionary<string, SupervisedAgentNew>();
        private List<ChatNew> chatsWhenLoggingIn = new List<ChatNew>();
        private ChatIntrinsics intrinsics = new ChatIntrinsics();

        public void AddAgent(SupervisedAgentNew newAgent)
        {
            try
            {
                log.Debug(String.Format("Adding new agent to supervise with ID {0}", newAgent.agentId));

                if (listOfMyAgents.ContainsKey(newAgent.agentId))
                {
                    log.Debug(String.Format("ID {0} found in map so removing", newAgent.agentId));
                    listOfMyAgents.Remove(newAgent.agentId);
                }

                listOfMyAgents.Add(newAgent.agentId, newAgent);
            }
            catch (Exception ex)
            {
                log.Error("Exception in AddAgent", ex);
            }
        }

        public void UpdateAgentLoggedInStatus(string agentId, bool loggedIn)
        {
            try
            {
                log.Debug(String.Format("Updating logged in status of agent {0}", agentId));

                SupervisedAgentNew agent = listOfMyAgents[agentId];
                agent.loggedIn = loggedIn;

                UpdateSidebar();
            }
            catch (Exception ex)
            {
                log.Error("Exception in UpdateAgentLoggedInStatus", ex);
            }
        }

        public void UpdateAgentReadyStatus(string agentId, AgentReadyState readyStatus)
        {
            try
            {
                log.Debug(String.Format("Updating readystatus of agent {0}", agentId));

                SupervisedAgentNew agent = listOfMyAgents[agentId];
                agent.readyState = readyStatus;

                UpdateSidebar();
            }
            catch (Exception ex)
            {
                log.Error("Exception in UpdateAgentReadyStatus", ex);
            }
        }

        public void UpdateChatData(string agentId, ChatNew chatToAddOrUpdate)
        {
            try
            {
                log.Debug(String.Format("Updating chat {0} belonging to agent {1}", chatToAddOrUpdate.guid, agentId));

                if (listOfMyAgents.ContainsKey(agentId))
                {
                    SupervisedAgentNew agent = listOfMyAgents[agentId];
                    if (agent.agentChats.ContainsKey(chatToAddOrUpdate.guid))
                    {
                        //Copy intrinsics flag & data
                        chatToAddOrUpdate.intrinsicData = agent.agentChats[chatToAddOrUpdate.guid].intrinsicData;
                        chatToAddOrUpdate.IntrinsicsFlag = agent.agentChats[chatToAddOrUpdate.guid].IntrinsicsFlag;
                        //Copy parent and flag for consult
                        chatToAddOrUpdate.parentID = agent.agentChats[chatToAddOrUpdate.guid].parentID;
                        chatToAddOrUpdate.isConsult = agent.agentChats[chatToAddOrUpdate.guid].isConsult;

                        agent.agentChats.Remove(chatToAddOrUpdate.guid);
                        agent.agentChats.Add(chatToAddOrUpdate.guid, chatToAddOrUpdate);
                    }
                    else
                    {
                        //Check if consult & add parent and flag
                        if (chatToAddOrUpdate.chatContact.Intrinsics != null)
                        {
                            foreach (String key in chatToAddOrUpdate.chatContact.Intrinsics.Keys)
                            {
                                if (key.Equals("ConsultContactParentId"))
                                {
                                    chatToAddOrUpdate.isConsult = true;
                                    chatToAddOrUpdate.parentID = long.Parse(chatToAddOrUpdate.chatContact.Intrinsics[key]);
                                }
                            }
                        }

                        agent.agentChats.Add(chatToAddOrUpdate.guid, chatToAddOrUpdate);
                    }
                }
                else
                {
                    log.Error(String.Format("Could not find agent {0}", agentId));
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception in UpdateChatData", ex);
            }
        }

        public void RemoveChatData(string agentId, ChatNew chatToRemove)
        {
            try
            {
                log.Debug(String.Format("Removing chat {0} belonging to agent {1}", chatToRemove.guid, agentId));
                SupervisedAgentNew agent = listOfMyAgents[agentId];
                agent.agentChats.Remove(chatToRemove.guid);
            }
            catch (Exception ex)
            {
                log.Error("Exception in RemoveChatData", ex);
            }
        }

        public void UpdateChatIntrinsics(string agentId, long guid, AgentDataNotification.Conversation chatIntrinsics)
        {
            try
            {
                if (listOfMyAgents.ContainsKey(agentId))
                {
                    SupervisedAgentNew agent = listOfMyAgents[agentId];
                    if (agent.agentChats.ContainsKey(guid))
                    {
                        log.Debug(String.Format("Updating chat intrinsics for agent {0} on chat {1}", agentId, guid));
                        ChatNew chat = agent.agentChats[guid];
                        chat.intrinsicData = chatIntrinsics;

                        intrinsics.UpdateFlagBasedOnIntrinsics(chat);
                    }
                    else
                    {
                        log.Error(String.Format("Could not find chat {0} belonging to agent {1} to update chat intrinsics", guid, agentId));
                    }
                }
                else
                {
                    log.Error(String.Format("Could not find agent {0} to update chat intrinsics", agentId));
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception in UpdateChatIntrinsics", ex);
            }
        }

        public void UpdateSidebar()
        {
            SupervisorTabUpdateEvent(listOfMyAgents);
        }
    }

    public class SupervisedAgentNew
    {
        public string agentId { get; set; }

        public string agentFirstName { get; set; }

        public string agentLastName { get; set; }

        public string agentFullName { get; set; }

        public bool loggedIn { get; set; }

        public AgentReadyState readyState { get; set; }

        public UserType userType { get; set; }

        public Dictionary<long, ChatNew> agentChats = new Dictionary<long, ChatNew>();

        public string ToDisplayAgent { get 
            {
                return "Agent: " + agentFullName + " [" + agentId + 
                    "]\nLoggedIn?: " + loggedIn + 
                    "\nReady State: " + readyState; 
            }
        }
    }

    public class ChatNew
    {
        public long guid { get; set; }

        public TerminalConnectionState status { get; set; }

        public IContact chatContact { get; set; }

        public ITerminal chatTerminal { get; set; }

        public IConnection chatConnection { get; set; }

        public string IntrinsicsFlag { get; set; }

        public bool isConsult { get; set; }

        public long parentID { get; set; }

        //Intrinsic
        public AgentDataNotification.Conversation intrinsicData { get; set; }

        public string ToDisplayChat
        {
            get
            {
                if (isConsult)
                {
                    return "Consulting on: " + parentID + "\nStatus: " + status.ToString();
                }
                else
                {
                    return "GUID: " + guid + "\nStatus: " + status.ToString();
                }
            }
        }

        public string ToDisplayIntrinsics { get 
            { 
            return "conversationLength_inSeconds: " + intrinsicData.conversationLength_inSeconds.ToString() + 
                "\r\nlastMessageIn_inSeconds: " + intrinsicData.lastMessageIn_inSeconds.ToString() + 
                "\r\nlastMessageOut_inSeconds: " + intrinsicData.lastMessageOut_inSeconds + 
                "\r\nnumberOfAgentMessages: " + intrinsicData.numberOfAgentMessages.ToString() + 
                "\r\nnumberOfUnansweredCustomerMessages: " + intrinsicData.numberOfUnansweredCustomerMessages.ToString(); 
            } 
        }

        public string intrin_clis_text { get 
            { 
            return "conversationLength_inSeconds: " + intrinsicData.conversationLength_inSeconds + 
                "/" + ChatIntrinsics.conversationLength_inSeconds; 
            } 
        }

        public string intrin_clis_colour
        {
            get
            {
                if (intrinsicData.conversationLength_inSeconds >= ChatIntrinsics.conversationLength_inSeconds)
                {
                    return "Red";
                }
                else
                {
                    return "Black";
                }
            }
        }

        public string intrin_noam_text { get 
            { 
                return "numberOfAgentMessages: " + intrinsicData.numberOfAgentMessages +
                    "/" + ChatIntrinsics.numberOfAgentMessages; 
            }
        }

        public string intrin_noam_colour
        {
            get
            {
                if (intrinsicData.numberOfAgentMessages >= ChatIntrinsics.numberOfAgentMessages)
                {
                    return "Red";
                }
                else
                {
                    return "Black";
                }
            }
        }

        public string intrin_noucm_text { get 
            { 
                return "numberOfUnansweredCustomerMessages: " + intrinsicData.numberOfUnansweredCustomerMessages + "/"
                + ChatIntrinsics.numberOfUnansweredCustomerMessages; 
            } 
        }

        public string intrin_noucm_colour
        {
            get
            {
                if (intrinsicData.numberOfUnansweredCustomerMessages >= ChatIntrinsics.numberOfUnansweredCustomerMessages)
                {
                    return "Red";
                }
                else
                {
                    return "Black";
                }
            }
        }


        public string intrin_lmo_text { get 
            { 
                return "lastMessageOut_inSeconds: " + intrinsicData.lastMessageOut_inSeconds + 
                "/" + ChatIntrinsics.lastMessageOut_inSeconds; 
            } 
        }

        public string intrin_lmo_colour
        {
            get
            {
                if (intrinsicData.lastMessageOut_inSeconds >= ChatIntrinsics.lastMessageOut_inSeconds)
                {
                    return "Red";
                }
                else
                {
                    return "Black";
                }
            }
        }

        public string intrin_lmi_text { get 
            { 
                return "lastMessageIn_inSeconds: " + intrinsicData.lastMessageIn_inSeconds +
                    "/" + ChatIntrinsics.lastMessageIn_inSeconds;
            } 
        }

        public string intrin_lmi_colour
        {
            get
            {
                if (intrinsicData.lastMessageIn_inSeconds >= ChatIntrinsics.lastMessageIn_inSeconds)
                {
                    return "Red";
                }
                else
                {
                    return "Black";
                }
            }
        }
    }
}