﻿
//////////////////////////////////////////////////////////////////////////////
// SideBar.xaml.cs
//
// Copyright © 2008-2014 Avaya Inc. All rights reserved.
// See: www.avaya.com
//////////////////////////////////////////////////////////////////////////////
using EncoderDecoder;
using EncoderDecoder.Notifications;
using EncoderDecoder.Requests;
using Nortel.CCT;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;

namespace CustomDesktop
{
    /// <summary>
    /// Interaction logic for SideBar.xaml
    /// </summary>
    public partial class SideBar : UserControl
    {
        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);
        private Dictionary<string, Dictionary<String, String>> contactDetails = new Dictionary<string, Dictionary<String, String>>();
        private Dictionary<long, List<CustomerHistory>> contactHistory = new Dictionary<long, List<CustomerHistory>>();
        private Dictionary<long, List<ScreenPop>> contactScreenpop = new Dictionary<long, List<ScreenPop>>();
        private long currContactID = 0;
        private Dictionary<long, ConsultState> consultStates = new Dictionary<long, ConsultState>();
        private Dictionary<long, List<string>> notAllowedConsult = new Dictionary<long, List<string>>();

        public static List<SupervisedAgentNew> supervisedAgentsNew = new List<SupervisedAgentNew>();

        public delegate void blindTransfer(CDEntities.CDContactCentreSkillset skillset);
        public delegate void consultAgent(long agentID);
        public delegate void chatHistory(CustomerHistory history);
        public delegate void observeContact(Contact contact);
        public delegate void bargeContact(Contact contact);
        public delegate void addScreenPop(string url, string title, long guid);
        public delegate void forceChatExtraOpen();
        public event blindTransfer BlindTransferRequested;
        public event consultAgent AgentConsultRequested;
        public event chatHistory NewChatHistoryEvent;
        public event observeContact ObserveContactEvent;
        public event bargeContact BargeContactEvent;
        public event addScreenPop AddNewScreenPop;
        public event forceChatExtraOpen ForceChatExtraOpen;

        private enum ConsultState
        {
            NONE,
            LOADING,
            LOADED
        }

        /// <summary>
        /// The Default constructor for the Sidebar control
        /// </summary>
        public SideBar()
        {
            InitializeComponent();
            cbSkillsets.DisplayMemberPath = "name";

            //WS handlers
            WsockEvent.GetDestinationsNotification += WsGetDestinationsHandler;
            WsockEvent.ReadCustomerHistoryNotification += WsReadCustomerHistoryHandler;
            WsockEvent.AgentDataNotification += WsAgentDataHandler;

            CCTEntities.SupervisorTabUpdateEvent += SupervisorTabUpdate;
        }

        #region GUI Manipulation       
        /// <summary>
        /// Resets this control to it's original state.
        /// NB: This method turns the sidebar invisible
        /// </summary>
        public void Reset(bool hide)
        {
            try
            {
                lbAgents.Items.Clear();  
                this.gTransfer.Visibility = Visibility.Hidden;
                gInitiateTransfer.Visibility = Visibility.Visible;
                spInfo.Visibility = Visibility.Visible;
                if (hide)
                {
                    custHist.Items.Clear();
                    if(Globals.usertype.Equals(UserType.AgentSupervisor)){
                        hasKickedOfAgentDataRequestTimer = false;
                        this.contactTab.Visibility = Visibility.Collapsed;
                        this.historyTab.Visibility = Visibility.Collapsed;
                        this.SidebarTabs.SelectedIndex = 1;
                        return;
                    }
                    screenPopTab.Visibility = Visibility.Collapsed;
                    this.Visibility = Visibility.Hidden;
                }
            }
            catch (Exception ex)
            {
                log.Error("An exception occured reseting the sidebar", ex);
            }
        }

        private void WsGetDestinationsHandler(object s, NotificationEvent e)
        {
            var messageEvent = e.Notification as GetDestinationsNotification;
            if (messageEvent != null)
            {
                try
                {
                    Globals.AllowBlindTransfer = messageEvent.body.allowTransferToSkillset;

                    List<GetDestinationsNotification.Queue> listOfQueues = messageEvent.body.queues;

                    CDEntities.CDContactCentreSkillset[] queueData = GetDataToPopulateSidebarWithAgentsAndQueues(listOfQueues);


                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        PopulateSidebar(queueData);
                    }));
                    

                }
                catch (Exception ex)
                {
                    log.Error("Could not decode new get destinations notification " + messageEvent.ToString(), ex);
                }

            }

        }

        private void WsReadCustomerHistoryHandler(object s, NotificationEvent e)
        {
            var messageEvent = e.Notification as ReadCustomerHistoryNotification;
            if (messageEvent != null)
            {
                try
                {
                    log.Info("Received customer historical transcripts notification");
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        //Clear customer history listbox
                        custHist.Items.Clear();

                        List<ReadCustomerHistoryNotification.History> custHistory = messageEvent.body.history;

                        //Check if blank notification was returned and display error
                        if (custHistory.Count < 1)
                        {
                            MessageBox.Show("No transcripts available");
                            return;
                        }

                        List<CustomerHistory> historyList = new List<CustomerHistory>();
                        //Add each historical transcript to GUI so user can select
                        foreach (ReadCustomerHistoryNotification.History history in custHistory)
                        {
                            CustomerHistory customerHist = new CustomerHistory();
                            customerHist.guid = currContactID;
                            customerHist.timestring = JDecoder.ConvertFromUnixTimeWithDate(history.startTime);
                            customerHist.transcript = history.transcript;
                            customerHist.source = history.source;
                            customerHist.contactType = history.contactType;

                            custHist.Items.Add(customerHist);
                            historyList.Add(customerHist);
                        }

                        if (contactHistory.ContainsKey(messageEvent.body.guid))
                        {
                            contactHistory.Remove(messageEvent.body.guid);
                        }
                        contactHistory.Add(messageEvent.body.guid, historyList);
                    }));

                }
                catch (Exception ex)
                {
                    log.Error("Exception new read customer history notification " + messageEvent.ToString(), ex);
                }

            }

        }

        private void WsAgentDataHandler(object s, NotificationEvent e)
        {
            var messageEvent = e.Notification as AgentDataNotification;
            if (messageEvent != null)
            {
                try
                {
                    log.Debug("Received agent data notification");
                        CCTEntities entities = new CCTEntities();

                        //Update threshold values
                        log.Debug("Setting threshold values");
                        ChatIntrinsics.UpdateThresholdValues(messageEvent.body.thresholds);

                        foreach (AgentDataNotification.Agent agent in messageEvent.body.agents)
                        {
                            log.Debug(String.Format("Processing intrinsics for agent {0}", agent.agentId));
                            foreach (AgentDataNotification.Conversation convo in agent.conversations)
                            {
                                log.Debug(String.Format("Processing intrinsics for agent {0} on chat {1}", agent.agentId, convo.guid));
                                entities.UpdateChatIntrinsics(agent.agentId, convo.guid, convo);
                            }
                        }

                        this.Dispatcher.Invoke((Action)(() =>
                        {
                            entities.UpdateSidebar();
                        }));

                }
                catch (Exception ex)
                {
                    log.Error("Exception in agent data notification ", ex);
                }

            }

        }

        

        private void RepopulateHistory(long guid)
        {
            try
            {
                custHist.Items.Clear();

                if (contactHistory.ContainsKey(guid))
                {
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        List<CustomerHistory> repopHistory = contactHistory[guid];

                        foreach (CustomerHistory history in repopHistory)
                        {
                            custHist.Items.Add(history);
                        }
                    }));
                }
            }
            catch (Exception ex)
            {
                log.Info("Exception repopulating history" + ex);
            }
        }

        private void RepopulateScreenpops(long guid)
        {
            try
            {
                screenPops.Items.Clear();

                if (contactScreenpop.ContainsKey(guid))
                {
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        List<ScreenPop> repopScreenpops = contactScreenpop[guid];

                        foreach (ScreenPop screenPop in repopScreenpops)
                        {
                            screenPops.Items.Add(screenPop);
                        }
                    }));
                }
            }
            catch (Exception ex)
            {
                log.Info("Exception repopulating screenpops " + ex);
            }
        }

        private CDEntities.CDContactCentreSkillset[] GetDataToPopulateSidebarWithAgentsAndQueues(List<GetDestinationsNotification.Queue> listOfQueues)
        {

            //
            // This is just temp for time being so we can figure out routepoints etc. i.e. whats needed
            //
            List<CDEntities.CDContactCentreSkillset> allQueues = new List<CDEntities.CDContactCentreSkillset>();

            foreach (GetDestinationsNotification.Queue queue in listOfQueues)
            {
                //Queue Name + details
                CDEntities.CDContactCentreSkillset skillset = new CDEntities.CDContactCentreSkillset();
                skillset.ccmsID = queue.queueCCMSId;
                skillset.ID = queue.queueId;
                skillset.name = queue.queueName;
                skillset.routepoint = queue.routepoint;

                //Get all the agents on the skillset
                List<GetDestinationsNotification.Agent> agentsOnSkillset = queue.agents;
                List<CDEntities.CDContactCenterAgent> listOfAgents = new List<CDEntities.CDContactCenterAgent>();
                foreach (GetDestinationsNotification.Agent agentDetail in agentsOnSkillset)
                {
                    CDEntities.CDContactCenterAgent agent = new CDEntities.CDContactCenterAgent();
                    agent.ID = Int32.Parse(agentDetail.agentId);
                    agent.logonID = agentDetail.agentId.ToString();
                    agent.name = agentDetail.displayName;
                    listOfAgents.Add(agent);
                }

                //Convert the list to array
                skillset.agents = listOfAgents.ToArray();
                allQueues.Add(skillset);
                
            }

            return allQueues.ToArray();
        }

        public void AllowRequestToConsultAnAgent(bool allowConsult, long guid, string agentId){

            // Allow consult to an agent
            if (allowConsult)
            {
                // Check if room is in map
                if (notAllowedConsult.ContainsKey(guid))
                {
                    //Get the list of not allowed
                    List<string> notAllowedList = notAllowedConsult[guid];


                    if (notAllowedList.Contains(agentId))
                    {
                        //Remove agentId if it exists
                        notAllowedList.Remove(agentId);

                        //Update notAllowedConsult
                        notAllowedConsult.Remove(guid);
                        notAllowedConsult.Add(guid, notAllowedList);
                    }

                }
                // Add room to map
                else
                {
                    notAllowedConsult.Add(guid, new List<string>());
                }
            }
            else
            {
                // Check if room not in map and create if so
                if (!notAllowedConsult.ContainsKey(guid))
                {
                    notAllowedConsult.Add(guid, new List<string>());
                }


                //Get the list of not allowed
                List<string> notAllowedList = notAllowedConsult[guid];

                //Remove if it is already there
                if (notAllowedList.Contains(agentId))
                {
                    //Remove agentId if it exists
                    notAllowedList.Remove(agentId);
                }

                //Add them to list
                notAllowedList.Add(agentId);

                //Update notAllowedConsult
                notAllowedConsult.Remove(guid);
                notAllowedConsult.Add(guid, notAllowedList);
            }
        }

        /// <summary>
        /// Populate the contact details textbox with the alerting contact
        /// </summary>
        public void SetContactDetails(long contactID)
        {
            try
            {
                RepopulateHistory(contactID);
                RepopulateScreenpops(contactID);

                currContactID = contactID;
                Dictionary<String, String> intrinsics = contactDetails[contactID.ToString()];
                //enable the transfer button in case it was disabled for an alerting contact
                btnInitiateTransfer.IsEnabled = true;
                if (intrinsics != null)
                {
                    dgInfo.Items.Clear();
                    foreach (String key in intrinsics.Keys)
                    {
                        dgInfo.Items.Add(new KeyValuePair<String, String>(key, intrinsics[key]));
                    }
                }
                ConsultState state = consultStates[currContactID];
                switch (state)
                {
                    case ConsultState.NONE:
                        Reset(false);
                        break;
                    case ConsultState.LOADING:
                        gInitiateTransfer.Visibility = Visibility.Hidden;
                        spInfo.Visibility = Visibility.Visible;
                        gLoadingConsult.Visibility = Visibility.Visible;
                        gTransfer.Visibility = Visibility.Hidden;
                        break;
                    case ConsultState.LOADED:
                        Reset(false);
                        break;
                    default:
                        log.Error("Invalid case in SetContactDetails");
                        break;
                }
            }
            catch (Exception ex)
            {
                log.Error("An exception occured setting contact details for " + contactID, ex);
            }
        }

        /// <summary>
        /// Populate the contact details textbox with the given contact, called when a contact is alerting
        /// </summary>
        public void AddContactDetails(Contact contact, long contactIDToUse)
        {
            try
            {
                contactDetails.Add(contactIDToUse.ToString(), contact.Intrinsics);
                consultStates.Add(contactIDToUse, ConsultState.NONE);
                SetContactDetails(contactIDToUse);
                //Disable the transfer button, because the contact is alerting.
                btnInitiateTransfer.IsEnabled = false;
            }
            catch (Exception ex)
            {
                log.Error("An exception occured setting contact details for " + contactIDToUse, ex);
            }
        }

        internal void RemoveContactDetails(long contactID)
        {
            if (contactDetails.ContainsKey(contactID.ToString()))
            {
                contactDetails.Remove(contactID.ToString());
                consultStates.Remove(contactID);
            }
            if (contactID == currContactID)
            {
                Reset(false);
            }
        }

        internal void RemoveContactDetails(Contact contact)
        {
            if (contactDetails.ContainsKey(contact.ID.ToString()))
            {
                contactDetails.Remove(contact.ID.ToString());
                consultStates.Remove(contact.ID);
            }
            else if (contact.IsConsultContact() && contactDetails.ContainsKey(contact.ParentID.ToString()))
            {
                contactDetails.Remove(contact.ParentID.ToString());
                consultStates.Remove(contact.ParentID);
            }
            if (contact.ID == currContactID || contact.ParentID == currContactID)
            {
                Reset(false);
            }
        }

        /// <summary>
        /// Populate the contact details textbox with the given contact, called when a contact is alerting
        /// </summary>
        public void AddContactDetails(Contact contact)
        {
            AddContactDetails(contact, contact.ID);
        }
        #endregion

        #region Event Handlers
        /// <summary>
        /// Event handler for the cancel transfer button, reverts the sidebar's appearance
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnCancelTransferClick(object sender, RoutedEventArgs e)
        {
            lbAgents.Items.Clear();
            this.gTransfer.Visibility = Visibility.Hidden;
            gInitiateTransfer.Visibility = Visibility.Visible;
            spInfo.Visibility = Visibility.Visible;
            consultStates.Remove(currContactID);
            consultStates.Add(currContactID, ConsultState.NONE);
        }

        /// <summary>
        /// Event handler for the initiate transfer button
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnInitiateTransferClick(object sender, RoutedEventArgs e)
        {
            try
            {
                gInitiateTransfer.Visibility = Visibility.Hidden;
                gLoadingConsult.Visibility = Visibility.Visible;
                consultStates.Remove(currContactID);
                consultStates.Add(currContactID, ConsultState.LOADING); 

                SendGetDestinations();
            }
            catch (Exception ex)
            {
                log.Error("An exception occured initializing blind transfer", ex);
            }
        }

        private void ViewHistoryClick(object sender, RoutedEventArgs e)
        {
            log.Info("View history button clicked");

            try
            {
                //Check user has selected a history item
                if (custHist.SelectedIndex < 0)
                {
                    MessageBox.Show("No historic transcript selected");
                    return;
                }

                CustomerHistory selectedHistory = custHist.SelectedItem as CustomerHistory;

                //Create a new window with the chat transcript
                if (NewChatHistoryEvent != null)
                {
                    NewChatHistoryEvent(selectedHistory);
                }
            }
            catch (Exception ex)
            {
                log.Error("An exception occured in customer historic chat history", ex);
            }

        }

        private void GetHistoryClick(object sender, RoutedEventArgs e)
        {
            log.Info("Get history button clicked");

            int amountOfResults;
            bool validNumber = int.TryParse(custHistAmount.Text, out amountOfResults);

            if (!validNumber || amountOfResults<=0)
            {
                MessageBox.Show("You must enter a valid numeric value greater than zero");
            }
            else
            {
                SendReadCustomerHistory(amountOfResults);
            }
        }

        private void ViewScreenPopClick(object sender, RoutedEventArgs e)
        {
            log.Info("View history button clicked");

            try
            {
                //Check user has selected a screenpop
                if (screenPops.SelectedIndex < 0)
                {
                    MessageBox.Show("No screenpop selected");
                    return;
                }

                //Check the location is not empty
                if (String.IsNullOrEmpty(screenpoplocation.Text))
                {
                    MessageBox.Show("Location cannot be empty");
                    return;
                }

                ScreenPop theScreenPop = screenPops.SelectedItem as ScreenPop;

                string location = screenpoplocation.Text;

                //Check if its a URL or a command
                if (location.StartsWith("www") || location.StartsWith("http"))
                {
                    if (!location.StartsWith("http"))
                    {
                        location = "http://" + location;
                    }
                    AddNewScreenPop(location, theScreenPop.name, currContactID);
                }
                else
                {
                    if (theScreenPop.name.ToUpper().Equals("NOTEPAD"))
                    {
                        Process myProcess = new Process();
                        myProcess.StartInfo.FileName = "notepad";
                        myProcess.StartInfo.Arguments = location.Replace("notepad ","");
                        myProcess.Start();
                    }
                    else
                    {
                        Process myProcess = new Process();
                        myProcess.StartInfo.FileName = screenpoplocation.Text;
                        myProcess.Start();
                    }
                }
            }
            catch (Exception ex)
            {
                log.Error("An exception occured in clicking screen pop launch", ex);
            }

        }

        private void SendGetDestinations()
        {
            //WS things
            GetDestinationsRequest destinationTypes = new GetDestinationsRequest();
            destinationTypes.apiVersion = GetDestinationsRequest.JSapiVersion.one_point_zero;
            destinationTypes.body.guid = currContactID;
            Wsock.Send(destinationTypes);
        }

        private void SendReadCustomerHistory(int amount)
        {
            ReadCustomerHistoryRequest history = new ReadCustomerHistoryRequest();
            history.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
            history.body.amount = amount;
            history.body.guid = currContactID;
            Wsock.Send(history);
        }

        private void ScreenpopsSelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            this.Dispatcher.Invoke((Action)(() =>
            {
                if (screenPops.SelectedIndex >= 0)
                {
                    screenpoplocation.Text = "";

                    ScreenPop thisScreenPop = screenPops.SelectedItem as ScreenPop;
                    if (thisScreenPop != null)
                    {
                        string url = thisScreenPop.location;

                        if (url.Contains("%VALUE%"))
                        {
                            url = url.Replace("%VALUE%", currContactID.ToString());
                        }
                        screenpoplocation.Text = url;
                    }
                }
            }));
        }

        private void CbAgentsSelectionChanged(object sender, SelectionChangedEventArgs e)
        {

            this.Dispatcher.Invoke((Action)(() =>
            {
                if (cbAgents.SelectedIndex >= 1)
                {
                    chatData.Items.Clear();

                    SupervisedAgentNew thisAgent = cbAgents.SelectedItem as SupervisedAgentNew;
                    if (thisAgent != null)
                    {
                        agentDataGrid.Visibility = System.Windows.Visibility.Visible;
                        agentDataGrid.Items.Clear();
                        agentDataGrid.Items.Add(thisAgent);

                        foreach (ChatNew agentsChat in thisAgent.agentChats.Values)
                        {
                            chatData.Items.Add(agentsChat);
                        }

                    }
                }
                else
                {
                    chatData.Items.Clear();

                    agentDataGrid.Visibility = System.Windows.Visibility.Collapsed;

                    //Add all agents chats
                    foreach (SupervisedAgentNew agents in CCTEntities.listOfMyAgents.Values)
                    {
                        foreach (ChatNew agentsChat in agents.agentChats.Values)
                        {
                            chatData.Items.Add(agentsChat);
                        }
                    }
                }

            }));
        }

        private void SupervisorTabUpdateAgents(Dictionary<string, SupervisedAgentNew> listOfAgents)
        {
            this.Dispatcher.Invoke((Action)(() =>
            {
                
                try
                {
                    //Get current selected agent so we can go back after update
                    int selectedAgent = cbAgents.SelectedIndex;

                    //Clear agent combobox and agent details
                    cbAgents.Items.Clear();

                    foreach (SupervisedAgentNew newd in listOfAgents.Values)
                    {
                        cbAgents.Items.Add(newd);
                    }

                    //Add in All Agents header
                    SupervisedAgentNew allAgent = new SupervisedAgentNew();
                    allAgent.agentFullName = "--All Agents--";
                    cbAgents.Items.Insert(0, allAgent);

                    
                    //Go to 'All Agents' on first entry
                    if (selectedAgent < 0)
                    {
                        cbAgents.SelectedIndex = 0;
                    }
                    //Or else go back to selected agent
                    else
                    {
                        cbAgents.SelectedIndex = selectedAgent;
                    }
                    

                }
                catch (Exception ex)
                {
                    log.Error("Failed to update supervisor tab", ex);
                }
            }));
        }

        private bool hasKickedOfAgentDataRequestTimer = false;

        private void SupervisorTabUpdate(Dictionary<string, SupervisedAgentNew> listOfAgents)
        {   
            //Update agent stats
            SupervisorTabUpdateAgents(listOfAgents);

            this.Dispatcher.Invoke((Action)(() =>
            {
                try
                {
                    if (!hasKickedOfAgentDataRequestTimer)
                    {
                        StartAgentDataRequestTimedWork();
                        hasKickedOfAgentDataRequestTimer = true;
                    }
                }
                catch (Exception ex)
                {
                    log.Error("Failed to update supervisor tab", ex);
                }
            }));
            
            
        }

        private long CheckConsultStateIfLoading()
        {
            foreach (KeyValuePair<long, ConsultState> pair in consultStates)
            {
                if (pair.Value == ConsultState.LOADING)
                {
                    return pair.Key;

                }
            }
            return -1;
        }
        private void PopulateSidebar(CDEntities.CDContactCentreSkillset[] queueData)
        {
            try
            {
                long toRemove = -1;

                toRemove = CheckConsultStateIfLoading();
                 
                consultStates.Remove(toRemove);
                consultStates.Add(toRemove, ConsultState.LOADED);

                if (AllowBlindTransfer())
                {
                    cbSkillsets.IsEnabled = true;
                    lbAgents.IsEnabled = true;
                    btnRefresh.IsEnabled = true;
                    btnConsult.IsEnabled = true;
                    btnCancel.IsEnabled = true;
                    lBlindTransfer.IsEnabled = true;
                    btnBlindtransfer.IsEnabled = true;
                    EnableBlindTransfer();
                    log.Info("Enabled blind transfers");
                }
                else
                {
                    cbSkillsets.IsEnabled = true;
                    lbAgents.IsEnabled = true;
                    btnRefresh.IsEnabled = true;
                    btnConsult.IsEnabled = true;
                    btnCancel.IsEnabled = true;

                    lBlindTransfer.IsEnabled = false;
                    btnBlindtransfer.IsEnabled = false;
                    log.Info("Blind transfer not enabled");
                }
                
                //add them to the drop down list
                if (queueData != null)
                {
                    cbSkillsets.Items.Clear();

                    //Add all agents type
                    CDEntities.CDContactCentreSkillset allAgentsGroup = new CDEntities.CDContactCentreSkillset();
                    List<CDEntities.CDContactCenterAgent> allAgents = new List<CDEntities.CDContactCenterAgent>();
                    allAgentsGroup.name = "--All Agents--";

                    foreach (CDEntities.CDContactCentreSkillset skillset in queueData)
                    {
                        //Add all these skillset specific agents to the 'All Agents' group
                        allAgents.AddRange(skillset.agents);

                        if (skillset.agents.Length > 0)
                        {
                            cbSkillsets.Items.Add(skillset);
                        }

                    }

                    
                    if (allAgents.Count == 0)
                    {
                        gInitiateTransfer.Visibility = Visibility.Visible;
                        gLoadingConsult.Visibility = Visibility.Hidden;
                        consultStates.Remove(currContactID);
                        MessageBox.Show("No agents are currently available");
                        return;
                    }
                    else
                    {
                        allAgentsGroup.agents = allAgents.ToArray();
                        cbSkillsets.Items.Insert(0, allAgentsGroup);
                        cbSkillsets.SelectedIndex = 0;
                    }                
                    
                }
                gLoadingConsult.Visibility = Visibility.Hidden;
                spInfo.Visibility = Visibility.Hidden;
                gTransfer.Visibility = Visibility.Visible;
            }
            catch (Exception ex)
            {
                log.Error("An exception occured in PopulateSidebar", ex);
            }
        }

        private void BtnObserveClick(object sender, RoutedEventArgs e)
        {
            log.Info("Observe button clicked");

            try
            {
                //Check user has selected a history item
                if (chatData.SelectedIndex < 0)
                {
                    MessageBox.Show("No agent chat selected");
                    return;
                }
               
                ChatNew selectedChat = chatData.SelectedItem as ChatNew;         

                if (selectedChat.status == TerminalConnectionState.Ringing)
                {
                    MessageBox.Show("Cannot observe chat in a ringing state");
                    return;
                }

                ChatObserve(selectedChat.chatContact);
            }
            catch (Exception ex)
            {
                log.Error("An exception occured in btnObserve_Click", ex);
            }
        }

        private System.Windows.Forms.Timer agentDataRequestTimer = new System.Windows.Forms.Timer();

        private void StartAgentDataRequestTimedWork()
        {
            log.Info("Starting subscription to agent data");
            agentDataRequestTimer.Interval = 5000;
            agentDataRequestTimer.Tick += new EventHandler(AgentDataRequestTimerTick);
            agentDataRequestTimer.Start();
        }

        public void StopAgentDataRequestTimedWork()
        {
            log.Info("Stopping subscription to agent data");
            agentDataRequestTimer.Stop();
        }

        private void AgentDataRequestTimerTick(object sender, EventArgs e)
        {
            try
            {
                lock (agentDataRequestTimer)
                {
                    if (this.agentDataRequestTimer.Enabled)
                    {
                        this.agentDataRequestTimer.Stop();
                        this.SendAgentDataRequest();
                        this.agentDataRequestTimer.Start();
                    }
                }
            }
            catch(Exception ex)
            {
                log.Error("An exception occured in agentDataRequestTimer_Tick", ex);
            }

        }

        private void SendAgentDataRequest()
        {
            try
            {
                if(Wsock.isOpen && Wsock.loggedIn){
                    AgentDataRequest dataRequest = new AgentDataRequest();
                    dataRequest.apiVersion = BaseRequest.JSapiVersion.one_point_zero;

                    Wsock.Send(dataRequest);
                }

            }
            catch (Exception ex)
            {
                log.Error("An exception occured in sendAgentDataRequest", ex);
            }
        }

        public static ITerminal GetCorrectWebCommsTerminal()
        {
            ITerminal correctTerminal = null;
            ITerminal[] iterms = CCTConnector.GetCCTSession().Terminals;
            foreach (ITerminal it in iterms)
            {
                log.Info("Found a terminal " + it.ToString() + " with types - " + it.ContactTypes.ToString());

                string[] termTypes = it.ContactTypes;
                int pos = Array.IndexOf(termTypes, "Web_Communications");
                if (pos > -1)
                {
                    log.Info("Setting this to the correct terminal");
                    correctTerminal = it;
                }
            }

            return correctTerminal;
        }

        public void ChatObserve(IContact contact)
        {
            try
            {
                log.Info("Observe: Calling CCT method");

                ITerminal correctTerminal = GetCorrectWebCommsTerminal();

                if (correctTerminal == null)
                {
                    log.Error("Could not find Web_Communications CCT terminal");
                    MessageBox.Show("Could not find Web_Communications CCT terminal");
                    return;
                }

                IConnection newObserveConnection = contact.Observe(correctTerminal, CCTConnector.GetCCTAddress());
                CCTConnector.observingChats.Add(contact.ID, newObserveConnection);

                log.Info("Observe: Sending JSON request");
                AgentJoinRoomRequest joinroom = new AgentJoinRoomRequest();
                joinroom.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
                joinroom.body.joinType = BaseRequest.JSjoinType.observe;
                joinroom.body.guid = Convert.ToInt64(contact.ID);

                Wsock.Send(joinroom);

                Contact contactToObserve = new Contact(contact);

                log.Info("Observe: Adding contact details to sidebar");
                AddContactDetails(contactToObserve);

                log.Info("Observe: Firing observe event");
                if (ObserveContactEvent != null)
                {
                    ObserveContactEvent(contactToObserve);
                }
            }
            catch (MissingContactTypeException ex2)
            {
                log.Info("MissingContactTypeException - " + ex2);
            }
            catch (Exception ex)
            {
                log.Info("Failed to observe - " + ex);
            }
        }

        public void ChatBarge(IContact contact,string guid)
        {
            try
            {
                if (CCTConnector.observingChats.ContainsKey(guid))
                {

                    ITerminal correctTerminal = GetCorrectWebCommsTerminal();

                    if (correctTerminal == null)
                    {
                        log.Error("Could not find Web_Communications CCT terminal");
                        MessageBox.Show("Could not find Web_Communications CCT terminal");
                        return;
                    }

                    contact.BargeIn(correctTerminal, CCTConnector.observingChats[guid].Address);

                    Contact contactToBarge = new Contact(contact);
                    if (BargeContactEvent != null)
                    {
                        BargeContactEvent(contactToBarge);
                    }
                }
                
            }
            catch (Exception ex)
            {
                log.Info("failed to barge - " + ex);
            }
        }
        

        private void EnableBlindTransfer()
        {
            btnBlindtransfer.Visibility = System.Windows.Visibility.Visible;
            lBlindTransfer.Visibility = System.Windows.Visibility.Visible;
        }

        private bool AllowBlindTransfer()
        {
            return Globals.AllowBlindTransfer;
        }

        /// <summary>
        /// Event handler for the blind transfer button
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnBlindtransferClick(object sender, RoutedEventArgs e)
        {
            try
            {
                if (cbSkillsets.SelectedIndex == 0)
                {
                    MessageBox.Show(StringResources.SidebarNoSkillsetSelected);
                    return;
                }
                RemoveContactDetails(currContactID);

                CDEntities.CDContactCentreSkillset selectedSkillset = cbSkillsets.SelectedItem as CDEntities.CDContactCentreSkillset;

                //Blind transfer the current contact to queue
                if (BlindTransferRequested != null)
                {
                    BlindTransferRequested(selectedSkillset);
                }

                //Send Agent Operation
                SendAgentOperation(this.currContactID, BaseRequest.JSdestinationType.queue, selectedSkillset.ID.ToString());
            }
            catch (Exception ex)
            {
                log.Error("An exception occured completing blind transfer", ex);
            }
        }

        /// <summary>
        /// Event handler for the Consult button
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnConsultClick(object sender, RoutedEventArgs e)
        {
            try
            {
                CDEntities.CDContactCenterAgent agentToConsult = lbAgents.SelectedItem as CDEntities.CDContactCenterAgent;
                if (agentToConsult == null)
                {
                    MessageBox.Show(StringResources.SidebarNoAgentSelected);
                    return;
                }
                if (notAllowedConsult.ContainsKey(this.currContactID))
                {
                    log.Info("Found map in not allowed to consult map");

                    List<string> dontAllowConsult = notAllowedConsult[this.currContactID];
                    if(dontAllowConsult.Contains(agentToConsult.logonID)){
                        log.Info("Cannot consult with this agent");
                        MessageBox.Show("Cannot consult with this agent as they are already in the room");
                        return;
                    }
                    else
                    {
                        log.Info("Can consult with this agent");
                    }

                }

                //transfer the current contact

                if (AgentConsultRequested != null)
                {
                    AgentConsultRequested(Int32.Parse(agentToConsult.logonID));
                }
                btnInitiateTransfer.IsEnabled = false;
                Reset(false);
                SetContactDetails(currContactID);

                //Consult with Agent
                SendAgentOperation(this.currContactID, BaseRequest.JSdestinationType.consult, agentToConsult.logonID);
            }
            catch (Exception ex)
            {
                log.Error("An exception occured consulting", ex);
            }
        }

        private void SendAgentOperation(long currentGuid, AgentOperationRequest.JSdestinationType destType, string destId)
        {
            try
            {
                AgentOperationRequest operationRequest = new AgentOperationRequest();
                operationRequest.apiVersion = AgentOperationRequest.JSapiVersion.one_point_zero;
                operationRequest.body.guid = currentGuid;
                operationRequest.body.destinationType = destType;
                operationRequest.body.destinationId = destId;

                Wsock.Send(operationRequest);
            }
            catch (Exception ex)
            {
                log.Error("An exception occured during the agent operation request", ex);
            }
        }

        /// <summary>
        /// Event handler for the Consult button
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnRefreshClick(object sender, RoutedEventArgs e)
        {
            try
            {
                cbSkillsets.IsEnabled = false;
                lbAgents.IsEnabled = false;
                btnRefresh.IsEnabled = false;
                btnConsult.IsEnabled = false;
                btnCancel.IsEnabled = false;
                btnBlindtransfer.IsEnabled = false;
                lbAgents.Items.Clear();

                SendGetDestinations();
            }
            catch (Exception ex)
            {
                log.Error("An exception occured refreshing", ex);
            }
        }

        private void CbSkillsetsSelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            if (cbSkillsets.SelectedIndex >= 0)
            {
                CDEntities.CDContactCentreSkillset skillset = cbSkillsets.SelectedItem as CDEntities.CDContactCentreSkillset;
                if (skillset != null)
                {
                    lbAgents.Items.Clear();
                    foreach (CDEntities.CDContactCenterAgent agent in skillset.agents)
                    {
                        //ensure that the current agent is not shown as a transfer target.
                        if (long.Parse(agent.logonID) != Globals.agentID)
                        {
                            if (lbAgents.Items.Contains(agent))
                            {
                                log.Debug("List already contains agent");
                            }
                            else
                            {
                                lbAgents.Items.Add(agent);
                            }
                            
                        }
                    }
                }
            }
        }

        public void EnableConsult(Boolean enable)
        {
            btnInitiateTransfer.IsEnabled = enable;
            if (!enable)
            {
                Reset(false);
            }
        }
        #endregion


        public void AddScreenPops(ScreenPopsNotification notification)
        {
            try
            {
                List<ScreenPopsNotification.ScreenPops> listOfScreenPops = notification.body.screenPops;
                log.Info("Received customer historical transcripts notification");
                this.Dispatcher.Invoke((Action)(() =>
                {
                    //Clear screenpops listbox
                    screenPops.Items.Clear();

                    //Check if no screenpops
                    if (listOfScreenPops.Count > 0)
                    {
                        //Add the screen pops tab
                        screenPopTab.Visibility = System.Windows.Visibility.Visible;
                    }
                    else
                    {
                        screenPopTab.Visibility = System.Windows.Visibility.Collapsed;
                        return;
                    }

                    bool extendChatExtraBasedOnForceOpen = false;

                    List<ScreenPop> allScreenpops = new List<ScreenPop>();
                    
                    //Add each screenpop to GUI so user can select
                    foreach (ScreenPopsNotification.ScreenPops sp in listOfScreenPops)
                    {
                        ScreenPop aScreenPop = new ScreenPop();
                        aScreenPop.name = sp.name;
                        aScreenPop.location = sp.path;
                        aScreenPop.forceOpen = sp.forceLaunch;

                        if (aScreenPop.forceOpen)
                        {
                            extendChatExtraBasedOnForceOpen = true;
                            string location = aScreenPop.location;
                            if (location.StartsWith("www") || location.StartsWith("http"))
                            {
                                location = location.Replace("%VALUE%", currContactID.ToString());

                                if (!location.StartsWith("http"))
                                {
                                    location = "http://" + location;
                                }
                                AddNewScreenPop(location, aScreenPop.name, 123);
                            }
                            else
                            {
                                Process myProcess = new Process();
                                myProcess.StartInfo.FileName = location;
                                myProcess.Start();
                            }
                        }
                        allScreenpops.Add(aScreenPop);
                        screenPops.Items.Add(aScreenPop);
                    }

                    contactScreenpop.Add(notification.body.guid, allScreenpops);

                    if (extendChatExtraBasedOnForceOpen)
                    {
                        ForceChatExtraOpen();
                    }

                }));

            }
            catch (Exception ex)
            {
                log.Error("Exception populating screenpops", ex);
            }
        }
    }

    // Need to move this
    public class CustomerHistory {

        public long guid { get; set; }
        public string timestring { get; set; }
        public string transcript { get; set; }
        public string source { get; set; }
        public string contactType { get; set; }

        public string ToDisplay { get { return "Date: " + timestring + "\nSource: " + source + "\nType: " + contactType; } }
        
    }

    // Need to move this
    public class ScreenPop
    {

        public string name { get; set; }
        public string location { get; set; }
        public bool forceOpen { get; set; }

        public string ToDisplay { get { return name; } }

    }

}
