﻿//////////////////////////////////////////////////////////////////////////////
// UcChatWindow.xaml.cs
//
// Copyright © 2008-2014 Avaya Inc. All rights reserved.
// See: www.avaya.com
//////////////////////////////////////////////////////////////////////////////

using EncoderDecoder;
using EncoderDecoder.Notifications;
using EncoderDecoder.Requests;
using Microsoft.Win32;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Net;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;

namespace CustomDesktop
{
    /// <summary>
    /// Interaction logic for ucMessagingInterface.xaml
    /// </summary>
    public partial class UcChatWindow:UserControl
    {
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);

        public long? contactID { get; internal set; }
        public bool run = true;
        private bool lastTypingState = false;

        private Brush systemTextBrush = Brushes.Indigo;
        private Brush systemHighlightBrush = Brushes.PaleVioletRed;
        private Brush otherAgentTextBrush = Brushes.Black;
        private Brush otherAgentHighlightBrush = Brushes.Transparent;
        private Brush consultAgentTextBrush = Brushes.Black;
        private Brush consultAgentHighlightBrush = Brushes.DarkGray;
        private Brush consultOtherAgentTextBrush = Brushes.Chartreuse;
        private Brush consultOtherAgentHighlightBrush = Brushes.DarkGray;
        private Brush thisAgentTextBrush = Brushes.ForestGreen;
        private Brush thisAgentHighlightBrush = Brushes.Azure;
        private Brush customerTextBrush = Brushes.IndianRed;
        private Brush customerHighlightBrush = Brushes.Ivory;
        private Brush consultBrush = Brushes.DarkSlateGray;
        private Brush conferenceBrush = Brushes.MediumSeaGreen;
        private BitmapImage greenStatus = new BitmapImage(new Uri("Images/green_status.png", UriKind.Relative));
        private BitmapImage yellowStatus = new BitmapImage(new Uri("Images/yellow_status.png", UriKind.Relative));
        private BitmapImage redStatus = new BitmapImage(new Uri("Images/red_status.png", UriKind.Relative));
        private BitmapImage tablet = new BitmapImage(new Uri("Images/method_tablet.png", UriKind.Relative));
        private BitmapImage phone = new BitmapImage(new Uri("Images/method_phone.png", UriKind.Relative));
        private BitmapImage desktop = new BitmapImage(new Uri("Images/method_computer.png", UriKind.Relative));
        private ToolTip custStatusTooltip = new ToolTip();

        //events + delegates
        public delegate void closed(UcChatWindow chatWindow);
        public delegate void closedReasonState(long contactID);
        public delegate void windowTypeChange(ChatWindowTypeChangeEventArgs e);
        public event closed ChatClosed;
        public event closedReasonState ClosedReasonState;
        public event windowTypeChange WindowTypeChange;
        
        private AgentJoinRoomNotification joinRoomDetails;
        public string agentLabel = "Agent(Defaulted)"; // Default
        public string customerLabel = "Customer(Defaulted)"; // Default 
        public bool RequestedToSendEmail = false;
        public bool customerLeftChat = false;
        public bool agentLeftChat = false;
        public bool IsBarge;
        public bool SupervisorLeftChat;
        public int remainingOtherAgentsInChat;
        public bool ConferenceOwner { get; set; }
        public string supervisorId = "";
        public bool consultReceiverBeforeBarge = false;
        System.Windows.Forms.Timer isTypingTimer = new System.Windows.Forms.Timer();

        public Email receivedEmail;
        
        /// <summary>
        /// An enum that keeps track of the current chat window's state
        /// </summary>
        public enum State
        {
            WAITING,
            IN_CHAT,
            CLOSED_REASON,
            FINISHED
        }

        private State state = State.WAITING;

        public enum WindowType
        {
            NORMAL,
            CONSULT_ORIGINATOR,
            CONSULT_RECEIVER,
            CONFERENCE,
            CONFERENCE_WHISPER,
            HISTORY,
            OBSERVE,
            BARGE,
            BARGE_WHISPER,
            EMAIL

        }

        private WindowType type = WindowType.NORMAL;

        public WindowType Type { get { return type; } set { SetType(value); } }
        /// <summary>
        /// Constructor, takes the contact ID of the chat
        /// </summary>
        /// <param name="contactID"></param>
        public UcChatWindow(long contactID, AgentJoinRoomNotification notification)
            : this(contactID, WindowType.NORMAL, notification)
        {
        }

        public UcChatWindow(long name)
        {
            InitializeComponent();
            this.type = WindowType.HISTORY;
            SetType(type);
        }

        //Email
        public UcChatWindow(Contact contact)
        {
            InitializeComponent();
            this.type = WindowType.EMAIL;
            this.gChatWindow.Visibility = Visibility.Hidden;
            this.gEmailWindow.Visibility = Visibility.Visible;
            SetType(type);
        }

        public UcChatWindow(long contactID, WindowType type, AgentJoinRoomNotification notification)
        {
            try
            {
                joinRoomDetails = notification;
                InitializeComponent();
                ToolTipService.SetToolTip(imgCustStatus, custStatusTooltip);
                ToolTipService.SetToolTip(lblCustomer, custStatusTooltip);
                this.type = type;
                this.contactID = contactID;
                imgCustStatus.Source = greenStatus;

                GetConnectionType(notification);

                //Check if we are to email the customer the transcript
                if (notification != null && notification.body.timerList.emailChatLogToCustomer)
                {
                    RequestedToSendEmail = true;
                }

                //Count number of agents
                if (notification != null)
                {
                    remainingOtherAgentsInChat = notification.body.currentParticipantList.participants.Count-1;
                }

                //WS stuff
                if (type != WindowType.CONSULT_ORIGINATOR && type != WindowType.CONFERENCE_WHISPER && type != WindowType.BARGE_WHISPER)
                {
                    PopulateTranscriptToDate();
                }

                ProcessResources();
                PopulateFrequentlyUsedPhrases();
                PopulateFrequentlyUsedURLs();

                //IsTypingTimer
                isTypingTimer.Interval = 1000;
                isTypingTimer.Tick += new EventHandler(IsTypingTick);
                lblAgentIsTyping.FontSize = 12;

                //add event handlers
                btnSend.Click += BtnSendClick;               
                SetType(type);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                MessageBox.Show(ex.StackTrace);
            }
        }

        private void GetConnectionType(AgentJoinRoomNotification notification)
        {
            if (notification != null && notification.body.deviceType != null){
                //More work required here to check the user agent
                if (notification.body.deviceType.Contains("Chrome"))
                {
                    customerConnectMethod.Source = desktop;
                }
                else if (notification.body.deviceType.Contains("Firefox"))
                {
                    customerConnectMethod.Source = tablet;
                }
                //Internet Explorer
                else if (notification.body.deviceType.Contains("MSIE"))
                {
                    customerConnectMethod.Source = phone;
                }
            }
            else
            {
                customerConnectMethod.Source = null;
            }
            
        }
        
        public void WsSendMessage(String msg, BaseRequest.JSmessageType type)
        {
            //Create new message object
            NewMessageRequest newMessage = new NewMessageRequest();
            newMessage.apiVersion = NewMessageRequest.JSapiVersion.one_point_zero;
            newMessage.body.guid = this.contactID.Value;
            newMessage.body.message = msg;
            newMessage.body.messageType = type;

            //Encode and send down socket
            Wsock.Send(newMessage);

            //Append message to chat window
            if (type.Equals(BaseRequest.JSmessageType.normal))
            {
                AppendAsAgent(JEncoder.ConvertToUnixTime(DateTime.UtcNow), agentLabel, msg);
            }
            else
            {
                AppendAsWhisperAgent(JEncoder.ConvertToUnixTime(DateTime.UtcNow), agentLabel, msg);
            }
        }

        public void SetType(WindowType type)
        {
            //Set up the args object before the previous type is lost.
            ChatWindowTypeChangeEventArgs args = new ChatWindowTypeChangeEventArgs(){ ChatWindow = this, CurrentType = type, PreviousType = this.type};

            this.type = type;
            switch (this.type)
            {
                case WindowType.NORMAL:
                    //Use Default Header
                    lblTitle.Content = StringResources.ChatWindowCustomerTitle;
                    lblChatEnded.Content = StringResources.ChatWindowCustomerChatEndedMessage;
                    Background = Brushes.Transparent;
                    imgCustStatus.Visibility = Visibility.Visible;
                    lblCustomer.Visibility = Visibility.Visible;
                    customerConnectMethod.Visibility = Visibility.Visible;
                    cbFreqUsedPhrases.Visibility = Visibility.Visible;
                    cbFreqUsedURLs.Visibility = Visibility.Visible;
                    break;
                case WindowType.HISTORY:
                    lblTitle.Content = "Customer Chat History";
                    Background = Brushes.Transparent;
                    imgCustStatus.Visibility = Visibility.Hidden;
                    lblCustomer.Visibility = Visibility.Hidden;
                    btnSend.IsEnabled = false;
                    cbFreqUsedPhrases.IsEnabled = false;
                    cbFreqUsedURLs.IsEnabled = false;
                    tbComment.IsEnabled = false;
                    break;
                case WindowType.OBSERVE:
                    lblTitle.Content = "Observing Chat";
                    Background = Brushes.Transparent;
                    imgCustStatus.Visibility = Visibility.Visible;
                    lblCustomer.Visibility = Visibility.Visible;
                    btnSend.IsEnabled = false;
                    btnSend.Visibility = Visibility.Hidden;
                    btnBarge.IsEnabled = true;
                    btnBarge.Visibility = Visibility.Visible;
                    cbFreqUsedPhrases.IsEnabled = false;
                    cbFreqUsedURLs.IsEnabled = false;
                    tbComment.IsEnabled = false;
                    break;
                case WindowType.CONSULT_ORIGINATOR:
                    lblTitle.Content = StringResources.ChatWindowConsultTitle;
                    lblChatEnded.Content = StringResources.ChatWindowAgentChatEndedMessage;
                    btnCompleteConference.Visibility = Visibility.Visible;
                    btnCompleteTransfer.Visibility = Visibility.Visible;
                    UpdateIconColourStatus("Yellow");
                    imgCustStatus.Visibility = Visibility.Visible;
                    lblCustomer.Content = "Consult Status: ";
                    lblCustomer.Visibility = Visibility.Visible;
                    cbFreqUsedPhrases.Visibility = Visibility.Hidden;
                    cbFreqUsedURLs.Visibility = Visibility.Hidden;
                    Background = consultBrush;
                    break;
                case WindowType.CONSULT_RECEIVER:
                    lblTitle.Content = StringResources.ChatWindowConsultTitle;
                    lblChatEnded.Content = StringResources.ChatWindowAgentChatEndedMessage;
                    imgCustStatus.Visibility = Visibility.Hidden;
                    lblCustomer.Visibility = Visibility.Hidden;
                    cbFreqUsedPhrases.Visibility = Visibility.Hidden;
                    cbFreqUsedURLs.Visibility = Visibility.Hidden;
                    Background = consultBrush;
                    break;
                case WindowType.CONFERENCE:
                    lblTitle.Content = StringResources.ChatWindowConferenceTitle;
                    lblChatEnded.Content = StringResources.ChatWindowCustomerChatEndedMessage;
                    imgCustStatus.Visibility = Visibility.Visible;
                    lblCustomer.Visibility = Visibility.Visible;
                    cbFreqUsedPhrases.Visibility = Visibility.Visible;
                    cbFreqUsedURLs.Visibility = Visibility.Visible; 
                    Background = conferenceBrush;
                    break;
                case WindowType.CONFERENCE_WHISPER:
                    lblTitle.Content = "Conference Whisper";
                    lblChatEnded.Content = StringResources.ChatWindowCustomerChatEndedMessage;
                    imgCustStatus.Visibility = Visibility.Hidden;
                    lblCustomer.Visibility = Visibility.Hidden;
                    cbFreqUsedPhrases.Visibility = Visibility.Hidden;
                    cbFreqUsedURLs.Visibility = Visibility.Hidden;
                    btnClose.Visibility = Visibility.Hidden;
                    Background = consultBrush;
                    break;
                case WindowType.BARGE:
                    lblTitle.Content = "Chat Barge";
                    lblChatEnded.Content = StringResources.ChatWindowCustomerChatEndedMessage;
                    imgCustStatus.Visibility = Visibility.Visible;
                    lblCustomer.Visibility = Visibility.Visible;
                    cbFreqUsedPhrases.Visibility = Visibility.Visible;
                    cbFreqUsedURLs.Visibility = Visibility.Visible;
                    cbFreqUsedPhrases.IsEnabled = true;
                    cbFreqUsedURLs.IsEnabled = true;
                    tbComment.IsEnabled = true;
                    btnBarge.Visibility = Visibility.Hidden;
                    btnBarge.IsEnabled = false;
                    btnSend.Visibility = Visibility.Visible;
                    btnSend.IsEnabled = true;
                    Background = conferenceBrush;
                    break;
                case WindowType.BARGE_WHISPER:
                    lblTitle.Content = "Barge Whisper";
                    lblChatEnded.Content = StringResources.ChatWindowCustomerChatEndedMessage;
                    imgCustStatus.Visibility = Visibility.Hidden;
                    lblCustomer.Visibility = Visibility.Hidden;
                    cbFreqUsedPhrases.Visibility = Visibility.Hidden;
                    cbFreqUsedURLs.Visibility = Visibility.Hidden;
                    btnClose.Visibility = Visibility.Hidden;
                    btnSend.IsEnabled = true;
                    Background = consultBrush;
                    break;
                default:
                    log.Error("Invalid case in SetType");
                    break;
                    
            }
            if (WindowTypeChange != null)
            {
                WindowTypeChange(args);
            }

        }

        public void ShowConferenceControls(bool visible)
        {
            if (visible)
            {
                gConferenceControls.Visibility = Visibility.Visible;
                btnDrop.IsEnabled = true;
                btnPromote.IsEnabled = true;
            }
            else
            {
                gConferenceControls.Visibility = Visibility.Collapsed;
                btnDrop.IsEnabled = false;
                btnPromote.IsEnabled = false;
            }
            
        }

        public void ShowTransferControls()
        {
            gTransferControls.Visibility = Visibility.Visible;
        }

        /// <summary>
        /// Event handler for the Send button
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnSendClick(object sender, EventArgs e)
        {
            SendCorrectMessage();
        }
       

        /// <summary>
        /// Removes the send functionality available to the agent as the customer has left chat
        /// Thinking to not auto transition to closed reason selection for the reason that the agent 
        /// may wish to review the chat first
        /// </summary>
        public void DisableSendingNewMessages()
        {
            imgCustStatus.Source = redStatus;
            btnSend.IsEnabled = false;
            tbComment.IsEnabled = false;
            cbFreqUsedPhrases.IsEnabled = false;
            cbFreqUsedURLs.IsEnabled = false;
            log.Info("Disabled sending of new messages, page push URLs, Frequently used phrases");

            if (type == WindowType.CONSULT_ORIGINATOR)
            {
                btnCompleteConference.IsEnabled = false;
                btnCompleteTransfer.IsEnabled = false;
                log.Info("Disabled complete conference & complete transfer buttons");

            }

            if (type == WindowType.CONFERENCE_WHISPER)
            {
                btnDrop.IsEnabled = false;
                btnPromote.IsEnabled = false;
                log.Info("Disabled drop & promote passive buttons");

            }
        }

        private void PopulateFrequentlyUsedPhrases()
        {
            try
            {
                List<AgentJoinRoomNotification.Phrases> myPhrases = new List<AgentJoinRoomNotification.Phrases>();
                List<AgentJoinRoomNotification.Phrases> SortedPhraseList = new List<AgentJoinRoomNotification.Phrases>();

                List<AgentJoinRoomNotification.FrequentlyUsedPhraseList> freqUsedPhrases = joinRoomDetails.body.frequentlyUsedPhraseList;
                if (freqUsedPhrases != null)
                {
                    foreach (var skillset in freqUsedPhrases)
                    {
                        if (skillset.phrases != null)
                        {
                            foreach (var phrase in skillset.phrases)
                            {
                                AgentJoinRoomNotification.Phrases aPhrase = new AgentJoinRoomNotification.Phrases();
                                aPhrase.name = phrase.name;
                                aPhrase.phraseText = phrase.phraseText;

                                myPhrases.Add(aPhrase);
                                
                            }
                        }
                    }

                    //TEMP WORKAROUND
                    myPhrases = (
                                    from o in myPhrases
                                    orderby o.name, o.phraseText descending
                                    group o by o.name into g
                                    select g.First()
                                  ).ToList();

                    //Sort the list
                    SortedPhraseList = myPhrases.OrderBy(o => o.name).ToList();
                }

                
                cbFreqUsedPhrases.Items.Clear();

                if (SortedPhraseList != null && SortedPhraseList.Count > 0)
                {
                    foreach (AgentJoinRoomNotification.Phrases freqUsedPhrase in SortedPhraseList)
                    {
                        cbFreqUsedPhrases.Items.Add(freqUsedPhrase);
                    }

                    log.Info("Finished populating frequently used phrases for contactId  - " + contactID);
                }

                else
                {
                    //no closed reasons
                }
            }
            catch (Exception ex)
            {
                log.Error("An exception occured in populating frequently used phrases", ex);
            }
        }

        private void PopulateFrequentlyUsedURLs()
        {
            try
            {
                List<AgentJoinRoomNotification.Url> myUrls = new List<AgentJoinRoomNotification.Url>();
                List<AgentJoinRoomNotification.Url> SortedURLList = new List<AgentJoinRoomNotification.Url>();

                List<AgentJoinRoomNotification.PagePushURLList> pushUrls = joinRoomDetails.body.pagePushURLList;
                if (pushUrls != null)
                {
                    foreach (var skillset in pushUrls)
                    {
                        if (skillset.urls != null)
                        {
                            foreach (var url in skillset.urls)
                            {
                                AgentJoinRoomNotification.Url aUrl = new AgentJoinRoomNotification.Url();
                                aUrl.description = url.description;
                                aUrl.url = url.url;
                                
                                myUrls.Add(aUrl);

                            }
                        }
                    }

                    //TEMP WORKAROUND
                    myUrls = (
                                    from o in myUrls
                                    orderby o.url, o.description descending
                                    group o by o.url into g
                                    select g.First()
                                  ).ToList();

                    //Sort the list
                    SortedURLList = myUrls.OrderBy(o => o.url).ToList();
                }

                cbFreqUsedURLs.Items.Clear();

                if (SortedURLList != null && SortedURLList.Count > 0)
                {
                    foreach (AgentJoinRoomNotification.Url pagePushURL in SortedURLList)
                    {
                        cbFreqUsedURLs.Items.Add(pagePushURL);
                    }

                    log.Info("Finished populating page push URLs for contactId - " + contactID);
                }

                else
                {
                    //no closed reasons
                }
            }
            catch (Exception ex)
            {
                log.Error("An exception occured in populating page push URLs", ex);
            }
        }


        public string FormatLabel(string label)
        {
            string removeLeadingAndEndingBrackets;
            removeLeadingAndEndingBrackets = label.TrimStart('[');
            removeLeadingAndEndingBrackets = removeLeadingAndEndingBrackets.TrimEnd(']');
            return String.Format("[{0}]", removeLeadingAndEndingBrackets);
        }

        DateTime lastAgentNormal;
        DateTime lastAgentWhisper;
        DateTime lastCustomerNormal;
        int timeCounter = 0;
        TimeSpan timeToDisplayIsTyping = new TimeSpan(0, 0, 5);

        public void DisplayIsTyping(IsTypingNotification isTypingNotification)
        {
            string messageType = isTypingNotification.body.messageType;
            string senderType = isTypingNotification.body.sender.type;

            //Find out IsTyping message sender and type and update 'last' time
            if (senderType.ToUpper().Equals("CUSTOMER"))
            {
                lastCustomerNormal = DateTime.Now;
            }
            else if (senderType.ToUpper().Equals("AGENT") && messageType.ToUpper().Equals("NORMAL"))
            {
                lastAgentNormal = DateTime.Now;
            }
            else if (senderType.ToUpper().Equals("AGENT") && messageType.ToUpper().Equals("WHISPER"))
            {
                lastAgentWhisper = DateTime.Now;
            }

            //Check for update to text string
            UpdateIsTypingText();

            //Display label
            lblAgentIsTyping.Visibility = Visibility.Visible;

            //Stop timer so we can restart for full amount of seconds
            if (isTypingTimer.Enabled)
            {
                isTypingTimer.Stop();
            }

            //Reset counter back to zero and start timer
            timeCounter = 0;
            isTypingTimer.Start();

        }

        
        private void IsTypingTick(object sender, EventArgs e)
        {
            timeCounter++;
            if (timeCounter == 5)
            {
                //Hide label and stop timer
                lblAgentIsTyping.Visibility = Visibility.Hidden;
                isTypingTimer.Stop();
            }
            else
            {
                //Check for updates
                UpdateIsTypingText();
            }
        }

        private void UpdateIsTypingText()
        {
            bool agentNormal = DateTime.Now < lastAgentNormal.Add(timeToDisplayIsTyping);
            bool agentWhisper = DateTime.Now < lastAgentWhisper.Add(timeToDisplayIsTyping);
            bool custNormal = DateTime.Now < lastCustomerNormal.Add(timeToDisplayIsTyping);

            string typing = "";

            if (agentWhisper || agentNormal || custNormal)
            {
                typing = "Typing:";
                if (agentNormal)
                {
                    typing = typing + " Agent";
                }

                if (agentWhisper)
                {
                    typing = typing + " AgentWhisper";
                }

                if (custNormal)
                {
                    typing = typing + " Customer";
                }
            }

            lblAgentIsTyping.Content = typing;

        }


        /// <summary>
        /// Populate GUI with previous transcript sorted by timestamp
        /// </summary>
        private void PopulateTranscriptToDate()
        {
            try
            {
                AgentJoinRoomNotification.ConversationTranscript transcript = joinRoomDetails.body.conversationTranscript;
                if (transcript != null && transcript.messages != null)
                {
                    List<AgentJoinRoomNotification.TranscriptEvent> messages = transcript.messages.OrderBy(o => o.timestamp).ToList();

                    if (messages.Count > 0)
                    {
                        AppendAsSystemInfo("Transcript", "Start of transcript");
                        foreach (AgentJoinRoomNotification.TranscriptEvent msg in messages)
                        {
                            if (msg is AgentJoinRoomNotification.TranscriptMessage) {
                                AgentJoinRoomNotification.TranscriptMessage message = (AgentJoinRoomNotification.TranscriptMessage) msg;
                                if (message.sender.type.ToUpper().Equals("CUSTOMER"))
                                {
                                    AppendAsCustomer(message.timestamp, "[Customer - " + message.displayName + "]", message.message);
                                }
                                else
                                {
                                    string messageType = message.messageType.ToUpper();
                                    string prefix;
                                    switch (messageType)
                                    {
                                        case "COMFORTMESSAGE":
                                            prefix = "[ComfortMessage - ";
                                            break;
                                        case "WHISPER":
                                            prefix = "[Whisper - ";
                                            break;
                                        default:
                                            prefix = "[Agent - ";
                                            break;
                                    }

                                    AppendAsAgent(message.timestamp, prefix + message.displayName + "]", message.message);
                                }
                            }
                            else if (msg is AgentJoinRoomNotification.TranscriptPagePush)
                            {
                                AgentJoinRoomNotification.TranscriptPagePush message = (AgentJoinRoomNotification.TranscriptPagePush) msg;
                                if (message.sender.type.ToUpper().Equals("CUSTOMER"))
                                {
                                    AppendAsCustomerURL(message.timestamp, message.displayName, message.url);
                                } else {
                                    AppendAsAgentURL(message.timestamp, message.displayName, message.url);
                                }
                            }
                        }

                        AppendAsSystemInfo("Transcript", "End of transcript");
                        log.Info("Finished processing transcript for contactId - " + contactID);
                    }
                }
            }
            catch (Exception ex)
            {
                log.Error("An exception occured in processing transcript", ex);
            }
        }


        /// <summary>
        /// Method to process resources received in AgentJoinRoom notification
        /// </summary>
        private void ProcessResources()
        {
            try
            {
                AgentJoinRoomNotification.Resources resources = joinRoomDetails.body.resources;
                if (resources != null)
                {
                    //Set the Agent Label 
                    if (!String.IsNullOrEmpty(resources.agentLabel))
                    {
                        this.agentLabel = FormatLabel(resources.agentLabel);
                    }
                    else
                    {
                        this.agentLabel = FormatLabel(agentLabel);
                    }

                    //Set the customer label
                    if (!String.IsNullOrEmpty(resources.customerLabel))
                    {
                        this.customerLabel = FormatLabel(resources.customerLabel);
                    }
                    else
                    {
                        this.customerLabel = FormatLabel(customerLabel);
                    }

                    log.Info("Finished processing resources for contactId - " +  contactID);
                }
            }
            catch (Exception ex)
            {
                this.agentLabel = FormatLabel(agentLabel);
                this.customerLabel = FormatLabel(customerLabel);
                log.Error("An exception occured in processing resources", ex);
            }
        }


        /// <summary>
        /// Populate the dropdown list with the list of closed reason codes
        /// </summary>
        /// <param name="codes"></param>
        public void PopulateClosedReasons(ClosedReasonCodes codes)
        {

            cbClosedReasons.Items.Clear();
            if (codes != null && codes.codes != null && codes.codes.Count > 0)
            {
                int defaultCode = codes.defaultReasonCode;
                foreach (Code closedReasonCode in codes.codes)
                {
                    if (closedReasonCode.closedReason != null)
                    {
                        cbClosedReasons.Items.Add(closedReasonCode);
                        if (closedReasonCode.id == defaultCode)
                        {
                            cbClosedReasons.SelectedItem = closedReasonCode;
                        }
                    }
                }
            }

            else
            {
                //no closed reasons
                cbClosedReasons.IsEnabled = false;
            }
        }


        public ComboBox ComboClosedReasons()
        {
            return cbClosedReasons;
        }

        /// <summary>
        /// Shows the closed reason screen
        /// </summary>
        public void ShowClosedReasonWindow()
        {
            if (RequestedToSendEmail)
            {
                cbEmailToCust.IsChecked = true;
                cbEmailToCust.IsEnabled = false;
            }
            //Hide the chat window as it's no longer needed
            gChatWindow.Visibility = Visibility.Hidden;
            gEmailWindow.Visibility = Visibility.Hidden;
            state = State.CLOSED_REASON;
            //Show the closed reason code + agent note display.
            gWrapUp.Visibility = Visibility.Visible;
            this.Background = Brushes.White;
        }

        public void EndChat()
        {
            try
            {
                state = State.CLOSED_REASON;
                if (ClosedReasonState != null)
                {
                    ClosedReasonState(this.contactID.Value);
                }

                btnClose.IsEnabled = false;
                tbComment.IsEnabled = false;
            }
            catch (Exception ex)
            {
                log.Error("An error occured in EndChat", ex);
            }

        }

        bool replyToAll = false;

        void EmailReplyToAllClick(object sender, RoutedEventArgs e)
        {
            PopulateReplyWindow(true);

            gEmailWindow.Visibility = Visibility.Hidden;
            gEmailReplyWindow.Visibility = Visibility.Visible;

        }

        void EmailReplyClick(object sender, RoutedEventArgs e)
        {
            PopulateReplyWindow(false);
            
            gEmailWindow.Visibility = Visibility.Hidden;
            gEmailReplyWindow.Visibility = Visibility.Visible;
        }

        void EmailForwardScreenClick(object sender, RoutedEventArgs e)
        {
            forwardEmailText.Document.Blocks.Clear();
            forwardEmailText.AppendText(Environment.NewLine);
            forwardEmailText.AppendText(Environment.NewLine);
            forwardEmailText.AppendText("------------------------------------");
            forwardEmailText.AppendText(Environment.NewLine);
            forwardEmailText.AppendText(receivedEmail.text);

            forwardEmailSummary.Text = "FW: " + receivedEmail.subject;

            gEmailWindow.Visibility = Visibility.Hidden;
            gEmailForwardWindow.Visibility = Visibility.Visible;
        }

        private void PopulateReplyWindow(bool replyAll)
        {
            replyEmailText.Document.Blocks.Clear();
            replyToAll = replyAll;

            replyEmailFrom.Text = receivedEmail.mailTo;
            replyEmailTo.Text = receivedEmail.mailFrom;
            replyEmailSummary.Text = "RE: " + receivedEmail.subject;

            if(replyAll){
                replyEmailCc.Text = receivedEmail.mailCC;
                replyEmailBcc.Text = receivedEmail.mailBCC;
            }
            else
            {
                replyEmailCc.Text = string.Empty;
                replyEmailBcc.Text = string.Empty;
            }

            replyEmailText.AppendText(Environment.NewLine);
            replyEmailText.AppendText(Environment.NewLine);
            replyEmailText.AppendText("------------------------------------");
            replyEmailText.AppendText(Environment.NewLine);
            replyEmailText.AppendText(receivedEmail.text);

        }


        void EmailBackClick(object sender, RoutedEventArgs e)
        {
            gEmailWindow.Visibility = Visibility.Visible;
            gEmailReplyWindow.Visibility = Visibility.Hidden;
        }

        void EmailForwardBackClick(object sender, RoutedEventArgs e)
        {
            gEmailWindow.Visibility = Visibility.Visible;
            gEmailForwardWindow.Visibility = Visibility.Hidden;
        }

        void EmailForwardClick(object sender, RoutedEventArgs e)
        {
            try
            {
                EmailManager em = new EmailManager();
                string body = new TextRange(forwardEmailText.Document.ContentStart, forwardEmailText.Document.ContentEnd).Text;
                em.Forward(contactID.ToString(), forwardEmailSummary.Text, body, forwardEmailTo.Text, forwardEmailCc.Text, forwardEmailBcc.Text);

                btnEmailClose.RaiseEvent(new RoutedEventArgs(ButtonBase.ClickEvent));
            }
            catch (Exception ex)
            {
                log.Error("Exception in EmailForwardClick", ex);
            }
        }

        void EmailSendClick(object sender, RoutedEventArgs e)
        {
            EmailManager em = new EmailManager();
            string body = new TextRange(replyEmailText.Document.ContentStart, replyEmailText.Document.ContentEnd).Text;

            if (replyToAll)
            {
                em.SendReply(EmailManager.Recipients.ALL, contactID.ToString(), replyEmailSummary.Text, body);
            }
            else
            {
                em.SendReply(EmailManager.Recipients.CUSTOMER, contactID.ToString(), replyEmailSummary.Text, body);
            }

            //Close
            btnEmailClose.RaiseEvent(new RoutedEventArgs(ButtonBase.ClickEvent));
            

        }

        private void CbAttachmentsSelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            if (cbAttachments.SelectedIndex >= 0)
            {
                EmailAttachment attachment = cbAttachments.SelectedItem as EmailAttachment;
                if (attachment != null)
                {
                    DownloadAttachment(attachment);
                }
            }
        }

        private void RcbAttachmentsSelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            try
            {
                log.Info("Removing attachment");
                if (rcbAttachments.SelectedIndex >= 0)
                {
                    CCMMAgentEmailWS.AWAttachment attachment = rcbAttachments.SelectedItem as CCMMAgentEmailWS.AWAttachment;

                    if (MessageBox.Show("Do you want to delete attachment " + attachment.displayFileName + "?", "Delete Attachment", MessageBoxButton.YesNo) != MessageBoxResult.Yes)
                    {
                        rcbAttachments.SelectedIndex = -1;
                        return;
                    }

                    EmailManager em = new EmailManager();
                    CCMMAgentEmailWS.ArrayOfAWAttachment attachments =  em.RemoveFromAttachmentsList(attachment, contactID.ToString());

                    rcbAttachments.Items.Clear();

                    foreach (CCMMAgentEmailWS.AWAttachment attach in attachments)
                    {
                        rcbAttachments.Items.Add(attach);
                    }
                }

                UpdateOutboundAttachmentsDisplay();
            }
            catch (Exception ex)
            {
                log.Error("Could not remove attachment", ex);
            }
        }

        private void DownloadAttachment(EmailAttachment attachment)
        {
            try
            {
                log.Info("Trying to download attachment");
                log.Info("Attachment URL - " + attachment.internalFilename);
                log.Info("Attachment resolved URL - " + attachment.attachmentURL);
                log.Info("Attachment location - " + Globals.DownloadAttachmentsLocation);
                log.Info("Attachment filename - " + attachment.displayFileName);

                //Check if download directory exists and create if doesn't
                Directory.CreateDirectory(Globals.DownloadAttachmentsLocation);

                WebClient webClient = new WebClient();
                webClient.Headers.Add(HttpRequestHeader.UserAgent, "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_1) AppleWebKit/537.73.11 (KHTML, like Gecko) Version/7.0.1 Safari/537.73.11");
                webClient.DownloadFileCompleted += new AsyncCompletedEventHandler(Completed);
                webClient.DownloadFileAsync(new Uri(attachment.attachmentURL), Globals.DownloadAttachmentsLocation + attachment.displayFileName);
                log.Info("Attachment downloaded to " + Globals.DownloadAttachmentsLocation);
            }
            catch (Exception ex)
            {
                log.Error("Attachment download failed" + ex.ToString());
            }


        }

        private void Completed(object sender, AsyncCompletedEventArgs e)
        {
            MessageBox.Show("Downloaded");
        }

        OpenFileDialog openFileDialog = new OpenFileDialog();
        void EmailAttachClick(object sender, RoutedEventArgs e)
        {

            if (openFileDialog.ShowDialog().HasValue && openFileDialog.ShowDialog().Value)
            {
                EmailManager em = new EmailManager();
                CCMMAgentEmailWS.ArrayOfAWAttachment attachArray = em.AddToAttachmentsList(openFileDialog, contactID.ToString());

                rcbAttachments.Items.Clear();
                foreach(CCMMAgentEmailWS.AWAttachment attach in attachArray){
                    rcbAttachments.Items.Add(attach);
                }

                UpdateOutboundAttachmentsDisplay();

            }

        }

        private void UpdateOutboundAttachmentsDisplay()
        {
            if (rcbAttachments.Items.Count > 0)
            {
                rcbAttachments.Visibility = Visibility.Visible;
                rlAttachments.Visibility = Visibility.Visible;
            }
            else
            {
                rcbAttachments.Visibility = Visibility.Collapsed;
                rlAttachments.Visibility = Visibility.Collapsed;

            }
        }

        /// <summary>
        /// Event handler for the submit button. Sends the agent note and closed reason to the server and closes the conversation.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void ClosedReasonSubmitClick(object sender, RoutedEventArgs e)
        {
            try
            {
                AgentSetClosedReasonRequest closedReason = new AgentSetClosedReasonRequest();
                closedReason.apiVersion = AgentSetClosedReasonRequest.JSapiVersion.one_point_zero;
                closedReason.body.guid = this.contactID.Value;

                //Check if we need to set the send email flag
                if (cbEmailToCust.IsChecked.HasValue && cbEmailToCust.IsChecked.Value)
                {
                    log.Info("Email flag = true");
                    closedReason.body.sendTranscript = true;
                }
                else
                {
                    log.Info("Email flag = false");
                    closedReason.body.sendTranscript = false;
                }

                // Ignore the close note if it is empty
                if (!String.IsNullOrEmpty(tbAgentNote.Text))
                {
                    closedReason.body.agentNote = tbAgentNote.Text;
                }

                if (cbClosedReasons.IsEnabled)
                {
                    Code code = new Code();
                    code = cbClosedReasons.SelectedItem as Code;

                    //Check agent has selected a code if there are codes
                    if (code == null && cbClosedReasons.Items.Count > 0)
                    {
                        MessageBox.Show(MainWindow.GetWindow(this), StringResources.ChatWindowSelectClosedReasonMessage);
                        return;
                    }

                    //Set the ID of the closed reason
                    closedReason.body.closedReasonCode = code.id;

                }
                else
                {
                    log.Info("Closed reasons is not enabled - No closed reason codes provided");
                }


                Wsock.Send(closedReason);
                log.Info("Send SetClosedReasonRequest on contact - " + this.contactID);

            }
            catch (Exception ex)
            {
                log.Error("An exception occured in closedReasonSubmit_Click", ex);
            }
            //regardless of closed reason exception, close chat
            CloseConversation();
        }

        /// <summary>
        /// Closes the current conversation.
        /// </summary>
        private void CloseConversation()
        {
            try
            {
                state = State.FINISHED;
                if (ChatClosed != null)
                {
                    ChatClosed(this);
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        
        public void AppendAsCustomer(long time, String label, String message)
        {
            AppendToChatAreaWithColor(String.Format("{0} {1} {2}", JDecoder.ConvertFromUnixTime(time), FormatLabel(label), message), customerTextBrush, customerHighlightBrush);
        }

        public void AppendAsCustomer(String message)
        {
            AppendAsCustomer(JEncoder.ConvertToUnixTime(DateTime.UtcNow), customerLabel, message);
        }

        public void AppendAsAgent(long time, String label, String message)
        {
            AppendToChatAreaWithColor(String.Format("{0} {1} {2}", JDecoder.ConvertFromUnixTime(time), FormatLabel(label), message), thisAgentTextBrush, thisAgentHighlightBrush);
        }

        public void AppendAsAgent(String message)
        {
            AppendAsAgent(JEncoder.ConvertToUnixTime(DateTime.UtcNow), agentLabel, message);
        }

        public void AppendAsWhisperAgent(long time, String label, String message)
        {
            AppendToChatAreaWithColor(String.Format("{0} {1} {2}", JDecoder.ConvertFromUnixTime(time), FormatLabel(label), message), consultAgentTextBrush, consultAgentHighlightBrush);
        }

        public void AppendAsWhisperAgent(String message)
        {
            AppendAsWhisperAgent(JEncoder.ConvertToUnixTime(DateTime.UtcNow), agentLabel, message);
        }

        public void AppendAsOtherWhisperAgent(long time, String label, String message)
        {
            AppendToChatAreaWithColor(String.Format("{0} {1} {2}", JDecoder.ConvertFromUnixTime(time), FormatLabel(label), message), consultOtherAgentTextBrush, consultOtherAgentHighlightBrush);
        }

        public void AppendAsOtherWhisperAgent(String message)
        {
            AppendAsOtherWhisperAgent(JEncoder.ConvertToUnixTime(DateTime.UtcNow), "WhisperThem", message);
        }

        /// <summary>
        /// Append to Agents chatbox as a System User - Used for system info
        /// </summary>
        /// <param name="message"></param>
        public void AppendAsSystemInfo(String type, String message)
        {
            AppendToChatAreaWithColor(String.Format("{0} {1}: {2}", JDecoder.ConvertFromUnixTime(JEncoder.ConvertToUnixTime(DateTime.UtcNow)), type, message), systemTextBrush, systemHighlightBrush);
        }

        public void AppendAsHistoricChat(String message)
        {
            AppendToChatAreaWithColor(message, customerTextBrush, customerHighlightBrush);
        }

        public void AppendAsAgentURL(long time, String label, String url)
        {
            AppendToChatAreaAsURL(JDecoder.ConvertFromUnixTime(time), label, url, consultAgentTextBrush, customerHighlightBrush);
        }

        public void AppendAsCustomerURL(long time, String label, String url)
        {
            AppendToChatAreaAsURL(JDecoder.ConvertFromUnixTime(time), label, url, customerTextBrush, customerHighlightBrush);
        }

        /// <summary>
        /// Method that appends a message to the chat area in the given colours
        /// </summary>
        /// <param name="text"></param>
        /// <param name="colorToUse"></param>
        /// <param name="background"></param>
        private void AppendToChatAreaWithColor(String text, Brush colorToUse, Brush background)
        {
            try
            {
                if (String.IsNullOrWhiteSpace(text))
                {
                    return;
                }
                //Unfortunately, adding text to RichTextBox is not straightforward and requires this layering of various objects.
                Paragraph p = new Paragraph(new Run(text));
                p.Foreground = colorToUse;
                p.Background = background;
                p.Margin = new System.Windows.Thickness(0);
                ChatArea.Blocks.Add(p);

                //Scroll to the end of the text box
                rTextBox.ScrollToEnd();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void AppendToChatAreaAsURL(String time, String label, String url, Brush colorToUse, Brush background)
        {
            try
            {
                if (String.IsNullOrWhiteSpace(url))
                {
                    return;
                }

                string theUrl = url;

                if (!theUrl.StartsWith("http"))
                {
                    theUrl = "http://" + theUrl;
                }
                //Unfortunately, adding text to RichTextBox is not straightforward and requires this layering of various objects.
                Paragraph p = new Paragraph();
                p.Foreground = colorToUse;
                p.Background = background;
                p.Margin = new System.Windows.Thickness(0);

                Run run = new Run(time + " " + FormatLabel(label) + " Pushed URL - ");

                Run run2 = new Run(theUrl);
                Hyperlink hlink = new Hyperlink(run2);

                hlink.NavigateUri = new Uri(theUrl);
                hlink.RequestNavigate += new System.Windows.Navigation.RequestNavigateEventHandler(HlinkRequestNavigate);
                hlink.IsEnabled = true;

                p.Inlines.Add(run);
                p.Inlines.Add(hlink);

                ChatArea.Blocks.Add(p);

                //Scroll to the end of the text box
                rTextBox.ScrollToEnd();
            }
            catch (Exception ex)
            {
                log.Error("Exception thrown during URL append", ex);
            }
        }

        void HlinkRequestNavigate(object sender, System.Windows.Navigation.RequestNavigateEventArgs e)
        {
            log.Info("Opening URL");

            if (!contactID.HasValue)
            {
                return;
            }

            if (MainWindow.consulting.ContainsKey(contactID.Value) && !MainWindow.chatrooms.ContainsKey(contactID.Value))
            {
                UcChatTab ct = MainWindow.chatrooms[MainWindow.consulting[contactID.Value]];

                if (MainWindow.chatExtraTabsList.ContainsKey(ct))
                {
                    TabControl tc = MainWindow.chatExtraTabsList[ct];
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        AddPagePushTab(e.Uri.AbsoluteUri, e.Uri.Host, tc);
                    }));
                }
            }
            else if(MainWindow.chatrooms.ContainsKey(contactID.Value))
            {
                UcChatTab ct = MainWindow.chatrooms[contactID.Value];

                if (MainWindow.chatExtraTabsList.ContainsKey(ct))
                {
                    TabControl tc = MainWindow.chatExtraTabsList[ct];
                    this.Dispatcher.Invoke((Action)(() =>
                    {
                        AddPagePushTab(e.Uri.AbsoluteUri, e.Uri.Host, tc);
                    }));
                }
            }
        }

        public void AddPagePushTab(string url, string name, TabControl tc)
        {
            //PagePush
            TabItem tabItem = new TabItem();
            Grid grid = null;
            WebBrowser wb = new WebBrowser();

            wb.HorizontalAlignment = System.Windows.HorizontalAlignment.Stretch;
            wb.VerticalAlignment = System.Windows.VerticalAlignment.Stretch;
            wb.Navigate(url);

            grid = new Grid();
            grid.Children.Add(wb);

            tabItem.Header = "PP: " + name;
            tabItem.Content = grid;

            tc.Items.Add(tabItem);
            tc.SelectedItem = tabItem;
        }

        

        public void UpdateIconColourStatus(String colour)
        {
            switch (colour)
            {
                case "Green":
                    imgCustStatus.Source = greenStatus;
                    return;
                case "Red":
                    imgCustStatus.Source = redStatus;
                    return;
                case "Yellow":
                    imgCustStatus.Source = yellowStatus;
                    return;
                default:
                    return;
            }
            
        }

        public void EnableConsult()
        {
            btnCompleteConference.IsEnabled = true;
            btnCompleteTransfer.IsEnabled = true;
        }

        private void SendCorrectMessage()
        {
            if (type == WindowType.NORMAL || type == WindowType.CONFERENCE || type == WindowType.BARGE)
            {
                WsSendMessage(tbComment.Text, BaseRequest.JSmessageType.normal);
            }

            if (type == WindowType.CONSULT_RECEIVER || type == WindowType.CONSULT_ORIGINATOR || type == WindowType.CONFERENCE_WHISPER
                || type == WindowType.BARGE_WHISPER)
            {
                WsSendMessage(tbComment.Text, BaseRequest.JSmessageType.whisper);
            }

            tbComment.Text = string.Empty;
            btnSend.IsEnabled = false;
            lastTypingState = false;
        }


        /// <summary>
        /// Event handler for the comment box. checks for a return key and sends the message if one is detected.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void TbCommentKeyUp(object sender, KeyEventArgs e)
        {
            try
            {
                if (e.Key == Key.Tab)
                {
                    // Ignore tab for when we alt-tab to/from application
                    return;
                }
                if (e.Key == Key.Return)
                {
                    if (tbComment.Text != string.Empty)
                    {
                        SendCorrectMessage();
                        e.Handled = true;
                    }
                }
                else
                {
                    bool typing = tbComment.Text.Length != 0;
                    if (lastTypingState != typing)
                    {
                        lastTypingState = typing;
                    }
                    SendIsTyping();
                    //if the agent is typing, enable the send button
                    btnSend.IsEnabled = typing;
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private DateTime lastIsTypingNormalSent = DateTime.Now;
        private DateTime lastIsTypingWhisperSent = DateTime.Now;
        TimeSpan timeBetweenMsgs = new TimeSpan(0, 0, 3);

        private void SendIsTyping()
        {
            //Whisper
            if (this.type == WindowType.CONFERENCE_WHISPER || this.type == WindowType.CONSULT_RECEIVER || this.type == WindowType.CONSULT_ORIGINATOR || this.type == WindowType.BARGE_WHISPER)
            {
                
                if (DateTime.Now > lastIsTypingWhisperSent.Add(timeBetweenMsgs))
                {
                    IsTypingRequest isTyping = new IsTypingRequest();
                    isTyping.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
                    isTyping.body.isTyping = true;
                    isTyping.body.guid = this.contactID.Value;
                    isTyping.body.messageType = BaseRequest.JSmessageType.whisper;

                    Wsock.Send(isTyping);

                    lastIsTypingWhisperSent = DateTime.Now;
                }

            }
            //Normal
            else
            {
                if (DateTime.Now > lastIsTypingNormalSent.Add(timeBetweenMsgs))
                {
                    IsTypingRequest isTyping = new IsTypingRequest();
                    isTyping.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
                    isTyping.body.isTyping = true;
                    isTyping.body.guid = this.contactID.Value;
                    isTyping.body.messageType = BaseRequest.JSmessageType.normal;

                    Wsock.Send(isTyping);
                    
                    lastIsTypingNormalSent = DateTime.Now;
                }
            }

            
        }

        /// <summary>
        /// Event handler for the closed reason combobox selection changed event.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void CbClosedReasonsSelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            //In wpf, if this event is not handled it is fired back up the control tree and handled by the parent tab control, but
            //That is not the behaviour needed here so flag the selection changed event as handled.
            e.Handled = true;
        }

        private void SendToChatbox(String s){
            tbComment.Text = s;
            btnSend.IsEnabled = true;
        }

        private void CbFreqPhrasesSelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            if (cbFreqUsedPhrases.SelectedIndex >= 0)
            {
                AgentJoinRoomNotification.Phrases phrase = cbFreqUsedPhrases.SelectedItem as AgentJoinRoomNotification.Phrases;
                SendToChatbox(phrase.phraseText);
                cbFreqUsedPhrases.SelectedIndex = -1;
            }
            e.Handled = true;
        }

        private void CbFreqURLsSelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            if (cbFreqUsedURLs.SelectedIndex >= 0)
            {
                AgentJoinRoomNotification.Url url = cbFreqUsedURLs.SelectedItem as AgentJoinRoomNotification.Url;

                WsSendPagePushMessage(url.url);
                AppendAsAgentURL(JEncoder.ConvertToUnixTime(DateTime.UtcNow), agentLabel, url.url);
                cbFreqUsedURLs.SelectedIndex = -1;
            }
            e.Handled = true;
        }

        private void WsSendPagePushMessage(String url){
            NewPagePushRequest pagePush = new NewPagePushRequest();
            pagePush.apiVersion = NewPagePushRequest.JSapiVersion.one_point_zero;
            pagePush.body.guid = this.contactID.Value;
            pagePush.body.destination = NewPagePushRequest.JSpagePushDestination.currentWindow;
            pagePush.body.url = url;

            Wsock.Send(pagePush);
        }

        /// <summary>
        /// Utility method to return the current state of this chat Window.
        /// </summary>
        /// <returns></returns>
        public State GetState()
        {
            return state;
        }
      
    }

    public struct ChatWindowTypeChangeEventArgs
    {
        public CustomDesktop.UcChatWindow.WindowType PreviousType { get; internal set; }
        public CustomDesktop.UcChatWindow.WindowType CurrentType { get; internal set; }
        public CustomDesktop.UcChatWindow ChatWindow { get; internal set; }
    }
}
