﻿//////////////////////////////////////////////////////////////////////////////
// CCTConnector.cs
//
// Copyright 2015 Avaya Inc. All Rights Reserved.
//
// Usage of this source is bound to the terms described in
// Licenses/License.txt
//
// Avaya - Confidential & Proprietary. Use pursuant to your signed agreement or
// Avaya Policy
//////////////////////////////////////////////////////////////////////////////
using Nortel.CCT;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows;

namespace CustomDesktop
{
    /// <summary>
    /// Class that handles all interaction with CCT. May update some fields on the main window through events (Ready state as an example).
    /// </summary>
    public class CCTConnector : ICCTConnector
    {
        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);

        //events
        public event contactReceived ContactReceived;
        public event contactDropped ContactDropped;
        public event consultContactReceived ConsultContactReceived;
        public event consultCreated ConsultCreated;
        public event conferenceComplete ConferenceComplete;
        public event readyStateChanged ReadyStateChanged;
        public event loginStateChanged LoginStateChanged;
        public event transferComplete TransferComplete;
        public event connectionStateChanged ConnectionStateChanged;
        public event consultRejected ConsultRejected;
        public event consultAccepted ConsultAccepted;

        //CCT objects
        //CCT Toolkit - CCT entry point
        private Toolkit cctToolkit = new Toolkit();
        //CCT Session - Models the current CCT Session
        private static ISession cctSession;
        //CCT Address - Models the current CCT Address
        private static IAddress address;
        //CCT Terminals - List of Terminals associated with the current session
        private List<ITerminal> terminals = new List<ITerminal>();
        //CCT Agent - The current in use CCT agent
        private IAgent agent;
        //CCT Terminal Connections - Dictionary of CCT Terminal Connections mapped to the contact ID
        private Dictionary<long, ITerminalConnection> terminalConnections = new Dictionary<long, ITerminalConnection>();
        //CCT Agents - If, on connection, the user is mapped to multiple CCT Agents, they are stored here for retrieval by the MultipleAgentDialog.
        private IAgent[] ReceivedAgents;
        public static Dictionary<string, IConnection> observingChats = new Dictionary<string, IConnection>();

        //Bool to hold connectivity state.
        public bool connected = false;
        //Dictionary of Contact objects against their contact IDs.
        private Dictionary<long, Contact> contacts = new Dictionary<long, Contact>();
        //Bool that is set if this application has requested a logout of the current agent. Unset when the logout event is received.
        private bool logoutRequested = false;

        private bool haveLoggedInAgent = false;

        public static bool connectionToActiveDropped = false;

        //Basic enum for use in the ConnectionStateChanged Event.
        public enum CCTConnectionState
        {
            CONNECTED,
            DISCONNECTED_BY_APPLICATION,
            DISCONNECTED_OTHER
        }

        public static ISession GetCCTSession(){
            return cctSession;
        }

        public static IAddress GetCCTAddress()
        {
            return address;
        }

        //Debugging
        public List<ITerminal> DebugTerminals()
        {
            return terminals;
        }
        public Dictionary<long, ITerminalConnection> DebugTerminalConnections(){
            return terminalConnections;
        }
        public Dictionary<long, Contact> DebugContacts()
        {
            return contacts;
        }

        public bool GetAgentIsReadyState()
        {
            return agent.IsReady;
        }
        /// <summary>
        /// Log in to CCT with the given credentials object
        /// </summary>
        /// <param name="credentials"></param>
        /// <returns></returns>
        public ResultCode Connect(LoginControl.LoginDetails credentials)
        {
            ResultCode result = new ResultCode();
            try
            {
                //Don't attempt connection if already connected
                if (!cctToolkit.IsConnected)
                {
                    //configure the toolkit with the given credentials.
                    ConfigureCCTToolkit(credentials);

                    RegisterForEventsBeforeConnect();
                    //actually connect to cct.
                    cctSession = cctToolkit.Connect();
                    //Register for relevant CCT Events.
                    RegisterForEvents();
                    //Extract relevant information from the cct session.
                    GetSessionData();
                }

                switch (cctSession.Users[0].UserType)
                {
                    case UserType.Agent:
                        log.Info("Login Type - Agent");
                        Globals.usertype = UserType.Agent;
                        break;
                    case UserType.AgentSupervisor:
                        log.Info("Login Type - AgentSupervisor");
                        Globals.usertype = UserType.AgentSupervisor;
                        break;
                    case UserType.Supervisor:
                        log.Info("Login Type - Supervisor");
                        log.Info("Exiting");
                        cctToolkit.Disconnect();
                        MessageBox.Show("Supervisor type is unsupported - Please log in as an agent or agentsupervisor");
                        Environment.Exit(0);
                        break;
                    default:
                        break;
                }

                return GetCctAgent();


            }
            catch (UnauthorizedAccessException uae)
            {
                //Usually an error with a password, or typo in userID or domain.
                log.Info("Exception caught in cct login: " + uae);               
                result.ResultType = ResultCode.ResultCodeType.CCT_CREDENTIALS_ERROR;
                result.Message = uae.ToString();
            }
            catch (OperationFailureException ofe)
            {
                //Something went wrong connecting to CCT.
                log.Info("Exception caught in cct login: " + ofe);
                if (ofe.Error == Error.ServerCommunicationFailure)
                {                    
                    result.ResultType = ResultCode.ResultCodeType.CCT_NOT_REACHABLE;
                    result.Message = ofe.ToString();
                }
                else
                {
                    result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                    result.Message = ofe.ToString();
                }
            }
            catch (Exception ex)
            {
                //General exception.
                log.Info("Exception caught in cct login: " + ex);
                if (cctToolkit.IsConnected)
                {
                    cctToolkit.Disconnect();
                }
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        private void RegisterForEventsBeforeConnect()
        {
            SessionConnectedEventHandler sessionConnectedEventHandler = new SessionConnectedEventHandler(OnSessionConnected);
            SessionDisconnectedEventHandler sessionDisconnectedEventHandler = new SessionDisconnectedEventHandler(OnSessionDisconnected);
            cctToolkit.SessionConnected += sessionConnectedEventHandler;
            cctToolkit.SessionDisconnected += sessionDisconnectedEventHandler;
            cctToolkit.LinkStateChanged += LinkStateEventArgsHandler;
        }

        private ResultCode GetCctAgent()
        {
            ResultCode result = new ResultCode();

            //Get the CCT agent.
            if (cctSession.Agents == null)
            {
                log.Info("No Agents found on CCT Session");
                result.ResultType = ResultCode.ResultCodeType.CCT_NO_AGENT_FOUND;
            }
            else if (cctSession.Agents.Length > 1)
            {
                log.Info(">1 Agents found on CCT Session");
                ReceivedAgents = cctSession.Agents;
                result.ResultType = ResultCode.ResultCodeType.CCT_MULTIPLE_AGENTS_REGISTERED;
            }
            else
            {
                result = SetAgent(cctSession.Agents.First());
                if (Globals.usertype == UserType.AgentSupervisor)
                {
                    log.Info("Usertype is supervisor so getting data of agents sessions");
                    CCTSupervisor cctSuper = new CCTSupervisor(cctToolkit);
                    cctSuper.GetInitialAgentData(cctSession);
                }
                else
                {
                    log.Info("Usertype is not supervisor so not getting data of agents sessions");
                }
            }
            return result;
        }
        /// <summary>
        /// Attempts to login the current agent. If the agent is already logged in, returns a result with type CCT_AGENT_ALREADY_LOGGED_IN
        /// </summary>
        /// <returns></returns>
        private ResultCode DoAgentLogin()
        {
            log.Info("DoAgentLogin called");
            ResultCode result = new ResultCode();
            //When a logout is requested by the client, The new agent state can take some time to be reflected in the local agent object,
            // so if the client has requested a logout treat the agent as logged out.
            if (agent.IsLoggedIn && !logoutRequested)
            {
                result.ResultType = ResultCode.ResultCodeType.CCT_AGENT_ALREADY_LOGGED_IN;
                log.Info("agent is already logged in");
            }
            else
            {
                agent.Login(agent.UserID, AgentState.NotReady);
                log.Info("logged in agent");
                haveLoggedInAgent = true;
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }           
            return result;
        }

        /// <summary>
        /// Get a Pretty String that represents the current logged in agent.
        /// </summary>
        /// <returns></returns>
        public String GetAgentString()
        {
            try
            {
                return (this.agent == null) ? "" : agent.FullName + " (" + agent.UserID + ")";
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in GetAgentString", ex);
                return "";
            }
        }


        /// <summary>
        /// Set the current agent. Also attempts to log in that agent using DoAgentLogin().
        /// </summary>
        /// <param name="agent"></param>
        /// <returns></returns>
        public ResultCode SetAgent(IAgent agent)
        {
            ResultCode result = new ResultCode();
            try
            {
                this.agent = agent;
                //set the agent ID
                Globals.agentID = long.Parse(agent.UserID);
                Globals.agentFirstName = agent.FirstName;
                Globals.agentLastName = agent.LastName;
                if (Globals.CCMMpassword == null)
                {
                    Globals.CCMMpassword = agent.UserID;
                }
                result = DoAgentLogin();
                log.Info(string.Format("Using agent ID {0}, Name: {1} and type {2}", agent.UserID, agent.FullName, agent.UserType));
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in SetAgent", ex);
                result.Message = ex.ToString();
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
            }
            return result;
        }
        
        /// <summary>
        /// Extracts terminals and addresses from the current CCT Session
        /// </summary>
        private void GetSessionData()
        {
            try
            {
                log.Info("Getting session data");
                if (cctSession.Terminals != null && cctSession.Terminals.Length > 0)
                {
                    log.Info(string.Format("{0} terminals returned in connection", cctSession.Terminals.Length));
                    foreach (ITerminal term in cctSession.Terminals)
                    {
                        terminals.Add(term);
                    }
                }
                if (cctSession.Addresses != null && cctSession.Addresses.Length > 0)
                {
                    address = cctSession.Addresses.First();
                    ITerminal[] terminals = address.RelatedTerminals;
                    foreach (ITerminal term in terminals)
                    {
                        if (!this.terminals.Contains(term))
                        {
                            this.terminals.Add(term);
                        }
                    }
                    log.Info(string.Format("address returned : {0}", address));
                }
                
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in GetSessionData", ex);
            }
        }

        /// <summary>
        /// Disconnects the current CCT Session
        /// </summary>
        public ResultCode Disconnect()
        {
            ResultCode result = new ResultCode();
            try
            {
                log.Info("Disconnect called.");
                if (!cctToolkit.IsConnected)
                {
                    result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
                    return result;
                }
                if (agent != null && agent.IsLoggedIn)
                {
                    agent.Logout();
                }
                cctToolkit.Disconnect();
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in Disconnect", ex);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        public void DisconnectObserveConnection(long guid)
        {
            if(observingChats.ContainsKey(guid.ToString())){
                try
                {
                    //observingChats[guid.ToString()].Disconnect();
                }
                catch (Exception ex)
                {
                    log.Error("Could not disconnect from observe", ex);
                }
                observingChats.Remove(guid.ToString());
            }

        }

        /// <summary>
        /// Toggles the current agent ready state
        /// </summary>
        public ResultCode ToggleReady()
        {
            ResultCode result = new ResultCode();
            try
            {
                bool newReadyState = !agent.IsReady;
                log.Info(string.Format("Toggling ready state, ready set to {0}", newReadyState.ToString()));
                agent.IsReady = newReadyState;
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in ToggleReady", ex);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        #region CCT Event Handling
        /// <summary>
        /// Regsiters the required event handlers from CCT
        /// For an explanation of each event, please consult the CCT SDK
        /// </summary>
        public void RegisterForEvents()
        {
            try
            {
                TerminalStateEventHandler terminalStateEventHandler = cctToolkit.CreateEventHandler(new TerminalStateEventHandler(OnTerminalStateChangedEvent));
                AddressStateEventHandler addressStateEventhandler = cctToolkit.CreateEventHandler(new AddressStateEventHandler(OnAddressStateChangedEvent));
                UserPropertyEventHandler userPropertyEventHandler = cctToolkit.CreateEventHandler(new UserPropertyEventHandler(OnUserPropertyChangedEvent));
                TermConnStateEventHandler termConnStateEventHandler = cctToolkit.CreateEventHandler(new TermConnStateEventHandler(OnTermConnStateEvent));
                UserStateEventHandler userStateEventHandler = cctToolkit.CreateEventHandler(new UserStateEventHandler(OnUserState));
                ConnectionStateEventHandler connStateEventHandler = cctToolkit.CreateEventHandler(new ConnectionStateEventHandler(OnConnectionState));
                TermConnPropertyEventHandler termConnPropertyChangeEventHandler = cctToolkit.CreateEventHandler(new TermConnPropertyEventHandler(OnTermConnPropertyChange));
                ContactPropertyEventHandler contactPropertyChangeEventHandler = cctToolkit.CreateEventHandler(new ContactPropertyEventHandler(OnContactPropertyChange));
                ConnectionStateEventHandler remoteConnectionStateEventHandler = cctToolkit.CreateEventHandler(new ConnectionStateEventHandler(OnRemoteConnectionStateChanged));

                cctSession.TerminalStateChanged += terminalStateEventHandler;
                cctSession.AddressStateChanged += addressStateEventhandler;
                cctSession.UserPropertyChanged += userPropertyEventHandler;
                cctSession.TermConnStateChanged += termConnStateEventHandler;
                cctSession.UserStateChanged += userStateEventHandler;
                cctSession.ConnectionStateChanged += connStateEventHandler;
                cctSession.TermConnPropertyChanged += termConnPropertyChangeEventHandler;
                cctSession.ContactPropertyChanged += contactPropertyChangeEventHandler;
                cctSession.RemoteConnectionStateChanged += remoteConnectionStateEventHandler;

                log.Info("Completed adding event handlers");
            }
            catch (Exception ex)
            {
                log.Fatal("Exception caught registering for cct events: ", ex);
                throw ex;
            }
        }


        //Used for debugging
        private void LinkStateEventArgsHandler(object sender, LinkStateEventArgs args)
        {
            bool debug = false;
            if (!debug)
            {
                return;
            }

            try {
                log.Info("LinkStateEventHandler");

                if (args.FailureInfo != null && args.FailureInfo.Reason.Equals(LinkStateReason.Unreachable))
                {
                    return;
                }
                
                if (args.FailureInfo != null )
                {
                    log.Info("--FAILURE INFO--");
                    log.Info("LSEH-" + args.FailureInfo.ToString());
                    log.Info("LSEH-" + args.FailureInfo.Exception);
                    log.Info("LSEH-" + args.FailureInfo.IsRetryPossible);
                    log.Info("LSEH-" + args.FailureInfo.Message);
                    log.Info("LSEH-" + args.FailureInfo.Reason);
                    log.Info("LSEH-" + args.FailureInfo.ReasonText);
                }
                

                log.Info("--LINK ID--");
                log.Info("LSEH-" + args.LinkID.ToString());

                log.Info("--SERVER ADDRESS--");
                log.Info("LSEH-" + args.ServerAddress.ToString());

                log.Info("--STATE--");
                log.Info("LSEH-" + args.State.ToString());
                log.Info("LSEH-" + args.State.Code);

                if (args.State.Code.Equals(LinkStateCode.ConnectedActive)){
                    log.Info("Switchover occured - Get new details");
                    log.Info("Primary - " + cctToolkit.GetLinkState(LinkDesignator.Primary).Code);
                    log.Info("Campus - " + cctToolkit.GetLinkState(LinkDesignator.CampusAlternate).Code);

                    log.Info("start get cct new session & agent");

                    log.Info("finished get cct new session & agent");
                }
                log.Info("LSEH-" + args.State.Reason);

            }
            catch(Exception e)
            {
                log.Error("Exception in linkStateEventHandler", e);
            }

        }

        private void OnRemoteConnectionStateChanged(ConnectionStateEventArgs e)
        {
            try
            {
                log.Info(String.Format("RemoteConnectionStateChanged - PreviousState:{0} NewState:{1} Reason:{2}", e.PreviousState.ToString(), e.NewState.ToString(), e.Reason.ToString()));
                //other party rejected or didnt pick up a consult request.
                long contactID = long.Parse(e.Contact.ID);
                if (!contacts.ContainsKey(contactID))
                {
                    log.Error(String.Format("Received Remoteconnectionstatechanged for contactID {0} but it doesn't exist in contact dictionary", contactID));
                    return;
                }
                Contact contact = GetContact(contactID);
                if (e.NewState == ConnectionState.Disconnected && e.Reason == Reason.ConferenceInitiated)
                {
                    if (ConsultRejected != null)
                    {
                        ITerminalConnection termConnection = terminalConnections.Where(x => x.Key == contactID).Select(x => x.Value).First();
                        termConnection.Connection.Disconnect();
                    }
                    log.Info(String.Format("Received Remoteconnectionstatechanged Disconnected for contactID {0}, removing it and setting {1} as the active contact", contact.ID, contact.ParentID));
                    RemoveContact(contactID);
                    //set the active contact as the original contact
                    SetActiveContact(contact.ParentID);
                    if (ConsultRejected != null)
                    {
                        ConsultRejected(contact);
                    }
                }
                else if (e.NewState == ConnectionState.Established && e.Reason == Reason.ConferenceInitiated)
                {
                    log.Info(String.Format("Received Remoteconnectionstatechanged Established for contactID {0}", contact.ID));
                    if (ConsultAccepted != null)
                    {
                        ConsultAccepted(contact);
                    }                    
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in OnRemoteConnectionStateChanged", ex);
            }
        }

        private void OnSessionDisconnected(SessionDisconnectedEventArgs args)
        {
            if (args.Cause.Equals(DisconnectCause.ConnectionDropped) && String.IsNullOrEmpty(args.SessionToolkit.ActiveServer))
            {
                log.Info("Setting connectionToActiveDropped to true");
                connectionToActiveDropped = true;
            }
            log.Info("New OnSessionDisconnected Cause - " + args.Cause);
            log.Info("New OnSessionDisconnected to ActiveServer - " + args.SessionToolkit.ActiveServer);
            log.Info("New OnSessionDisconnected to Server - " + args.SessionToolkit.Server);
            log.Info("New OnSessionDisconnected to CampusAlternateServer - " + args.SessionToolkit.CampusAlternateServer);

            try
            {
                connected = false;
                //Fire the event back to the main window to update the GUI
                if (this.ConnectionStateChanged != null)
                {
                    ConnectionStateChanged((args.Cause == DisconnectCause.ApplicationInitiated) ? CCTConnectionState.DISCONNECTED_BY_APPLICATION : CCTConnectionState.DISCONNECTED_OTHER);
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in OnSessionDisconnected", ex);
            }
        }

        private void OnSessionConnected(SessionConnectedEventArgs args)
        {
            if (connectionToActiveDropped)
            {
                if (Globals.usertype == UserType.AgentSupervisor)
                {
                    CCTEntities.listOfMyAgents.Clear();
                }
                log.Info("Rebuilding session due to switchover");

                log.Info("Getting CCT session");
                cctSession = args.Session;

                log.Info("Subscribe to events");
                RegisterForEvents();
                
                log.Info("Getting CCT agent");
                GetCctAgent();
                
                log.Info("Get session data");
                terminals.Clear();
                GetSessionData();
                GetTerminalConnectionsData();
                
                connectionToActiveDropped = false;

            }

            // Check if we connected to the RGN
            if(args.SessionToolkit.ActiveServer.Equals(args.SessionToolkit.GeographicAlternateServer, StringComparison.Ordinal))
            {
                log.Info("Connected to the RGN server");

                if (string.IsNullOrEmpty(Globals.RGNWebSocket.ToString()))
                {
                    log.Info("Active connected to the RGN Voice server - No RGN WS specified");
                }
                else
                {
                    log.Info("Active connected to the RGN Voice server - Updating websocket URL to RGN WS");
                    Globals.DefaultWSURL = Globals.RGNWebSocket;
                }
            }
            else
            {
                log.Info("Not connected to the RGN server");
            }

            log.Info("New OnSessionConnected to ActiveServer - " + args.SessionToolkit.ActiveServer);
            log.Info("OnSessionConnected to Server - " + args.SessionToolkit.Server);
            log.Info("OnSessionConnected to CampusAlternateServer - " + args.SessionToolkit.CampusAlternateServer);
            log.Info("Session ID - " + args.Session.ID);
            log.Info("Toolkit State - " + args.ToolkitState);
            try
            {
                connected = true;
                if (this.ConnectionStateChanged != null)
                {
                    ConnectionStateChanged(CCTConnectionState.CONNECTED);
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in OnSessionConnected", ex);
            }
        }

        private void GetTerminalConnectionsData()
        {
            try
            {
                log.Info("Get TerminalConnections data");
                terminalConnections.Clear();
                MainWindow.consulting.Clear();

                log.Info("Amount of Terminals on session " + cctSession.Terminals.Length);
                ITerminalConnection[] itc = cctSession.Terminals[0].TerminalConnections;
                log.Info("Amount of ITerminalConnections on session " + itc.Length);
                foreach (ITerminalConnection tc in itc)
                {
                    Contact c;
                    if (tc.ConsultContact != null)
                    {
                        log.Info("ConsultContact not NULL");
                        c = new Contact(tc.ConsultContact);
                    }
                    else
                    {
                        log.Info("ConsultContact was NULL");
                        c = new Contact(tc.Contact);
                    }

                    log.Info("Re-adding new terminal for ID: " + c.ID);
                    log.Info("Re-adding new terminal for ParentID: " + c.ParentID);


                    if (c.IsConsultContact())
                    {
                        log.Info("Adding a consult contact");
                        terminalConnections.Add(c.ID, tc);
                        MainWindow.consulting.Add(c.ParentID, c.ID);
                    }
                    else
                    {
                        log.Info("Adding a normal contact");
                        if (terminalConnections.ContainsKey(c.ID))
                        {
                            log.Info("Removing old contact first");
                            terminalConnections.Remove(c.ID);
                        }
                        terminalConnections.Add(c.ID, tc);
                        
                    }

                    if (observingChats.ContainsKey(c.ID.ToString()))
                    {
                        log.Info("Removing and adding observe chat for contact: " + c.ID);
                        observingChats.Remove(c.ID.ToString());
                        observingChats.Add(c.ID.ToString(), tc.Connection);

                        foreach (SupervisedAgentNew superAgent in CCTEntities.listOfMyAgents.Values)
                        {
                            if (superAgent.agentChats.ContainsKey(c.ID))
                            {
                                log.Info("Updating chatContact for supervisor agentChats");
                                superAgent.agentChats[c.ID].chatContact = tc.Contact;

                            }
                        }

                    }

                        
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception in GetTerminalConnectionsData", ex);
            }
        }

        public void OnContactPropertyChange(ContactPropertyEventArgs e)
        {
            try
            {
                log.Info("Contact Property event:" + e.ChangedProperty + " : " + e.CurrentCapabilities);
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in OnContactPropertyChanged", ex);
            }
        }

        public void OnTermConnPropertyChange(TermConnPropertyEventArgs e)
        {
            try
            {
                log.Info("TermConnState event:" + e.ChangedProperty + " : " + e.CurrentCapabilities);
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in OnTermConnPropertyChanged", ex);
            }
        }

        public void OnConnectionState(ConnectionStateEventArgs e)
        {
            log.Info("ConnectionStateEventArgs" + e.Reason);
            if (!haveLoggedInAgent)
            {
                log.Info("Ignoring ConnectionStateEvent as this client has not logged in the agent yet.");
                return;
            }
            try
            {
                log.Info("ConnState event:" + e.NewState);
                if (e.NewState == ConnectionState.Routing && e.CurrentCapabilities.CanAlert && e.Connection != null)
                {
                    try
                    {
                        log.Info("Trying to alert connection");
                        e.Connection.Alert();
                    }
                    catch (InvalidConnectionStateException icse)
                    {
                        //Sometimes, CCT will throw an exception saying "Alert cannot be called in alerting state" even though the state was routing. If this occurs, 
                        //contact processing can proceed as normal.
                        log.Error("Alert cannot be called in alerting state ## Check you don't have another instance of CustomDesktop open somewhere with same user logged in");
                        log.Info(e.PreviousState.ToString() + e.NewState.ToString() + e.Reason.ToString() + e.IsRemote.ToString() + icse.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in OnConnectionState", ex);
            }
        }

        
        // Agents to supervise
        public void OnUserState(UserStateEventArgs e)
        {
            try
            {
                CCTSupervisor agentToSuper = new  CCTSupervisor(cctToolkit);
                agentToSuper.SubscribeUserForEvents(e.User);
                //After agent is registered add in their current chats
                agentToSuper.AddExistingChats(e.User.UserID);
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in OnUserState", ex);
            }
        }

        

        public void AddObserveContact(long guid, Contact contact){
            contacts.Add(guid, contact);
        }

        public void RemoveObserveContact(long guid)
        {
            contacts.Remove(guid);
        }

        public void OnTermConnStateEvent(TermConnStateEventArgs e)
        {
            if (!haveLoggedInAgent)
            {
                log.Info("Ignoring TermConnStateEvent as this client has not logged in the agent yet.");
                return;
            }
            try
            {
                if (e == null)
                {
                    log.Error("Received null event in OnTermConnStateEvent");
                    return;
                }
                else if (e.TerminalConnection == null)
                {
                    log.Error("Received null terminalconnection in OnTermConnStateEvent");
                    return;
                }

                log.Info("OnTermConnStateEvent event:" + e.NewState);
                Contact contact;
                log.Info("BARGE TERM STATE CHECK| State - " + e.NewState + " | Reason - " + e.Reason + " | AGENT - " + Globals.agentID);
                switch (e.NewState)
                {
                        
                    case TerminalConnectionState.Active:
                        if (e.Reason == Reason.ConferenceComplete)
                        {
                            //Remove the consult contact and add the main contact and pass it back
                            long contactID = long.Parse(e.Contact.ID);
                            contacts.Remove(contactID);
                            contacts.Add(contactID, new Contact(e.Contact));
                            terminalConnections.Remove(contactID);
                            terminalConnections.Add(contactID, e.TerminalConnection);
                            contact = contacts[contactID];
                            log.Info("Conference has completed of contact " + contactID);
                            if (ConferenceComplete != null)
                            {
                                ConferenceComplete(contact);
                            }
                        }
                        if (e.Reason == Reason.TransferComplete)
                        {
                            //Remove the consult contact and add the main contact and pass it back
                            long contactID = long.Parse(e.Contact.ID);
                            contacts.Remove(contactID);
                            contacts.Add(contactID, new Contact(e.Contact));
                            terminalConnections.Remove(contactID);
                            terminalConnections.Add(contactID, e.TerminalConnection);
                            contact = contacts[contactID];
                            log.Info("Transfer has completed of contact " + contactID);
                            if (TransferComplete != null)
                            {
                                TransferComplete(contact);
                            }
                        }
                        if (e.Reason == Reason.Observe)
                        {
                            //Remove the consult contact and add the main contact and pass it back
                            long contactID = long.Parse(e.Contact.ID);

                            terminalConnections.Add(contactID, e.TerminalConnection);

                            log.Info("Observe has completed of contact " + contactID);
                        }
                        break;
                    case TerminalConnectionState.Ringing:
                        contact = new Contact(e.Contact);
                        if (e.TerminalConnection.GetCapabilities().CanAnswer)
                        {
                            string[] types = e.Contact.ContactTypes;
                            //Check if email
                            if (types == null || types.Length != 1 || String.Equals(types[0].ToUpper(), "EMAIL"))
                            {
                                log.Info("Received email contact");
                                terminalConnections.Add(long.Parse(e.Contact.ID), e.TerminalConnection);
                                contacts.Add(contact.ID, contact);
                                log.Info(String.Format("Adding normal email contact ID {0} to map", contact.ID));
                                if (ContactReceived != null)
                                {
                                    ContactReceived(contact);
                                }
                                return;
                            }
                            if (types == null || types.Length != 1 || !String.Equals(types[0], "Web_Communications"))
                            {
                                log.Info("Received non Web Comms contact, going not ready");
                                agent.IsReady = false;
                                MessageBox.Show(StringResources.NonWCContactErrorMessage, StringResources.Error, MessageBoxButton.OK);
                                return;
                            }
                            if (e.Contact != null && e.Contact.Intrinsics != null && e.Contact.Intrinsics["ConsultContactParentId"] != null)
                            {
                                //consult contact received
                                String consultParentID = e.Contact.Intrinsics["ConsultContactParentId"];
                                //check connections to see if this agent is the originator or the receiver
                                if (IsAgentConsultOriginator(consultParentID))
                                {
                                    terminalConnections.Add(long.Parse(e.Contact.ID), e.TerminalConnection);
                                    //automatically answer terminal connection if this agent is the originator
                                    e.TerminalConnection.Answer();
                                    contacts.Add(contact.ID, contact);
                                    log.Info(String.Format("Adding originating consult contact ID {0} to map", contact.ID));
                                    //ADD the consult window to the existing chat tab
                                    if (ConsultCreated != null)
                                    {
                                        ConsultCreated(contact);
                                    }
                                }
                                else
                                {
                                    terminalConnections.Add(long.Parse(consultParentID), e.TerminalConnection);
                                    contacts.Add(contact.ParentID, contact);
                                    log.Info(String.Format("Adding non originating consult contact ID {0} to map", contact.ParentID));
                                    if (ConsultContactReceived != null)
                                    {
                                        ConsultContactReceived(contact);
                                    }
                                }
                            }
                            else
                            {
                                terminalConnections.Add(long.Parse(e.Contact.ID), e.TerminalConnection);
                                contacts.Add(contact.ID, contact);
                                log.Info(String.Format("Adding normal contact ID {0} to map", contact.ID));
                                if (ContactReceived != null)
                                {
                                    ContactReceived(contact);
                                }
                            }
                        }
                        break;
                    case TerminalConnectionState.Dropped:
                        //TEMP
                        long thecontactID = long.Parse(e.Contact.ID);
                        if (contacts.ContainsKey(thecontactID))
                        {
                            if (e.Reason != Reason.ConferenceComplete)
                            {
                                contacts.Remove(thecontactID);
                                terminalConnections.Remove(thecontactID);
                                log.Info(String.Format("Removed contact ID {0} from map", thecontactID));
                            }
                            if (e.Reason != Reason.ConferenceComplete && e.Reason != Reason.TransferComplete)//dont fire contact dropped if it's the consult contact being dropped
                            {
                                if (e.Contact != null && ContactDropped != null)
                                {
                                    ContactDropped(thecontactID);
                                }
                            }
                        }
                        break;
                    case TerminalConnectionState.Held:
                        break;
                    default:
                        log.Error("Invalid case in OnTermConnStateEvent");
                        break;
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in OnTermConnStateEvent", ex);
            }
        }

        public Contact GetContact(long contactID)
        {
            try
            {
                return contacts[contactID];
            }
            catch (Exception ex)
            {
                log.Error("Exception occured getting contact", ex);
                return null;
            }
        }

        private bool IsAgentConsultOriginator(string consultParentID)
        {
            try
            {
                foreach (long id in terminalConnections.Keys)
                {
                    if (id == long.Parse(consultParentID))
                    {
                        return true;
                    }
                }
                return false;
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in IsAgentConsultOriginator", ex);
                return false;
            }
        }

        public void OnTerminalStateChangedEvent(TerminalStateEventArgs e)
        {
            try
            {
                ResourceState state = e.NewState;
                if (!terminals.Contains(e.Terminal))
                {
                    terminals.Add(e.Terminal);
                }
                log.Info("Terminal State change: " + state);
            }
            catch (Exception ex)
            {
                log.Error("Exception caught on terminal state event: ", ex);
            }
        }

        public void OnAddressStateChangedEvent(AddressStateEventArgs e)
        {
            try
            {
                // If the address is going InService search for existing related connections 
                foreach (ITerminal term in e.Address.RelatedTerminals)
                {
                    try
                    {
                        if (!terminals.Contains(term))
                        {
                            terminals.Add(term);
                        }
                        address = e.Address;
                    }
                    catch (Exception exception)
                    {
                        log.Info(String.Format("Failed to add connection: {0}", exception.Message));
                    }
                }

            }
            catch (Exception ex)
            {
                log.Info("Exception caught on address state event: " + ex);
            }
        }

        public void OnUserPropertyChangedEvent(UserPropertyEventArgs e)
        {
            try
            {
                if (!haveLoggedInAgent)
                {
                    log.Info("Ignoring OnUserPropertyChangedEvent as this client has not logged in the agent yet.");
                    return;
                }
                if (e.ChangedProperty is ReadyStatusAgentProperty)
                {
                    if (ReadyStateChanged != null)
                    {
                        ReadyStateChanged(e.ChangedProperty as ReadyStatusAgentProperty);
                    }
                }
                else if (e.ChangedProperty is LoginStatusAgentProperty )
                {
                    if (LoginStateChanged != null && !logoutRequested)
                    {
                        LoginStateChanged(e.ChangedProperty as LoginStatusAgentProperty);
                    }
                    if (logoutRequested)
                    {
                        logoutRequested = false;
                    }
                }
                log.Info(string.Format("The {0} property has changed on user {1}. current capabilities: {2}", e.ChangedProperty, e.Sender.UserID, e.CurrentCapabilities));
            }
            catch (Exception ex)
            {
                log.Error("Exception caught on user property event: ", ex);
            }
        }
        #endregion

        /// <summary>
        /// handles the rejection of a contact by the agent.
        /// </summary>
        /// <param name="contact"></param>
        public ResultCode RejectContact(Contact contact)
        {
            ResultCode result = new ResultCode();
            try
            {
                log.Info("User declined incoming chat ID: " + contact.ID + ", setting isReady to false");
                long contactID = 0;
                if (contact.IsConsultContact())
                {
                    if (IsAgentConsultOriginator(contact.ID.ToString()))
                    {
                        contactID = contact.ID;
                    }
                    else
                    {
                        contactID = contact.ParentID;
                    }

                    ITerminalConnection termConn = terminalConnections.Where(x => x.Key == contactID).Select(x => x.Value).First();
                    termConn.Connection.Disconnect();
                }
                else
                {
                    contactID = contact.ID;
                }
                ContactDropped(contactID);

                RemoveContact(contactID);
                agent.IsReady = false;
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception occured in RejectContact", ex);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        /// <summary>
        /// Handles acceptance of a contact by the agent.
        /// </summary>
        /// <param name="contact"></param>
        public ResultCode AcceptContact(Contact contact)
        {
            ResultCode result = new ResultCode();
            try
            {
                log.Info("Answering current ringing contact");
                ITerminalConnection terminalConnection = null;
                if (contact.IsConsultContact())
                {
                    terminalConnection = terminalConnections[contact.ParentID];
                }
                else
                {
                    terminalConnection = terminalConnections[contact.ID];
                }
                terminalConnection.Answer();
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception occured trying to answer the contact", ex);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        public ResultCode InitiateConference(long contactID, long agentID)
        {
            ResultCode result = new ResultCode();
            try
            {
                log.Info(string.Format("Trying to consult contact {0} to agent {1}", contactID, agentID));
                KeyValuePair<long, ITerminalConnection> pair = terminalConnections.Where(x => x.Key == contactID).First();
                pair.Value.InitiateConference(StringResources.CCMMTerminalPrefix + agentID);
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception occured trying to intiate the transfer of the previous contact", ex);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        /// <summary>
        /// Blind transfers he given contact to the give skillset
        /// </summary>
        /// <param name="contactID"></param>
        /// <param name="skillset"></param>
        /// <returns></returns>
        public ResultCode BlindTransfer(long contactID, CDEntities.CDContactCentreSkillset skillset)
        {
            ResultCode result = new ResultCode();
            try
            {
                log.Info(string.Format("Trying to blind transfer contact {0} to skillset {1} (CCMSID: {2})", contactID, skillset.name, skillset.ccmsID));
                ITerminalConnection termConn = terminalConnections.Where(x => x.Key == contactID).First().Value;
                //set the needed intrinsics on the contact before transferring
                termConn.Contact.Intrinsics[StringResources.CCTSkillsetNameIntrinsic] = skillset.name;
                termConn.Contact.Intrinsics[StringResources.CCTSkillsetCCMSIDIntrinsic] = skillset.ccmsID.ToString();
                termConn.BlindTransfer(skillset.routepoint);
                RemoveContact(contactID);
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception occured trying to transfer the previous contact", ex);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        /// <summary>
        /// Disconnects a CCT Connection to a specific contact.
        /// </summary>
        /// <param name="contactID"></param>
        /// <returns></returns>
        public ResultCode DisconnectConnection(long contactID)
        {
            ResultCode result = new ResultCode();
            try
            {
                log.Info("CCT Connection - Atemmpting to Disconnect single party from connection | GUID - " + contactID + " | AgentID - " + Globals.agentID);

                if (terminalConnections.ContainsKey(contactID))
                {
                    log.Info("CCT Connection - TerminalConnection: FOUND");
                    ITerminalConnection termConnection = terminalConnections[contactID];
                    termConnection.Connection.Disconnect();
                    RemoveContact(contactID);
                }
                else
                {
                    log.Info("CCT Connection - TerminalConnection: NOT FOUND");
                }
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception caught in disconnect connection for contactID " + contactID, ex);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        public ResultCode RemoveConnection(long contactID)
        {
            ResultCode result = new ResultCode();
            try
            {
                log.Info("CCT Connection - Atemmpting to Drop whole connection | GUID - " + contactID + " | AgentID - " + Globals.agentID);

                if (terminalConnections.ContainsKey(contactID))
                {
                    log.Info("CCT Connection - TerminalConnection: FOUND");
                    ITerminalConnection termConnection = terminalConnections[contactID];
                    termConnection.Connection.Contact.Drop();
                    RemoveContact(contactID);
                }
                else
                {
                    log.Info("CCT Connection - TerminalConnection: NOT FOUND");
                }
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception caught in disconnect connection for contactID " + contactID, ex);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        /// <summary>
        /// Completes the transfer of a contact to another agent.
        /// </summary>
        /// <param name="contact"></param>
        /// <returns></returns>
        public ResultCode CompleteTransfer(Contact contact)
        {
            ResultCode result = new ResultCode();
            try
            {
                ITerminalConnection termConnection = terminalConnections[contact.ParentID];
                //Call the transfer command on the parent's terminalconnection, using the consult contact's IContact
                termConnection.CompleteSupervisedTransfer(terminalConnections[contact.ID].Connection.Contact);
                //Need to remove the base contact and the consult contact
                RemoveContact(contact.ID);
                RemoveContact(contact.ParentID);
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception caught in complete transfer for contactID " + contact.ID, ex);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        /// <summary>
        /// Complete a conference of a consult contact.
        /// </summary>
        /// <param name="contact"></param>
        /// <returns></returns>
        public ResultCode CompleteConference(Contact contact)
        {
            ResultCode result = new ResultCode();
            try
            {
                ITerminalConnection termConnection = terminalConnections[contact.ParentID];
                //Call the conference command on the parent's terminalconnection, using the consult contact's IContact
                termConnection.CompleteConference(terminalConnections[contact.ID].Connection.Contact);
                RemoveContact(contact.ID);
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception caught in complete conference for contactID " + contact.ID, ex);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        /// <summary>
        /// Remove a contact from all internal lists and dictionaries.
        /// </summary>
        /// <param name="contactID"></param>
        public void RemoveContact(long contactID)
        {
            try
            {
                log.Info("Removing contact id " + contactID);
                terminalConnections.Remove(contactID);
                contacts.Remove(contactID);
            }
            catch (Exception ex)
            {
                log.Error("Exception caught in RemoveContact for contactID " + contactID, ex);
            }
        }
        
        /// <summary>
        /// Configures the CCT Toolkit with the given credentials.
        /// </summary>
        /// <param name="credentials"></param>
        private void ConfigureCCTToolkit(LoginControl.LoginDetails credentials)
        {
            try
            {
                cctToolkit.Server = credentials.server;
                cctToolkit.Port = Int32.Parse(credentials.port);

                if (Globals.HAEnabled)
                {
                    cctToolkit.CampusAlternateServer = Globals.DefaultCampusAlternateServer;
                    cctToolkit.GeographicAlternateServer = Globals.DefaultGeographicAlternateServer;
                    cctToolkit.AutoReconnect = Globals.DefaultAutoReconnect;
                }

                if (credentials.useCurrentUser)
                {
                    cctToolkit.Credentials = new Nortel.CCT.CCTCredentials();
                }
                else
                {
                    cctToolkit.Credentials = new Nortel.CCT.CCTCredentials(credentials.user, credentials.domain, credentials.password);
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception occured configuring CCT Toolkit",ex);
            }
        }

        /// <summary>
        /// Logs out the current agent.
        /// </summary>
        public void AgentLogout()
        {
            try
            {
                logoutRequested = true;
                log.Info("Logging out agent");
                this.agent.Logout();
                haveLoggedInAgent = false;
            }
            catch (Exception ex)
            {
                log.Error("Exception occured during agent logout", ex);
            }
        }

        public ResultCode SetActiveContact(long contactID)
        {
            ResultCode result = new ResultCode();
            try
            {
                ITerminalConnection termConnection = terminalConnections[contactID];
                if (termConnection.CurrentState != TerminalConnectionState.Active)
                {
                    //Call unhold on the Terminal connection
                    termConnection.Unhold();
                }
                result.ResultType = ResultCode.ResultCodeType.CCT_SUCCESS;
            }
            catch (Exception ex)
            {
                log.Error("Exception caught in SetActiveContact for contactID " + contactID + " StackTrace:\n" + ex.StackTrace);
                result.ResultType = ResultCode.ResultCodeType.CCT_EXCEPTION;
                result.Message = ex.ToString();
            }
            return result;
        }

        public bool IsConnected()
        {
            return connected;
        }

        public IAgent[] GetReceivedAgents()
        {
            return ReceivedAgents;
        }
    }
}
