﻿//////////////////////////////////////////////////////////////////////////////
// UcChatTab.xaml.cs
//
// Copyright © 2008-2014 Avaya Inc. All rights reserved.
// See: www.avaya.com
//////////////////////////////////////////////////////////////////////////////
using EncoderDecoder;
using EncoderDecoder.Notifications;
using EncoderDecoder.Requests;
using System;
using System.IO;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;

namespace CustomDesktop
{
    /// <summary>
    /// Class that wraps the chat window in a TabItem.
    /// </summary>
    public partial class UcChatTab : TabItem
    {
        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);
        public UcChatWindow ChatWindow { get; private set; }
        public UcChatWindow ConsultWindow { get; private set; }

        //Events + delegates
        public delegate void emailClose(UcChatTab chatTab);
        public delegate void CompleteConference(UcChatTab chatTab);
        public delegate void CompleteTransfer(UcChatTab chatTab);
        public delegate void closed(UcChatTab chatTab);
        public delegate void consultClosed(UcChatTab chatTab);
        public delegate void inClosedReasonState(UcChatTab chatTab);
        public delegate void nonOriginatingConsultClosed(UcChatTab chatTab);
        public delegate void historyClosed(UcChatTab chatTab);
        public delegate void activeConfConsultClosed(UcChatTab chatTab);
        public delegate void activeConfClosed(UcChatTab chatTab);
        public delegate void observeClosed(UcChatTab chatTab);
        public delegate void bargeWhisperClosed(UcChatTab chatTab);
        public delegate void passiveConfClosed(UcChatTab chatTab);
        public delegate void ChatWindowTypeChange(ChatWindowTypeChangeEventArgs e);
        public delegate void Barge(UcChatTab chatTab);
        public event emailClose EmailClose;
        public event CompleteConference CompleteConferenceEvent;
        public event CompleteTransfer CompleteTransferEvent;
        public event closed ChatClosed;
        public event historyClosed HistoryClosed;
        public event consultClosed OriginatingConsultClosed;
        public event nonOriginatingConsultClosed NonOriginatingConsultClosed;
        public event activeConfConsultClosed ActiveConfConsultClosed;
        public event activeConfClosed ActiveConfClosed;
        public event observeClosed ObserveClosed;
        public event bargeWhisperClosed BargeWhisperClosed;
        public event passiveConfClosed PassiveConfClosed;
        public event inClosedReasonState ClosedReasonState;
        public event ChatWindowTypeChange WindowTypeChange;
        public event Barge BargeEvent;


        private const int CONSULT_WINDOW_COLUMN_INDEX = 1;
        private Contact contact = null;
        public Contact Contact { get { return contact; } set { contact = value; } }
        private Contact consultContact = null;
        public Contact ConsultContact { get { return consultContact; } set { consultContact = value; } }
        public long consultedAgent;

        public CDEntities.leaveReason LeaveReason { get; set; }

        public bool PendingConsult { get; set; }

        //WS things
        public AgentJoinRoomNotification joinRoomDetails { get; set; }
        public bool IsCurrentTabInView { get; set; }
        public bool NotificationPending { get; set; }
    
        /// <summary>
        /// Constructor that takes a contact as a parameter. If the contact is a consult contact, a consult window is set up as the main window.
        /// Otherwise, sets up a normal chat window.
        /// </summary>
        /// <param name="contact">The contact representing this chat.</param>
        public UcChatTab(Contact contact)
        {
            InitializeComponent();
            this.contact = contact;
            PendingConsult = false;

            //Handle Email
            if (contact.Intrinsics["CONTACTSUBTYPE"].ToUpper().Equals("EMAIL"))
            {
                this.Header = "Email: " + contact.ID;
                this.ChatWindow = new UcChatWindow(contact);
                this.ChatWindow.contactID = contact.ID;
                this.gContentGrid.Children.Add(ChatWindow);
                this.gContentGrid.ColumnDefinitions[0].MinWidth = 300;

                //Add event handlers
                this.ChatWindow.btnEmailClose.Click += BtnEmailCloseClick;
                return;
            }

            if (contact.IsConsultContact())
            {
                //because the consult contact is creating a new tab instead of adding a window to an existing one, we know
                //that the we are the consult receiver
                this.ChatWindow = new UcChatWindow(contact.ParentID, UcChatWindow.WindowType.CONSULT_RECEIVER, joinRoomDetails);
            }
            else
            {
                //Create a normal ChatWindow
                this.ChatWindow = new UcChatWindow(contact.ID, joinRoomDetails);
            }
            this.Header = "Contact ID: " + ChatWindow.contactID;
            this.gContentGrid.Children.Add(ChatWindow);
            this.gContentGrid.ColumnDefinitions[0].MinWidth = 300;
            //Add event handlers
            this.ChatWindow.ChatClosed += ChatWindowChatClosed;
            this.ChatWindow.btnClose.Click += BtnCloseClick;
            this.ChatWindow.ClosedReasonState += ClosedReasonStateHandler;
            this.ChatWindow.WindowTypeChange += WindowTypeChangeHandler;

        }

        private void BtnEmailCloseClick(object sender, RoutedEventArgs e)
        {
            EmailManager em = new EmailManager();
            em.CloseContact(contact.ID.ToString(), "Closed comment", 0, false);

            if (EmailClose != null)
            {
                EmailClose(this);
            }
        }

        //For customer chat history
        public UcChatTab(long guid)
        {
            InitializeComponent();

            this.Header = "History: " + guid;
            this.ChatWindow = new UcChatWindow(guid);
            this.gContentGrid.Children.Add(ChatWindow);
            this.gContentGrid.ColumnDefinitions[0].MinWidth = 300;
            //Add event handlers
            this.ChatWindow.btnClose.Click += BtnCloseHistoryClick;
        }

        //NEW WS VERSION OF CONSTUCTOR
        public UcChatTab(Contact contact, AgentJoinRoomNotification notification)
        {
            try
            {
                joinRoomDetails = notification;
                InitializeComponent();
                this.contact = contact;
                PendingConsult = false;

                if (contact.IsConsultContact())
                {
                    //because the consult contact is creating a new tab instead of adding a window to an existing one, we know
                    //that the we are the consult receiver
                    this.ChatWindow = new UcChatWindow(contact.ParentID, UcChatWindow.WindowType.CONSULT_RECEIVER, joinRoomDetails);
                }
                else
                {
                    //Create a normal ChatWindow
                    this.ChatWindow = new UcChatWindow(contact.ID, joinRoomDetails);
                }
                this.Header = "Contact ID: " + ChatWindow.contactID;
                this.gContentGrid.Children.Add(ChatWindow);
                this.gContentGrid.ColumnDefinitions[0].MinWidth = 300;
                //Add event handlers
                this.ChatWindow.ChatClosed += ChatWindowChatClosed;
                this.ChatWindow.btnClose.Click += BtnCloseClick;
                this.ChatWindow.btnBarge.Click += BtnBargeClick;
                this.ChatWindow.ClosedReasonState += ClosedReasonStateHandler;
                this.ChatWindow.WindowTypeChange += WindowTypeChangeHandler;
            }
            catch (Exception ex)
            {
                log.Error(ex);
            }

        }


        private void WindowTypeChangeHandler(ChatWindowTypeChangeEventArgs e)
        {
            //fire the event to the parent form.
            if (WindowTypeChange != null)
            {
                WindowTypeChange(e);
            }
        }

        private void ClosedReasonStateHandler(long contactID)
        {
            if (this.ClosedReasonState != null)
            {
                ClosedReasonState(this);
            }
        }

        /// <summary>
        /// Handles the chat window closed event and fires the ChatClosed event to any listeners
        /// </summary>
        /// <param name="ChatWindow"></param>
        private void ChatWindowChatClosed(UcChatWindow ChatWindow)
        {
            if (this.ChatClosed != null)
            {
                ChatClosed(this);
            }
        }


        private void BtnBargeClick(object sender, RoutedEventArgs e)
        {

            log.Info("Barging of " + contact.ID);


            if (BargeEvent != null)
            {
                BargeEvent(this);
            }

            this.AddBargeWhisperWindow(Contact);
            AmmendConferenceGUI(true);
            ConsultWindow.btnPromote.Visibility = System.Windows.Visibility.Hidden;

            SendAgentOperation(ChatWindow.contactID.Value, BaseRequest.JSdestinationType.barge, ChatWindow.contactID.ToString());
        }

        /// <summary>
        /// Handles the conference complete button click event coming from the ChatWindow.
        /// Removes the consult window and fires the CompleteConferenceEvent.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnCompleteConferenceClick(object sender, RoutedEventArgs e)
        {
            try {
                log.Info("Completing Conference of " + contact.ID);

                //Get previous whisper messages
                FlowDocument fd = ConsultWindow.ChatArea;

                RemoveConsultWindow();
                ChatWindow.SetType(UcChatWindow.WindowType.CONFERENCE);
                if (CompleteConferenceEvent != null)
                {
                    CompleteConferenceEvent(this);
                }

                //Add new conference whisper window and enable active agent functionality
                this.AddConferenceWhisperWindow(contact);
                AmmendConferenceGUI(true);

                //Add previous whisper messages
                AddDocument(fd, ConsultWindow.ChatArea);

                //Send agent operation
                SendAgentOperation(ConsultWindow.contactID.Value, BaseRequest.JSdestinationType.conference, consultedAgent.ToString());
            }
            catch (Exception ex)
            {
                log.Error("Exception in complete conference click", ex);
            }
        }

        private void AddDocument(FlowDocument from, FlowDocument to)
        {
            try {
                log.Info("Copying chat box");
                TextRange range = new TextRange(from.ContentStart, from.ContentEnd);
                MemoryStream stream = new MemoryStream();
                System.Windows.Markup.XamlWriter.Save(range, stream);
                range.Save(stream, DataFormats.XamlPackage);
                TextRange range2 = new TextRange(to.ContentEnd, to.ContentEnd);
                range2.Load(stream, DataFormats.XamlPackage);
            }
            catch(Exception ex)
            {
                log.Error("Exception copying chat box");
            }
        }

        /// <summary>
        /// Handles the transfer button click event coming from the ChatWindow.
        /// Removes the consult window, ends the chat for the current agent and fires the CompleteTransferEvent.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnCompleteTransferClick(object sender, RoutedEventArgs e)
        {
            log.Info("Completing transfer of " +contact.ID);

            RemoveConsultWindow();
            ChatWindow.EndChat();
            LeaveReason = CDEntities.leaveReason.AGENT_TRANSFER;
            if (CompleteTransferEvent != null)
            {
                CompleteTransferEvent(this);
            }
            SendAgentOperation(ConsultWindow.contactID.Value, BaseRequest.JSdestinationType.agent, consultedAgent.ToString());
        }

        private void SendAgentOperation(long currentGuid, AgentOperationRequest.JSdestinationType destType, string destId)
        {
            try
            {
                AgentOperationRequest operationRequest = new AgentOperationRequest();
                operationRequest.apiVersion = AgentOperationRequest.JSapiVersion.one_point_zero;
                operationRequest.body.guid = currentGuid;
                operationRequest.body.destinationType = destType;
                operationRequest.body.destinationId = destId;

                Wsock.Send(operationRequest);
            }
            catch (Exception ex)
            {
                log.Error("An exception occured during the agent operation request", ex);
            }
        }

        /// <summary>
        /// Adds a consult chat window to an existing customer chat and displays them side by side in the tab.
        /// </summary>
        /// <param name="consultContact"></param>
        public void AddConsultWindow(Contact consultContact)
        {
            log.Info("Adding consult window for " + contact.ID);
            this.consultContact = consultContact;
            gContentGrid.ColumnDefinitions.Add(new ColumnDefinition() { MinWidth = 300});
            ConsultWindow = new UcChatWindow(ChatWindow.contactID.Value, UcChatWindow.WindowType.CONSULT_ORIGINATOR, joinRoomDetails);
            ConsultWindow.ShowTransferControls();
            Grid.SetColumn(ConsultWindow, CONSULT_WINDOW_COLUMN_INDEX);
            gContentGrid.Children.Add(ConsultWindow);
            ConsultWindow.btnClose.Click += BtnCloseConsultClick;
            ConsultWindow.btnCompleteConference.Click += BtnCompleteConferenceClick;
            ConsultWindow.btnCompleteTransfer.Click += BtnCompleteTransferClick;
        }

        /// <summary>
        /// Adds a conference whisper window to an existing customer chat and displays them side by side in the tab.
        /// </summary>
        /// <param name="consultContact"></param>
        public void AddConferenceWhisperWindow(Contact consultContact)
        {
            log.Info("Adding conference whisper window for " + contact.ID);
            this.consultContact = consultContact;
            gContentGrid.ColumnDefinitions.Add(new ColumnDefinition() { MinWidth = 300 });
            ConsultWindow = new UcChatWindow(ChatWindow.contactID.Value, UcChatWindow.WindowType.CONFERENCE_WHISPER, joinRoomDetails);

            Grid.SetColumn(ConsultWindow, CONSULT_WINDOW_COLUMN_INDEX);
            gContentGrid.Children.Add(ConsultWindow);
            ConsultWindow.btnDrop.Click += BtnDropClick;
            ConsultWindow.btnPromote.Click += BtnPromoteClick;
        }

        public void AddBargeWhisperWindow(Contact bargeContact)
        {
            log.Info("Adding barge whisper window for " + contact.ID);
            this.consultContact = bargeContact;
            gContentGrid.ColumnDefinitions.Add(new ColumnDefinition() { MinWidth = 300 });
            ConsultWindow = new UcChatWindow(ChatWindow.contactID.Value, UcChatWindow.WindowType.BARGE_WHISPER, joinRoomDetails);

            Grid.SetColumn(ConsultWindow, CONSULT_WINDOW_COLUMN_INDEX);
            gContentGrid.Children.Add(ConsultWindow);
            ConsultWindow.btnDrop.Click += BtnDropClick;
            ConsultWindow.btnPromote.Click += BtnPromoteClick;
        }

        public void BtnDropClick(object sender, RoutedEventArgs e)
        {
            log.Info("Dropping conference passive agent");

            AgentDropPassiveParticipantRequest dropPassive = new AgentDropPassiveParticipantRequest();
            dropPassive.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
            dropPassive.body.guid = Contact.ID;

            Wsock.Send(dropPassive);

            //Go back to a 1-1 chat
            RemoveConsultWindow();
            if (ActiveConfConsultClosed != null)
            {
                ActiveConfConsultClosed(this);
            }
            ChatWindow.SetType(UcChatWindow.WindowType.NORMAL);
        }

        public void BtnPromoteClick(object sender, RoutedEventArgs e)
        {
            log.Info("Promoting conference passive agent");

            AgentChangeConferenceOwnerRequest changeOwner = new AgentChangeConferenceOwnerRequest();
            changeOwner.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
            changeOwner.body.guid = Contact.ID;

            Wsock.Send(changeOwner);


        }

        public void BtnCloseHistoryClick(object sender, RoutedEventArgs e)
        {
            log.Info("History close button clicked");

            if (HistoryClosed != null)
            {
                HistoryClosed(this);
            }
        }

        public void BtnCloseConsultClick(object sender, RoutedEventArgs e)
        {
            log.Info("Destination consult close button clicked");

            //ToDo
            //(ChatWindow.Type == UcChatWindow.WindowType.CONSULT_RECEIVER)
            //ChatWindow.agentLeftChat

            if (!ConsultWindow.agentLeftChat && !ChatWindow.customerLeftChat)
            {
                log.Info("Other agent & customer has not left chat so sending AgentDropPassiveParticipantRequest");
                AgentDropPassiveParticipantRequest dropPassive = new AgentDropPassiveParticipantRequest();
                dropPassive.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
                dropPassive.body.guid = Contact.ID;

                Wsock.Send(dropPassive);
            }
            else
            {
                //Handle destination consult already having left the whisperchat
                log.Info("Other agent or customer has left chat so not sending LeaveRoomRequest");
            }

            //Close button clicked in a consult, so close the consult window and remove.
            RemoveConsultWindow();
            if (OriginatingConsultClosed != null)
            {
                OriginatingConsultClosed(this);
            }
        }

        public void BtnCloseConfWhisperClick(object sender, RoutedEventArgs e)
        {
            log.Info("Close conf whisper button clicked");

            RemoveConsultWindow();
            if (ActiveConfConsultClosed != null)
            {
                ActiveConfConsultClosed(this);
            }
        }

        public void BtnCloseBargeWhisperClick(object sender, RoutedEventArgs e)
        {
            log.Info("Close barge whisper button clicked");

            RemoveConsultWindow();
            if (BargeWhisperClosed != null)
            {
                BargeWhisperClosed(this);
            }
        }

        /// <summary>
        /// Handles the close button click event in (potentially either of) the ChatWindow.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        public void BtnCloseClick(object sender, RoutedEventArgs e)
        {
            log.Info("Click close chat button - LeaveReason:" + LeaveReason);
            //WS Quit Chat Request
            ChatWindow.DisableSendingNewMessages();

            if (ChatWindow.Type == UcChatWindow.WindowType.OBSERVE)
            {
                if (!ChatWindow.agentLeftChat && !ChatWindow.customerLeftChat)
                {
                    log.Info("Other agent has not left chat so sending LeaveRoomRequest - ConsultReceiver");
                    AgentLeaveRoomRequest leaveRoom = new AgentLeaveRoomRequest();
                    leaveRoom.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
                    leaveRoom.body.guid = Contact.ID;

                    Wsock.Send(leaveRoom);
                }

                //Close CCT conenction on observe
                //Remove window
                if (ObserveClosed != null)
                {
                    ObserveClosed(this);
                }
                return;
            }

            if (ChatWindow.Type == UcChatWindow.WindowType.BARGE)
            {
                if (!ChatWindow.agentLeftChat && !ChatWindow.customerLeftChat)
                {
                    SendQuitChatRequest();
                }
                else
                {
                    this.ChatWindow.ShowClosedReasonWindow();
                }
                RemoveConsultWindow();
                
                return;
            }

            //Need to check here if contact is still in chat
            if (!ChatWindow.customerLeftChat && ChatWindow.Type != UcChatWindow.WindowType.CONSULT_RECEIVER && ChatWindow.Type != UcChatWindow.WindowType.CONFERENCE)
            {
                if (ActiveConfClosed != null)
                {
                    ActiveConfClosed(this);

                }
                SendQuitChatRequest();
            }

            //Check if this is accurate - Closing chat with consult open????
            if (ConsultWindow != null && ConsultWindow.Visibility == Visibility.Visible && ChatWindow.Type != UcChatWindow.WindowType.CONFERENCE)
            {

                //ToDo
                //(ChatWindow.Type == UcChatWindow.WindowType.CONSULT_RECEIVER)
                //ChatWindow.agentLeftChat

                if (!ConsultWindow.agentLeftChat)
                {
                    //Handle closing of a chat with consult open
                    log.Info("Other agent has not left consult - ConsultWindow still open and closing customer chat");
                }
                else
                {
                    //Handle consult desination having already left the consult
                    log.Info("Think this shouldn't happen as when passive leaves consult the window is removed");
                    log.Info("Maybe case when he has left but originating hasn't closed their consult window yet");
                    log.Info("Other agent has left chat so not sending LeaveRoomRequest - ConsultWindow still open and closing customer chat?");
                }

                //Close button clicked in a consult, so close the consult window and remove.
                //Remove window
                RemoveConsultWindow();


                if (OriginatingConsultClosed != null)
                {
                    OriginatingConsultClosed(this);
                    
                }
            }

            if (ChatWindow.Type == UcChatWindow.WindowType.CONFERENCE)
            {
                if (ChatWindow.ConferenceOwner)
                {
                    EndConferenceAsActive();
                    //Handle going to closed reason codes...
                }
                if (!ChatWindow.ConferenceOwner)
                {
                    EndConferenceAsPassive();
                }

            }

            else if (ChatWindow.Type == UcChatWindow.WindowType.CONSULT_RECEIVER)
            {

                if (ChatWindow.IsBarge && !ChatWindow.SupervisorLeftChat && !ChatWindow.consultReceiverBeforeBarge)
                {
                    log.Info("Chat is a barge and supervisor hasn't left yet so sending leave room");
                    AgentLeaveRoomRequest leaveRoom = new AgentLeaveRoomRequest();
                    leaveRoom.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
                    leaveRoom.body.guid = Contact.ID;
                    Wsock.Send(leaveRoom);

                }
                else if ( (!ChatWindow.agentLeftChat || (!ChatWindow.SupervisorLeftChat && ChatWindow.agentLeftChat))  && !ChatWindow.customerLeftChat)
                {
                    log.Info("Other agent has not left chat so sending LeaveRoomRequest - ConsultReceiver");
                    AgentLeaveRoomRequest leaveRoom = new AgentLeaveRoomRequest();
                    leaveRoom.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
                    leaveRoom.body.guid = Contact.ParentID;

                    Wsock.Send(leaveRoom);
                }
                else
                {
                    //Handle consult originator having already left the consult
                    log.Info("Other agent has left chat so not sending LeaveRoomRequest - ConsultReceiver");
                }


                LeaveReason = CDEntities.leaveReason.AGENT_CONSULT;
                if (NonOriginatingConsultClosed != null)
                {
                    NonOriginatingConsultClosed(this);
                }
            }
            else
            {
                LeaveReason = CDEntities.leaveReason.CLOSING_CONTACT;
                //Normal operation, just end the chat.

                //Removed this as waiting for notification with closed reason codes
                //this.ChatWindow.EndChat(); --> Called GetClosedReasonCodesComplete

                //Already have codes fro customer disconnect - Dont need to wait for agent to get them
                if (ChatWindow.customerLeftChat)
                {
                    this.ChatWindow.ShowClosedReasonWindow();
                }
            }
        }

        private void EndConferenceAsActive()
        {
            RemoveConsultWindow();
            // TODO Send notification
            // Tear down conference as the active
            if (!ChatWindow.customerLeftChat)
            {
                AgentConferenceCloseRequest closeConference = new AgentConferenceCloseRequest();
                closeConference.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
                closeConference.body.guid = Contact.ID;

                Wsock.Send(closeConference);
            }
            else
            {
                this.ChatWindow.ShowClosedReasonWindow();
            }

            if (ActiveConfClosed != null)
            {
                ActiveConfClosed(this);
            }
        }

        private void EndConferenceAsPassive()
        {
            // TODO Send notification
            if (!ConsultWindow.agentLeftChat && !ChatWindow.customerLeftChat)
            {
                // Leave conference as a passive
                AgentLeaveRoomRequest leaveRoom = new AgentLeaveRoomRequest();
                leaveRoom.apiVersion = BaseRequest.JSapiVersion.one_point_zero;
                leaveRoom.body.guid = Contact.ID;

                Wsock.Send(leaveRoom);
            }
            
            if (PassiveConfClosed != null)
            {
                PassiveConfClosed(this);
            }

            
        }

        public void AmmendConferenceGUI(bool isOwner)
        {
            ConsultWindow.ShowConferenceControls(isOwner);
            ChatWindow.ConferenceOwner = isOwner;
        }

        public void SetAsConferenceOwner(bool isOwner)
        {
            if (isOwner)
            {
                log.Info("Writing that agent is now Active conference owner to chat window");
                ConsultWindow.AppendAsSystemInfo("ConferenceOwner", "You have been promoted to conference owner");
            }
            else
            {
                log.Info("Writing that agent is now Passive conference owner to chat window");
                ConsultWindow.AppendAsSystemInfo("ConferenceOwner", "You are no longer the conference owner");
            }
            AmmendConferenceGUI(isOwner);
        }
        
        private void SendQuitChatRequest(){
            AgentQuitChatRequest quitChat = new AgentQuitChatRequest();
            quitChat.apiVersion = AgentQuitChatRequest.JSapiVersion.one_point_zero;
            quitChat.body.guid = this.contact.ID;

            Wsock.Send(quitChat);
            log.Info("Sent QuitChatRequest for contact ID - " + this.contact.ID.ToString());
        }

        /// <summary>
        /// Removes the consult window from the tab.
        /// </summary>
        public void RemoveConsultWindow()
        {
            if (ConsultWindow.Visibility != Visibility.Collapsed)
            {
                //Agent consulted on this contact, so close the consult window
                ConsultWindow.Visibility = Visibility.Collapsed;
                //remove the grid entry for the consult window
                gContentGrid.ColumnDefinitions.RemoveAt(CONSULT_WINDOW_COLUMN_INDEX);
            }
        }

        /// <summary>
        /// Called when a conference has been completed. Sets the contact and window type.
        /// </summary>
        /// <param name="contact"></param>
        public void ConferenceComplete(CustomDesktop.Contact contact)
        {
            this.Contact = contact;
            MainWindow.chatrooms.Add(Contact.ID, this);
            MainWindow.chatrooms.Remove(MainWindow.consulting[Contact.ID]);
            ChatWindow.SetType(UcChatWindow.WindowType.CONFERENCE);
            this.AddConferenceWhisperWindow(Contact);
        }

        /// <summary>
        /// Called when a transfer has been completed. Sets the contact and window type.
        /// </summary>
        /// <param name="contact"></param>
        public void TransferComplete(CustomDesktop.Contact contact)
        {
            this.Contact = contact;
            ChatWindow.SetType(UcChatWindow.WindowType.NORMAL);
        }
    }
}
