﻿//////////////////////////////////////////////////////////////////////////////
// WsockEvent.cs
//
// Copyright 2015 Avaya Inc. All Rights Reserved.
//
// Usage of this source is bound to the terms described in
// Licenses/License.txt
//
// Avaya - Confidential & Proprietary. Use pursuant to your signed agreement or
// Avaya Policy
//////////////////////////////////////////////////////////////////////////////


using EncoderDecoder.Notifications;
using System;
using System.Collections.Generic;

namespace EncoderDecoder
{
    /// <summary>
    /// Web Socket events to subscribe to which get fired when an appropriate websocket notification is received
    /// </summary>
    public class WsockEvent
    {
        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);

        // Dictionary to map functions to each event
        private Dictionary<string, Action<NotificationEvent>> Events;

        public WsockEvent()
        {
            PopulateEventsDictionary();
        }

        #region EVENTS

        /// <summary>
        /// Event to handle Reconnects
        /// </summary>
        public static event ReconnectHandler Reconnect;
        public delegate void ReconnectHandler(bool firstPassThrough);

        /// <summary>
        /// Event to handle Acknowledgement Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AcknowledgementNotification;

        /// <summary>
        /// Event to handle Agent Active Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentActiveNotification;

        /// <summary>
        /// Event to handle Agent Change Conference Owner Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentChangeConferenceOwnerNotification;

        /// <summary>
        /// Event to handle Agent Complete Conference Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentCompleteConferenceNotification;

        /// <summary>
        /// Event to handle Agent Exit Room Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentExitRoomNotification;

        /// <summary>
        /// Event to handle Agent Get Observed Intrinsics Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentGetObservedIntrinsicsNotification;

        /// <summary>
        /// Event to handle Agent Join Room Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentJoinRoomNotification;

        /// <summary>
        /// Event to handle Agent Login Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentLoginNotification;

        /// <summary>
        /// Event to handle Agent Operation Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentOperationNotification;

        /// <summary>
        /// Event to handle Agent Quit Chat Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentQuitChatNotification;

        /// <summary>
        /// Event to handle Agent Set Closed Reason Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentSetClosedReasonNotification;

        /// <summary>
        /// Event to handle Agent Whisper Message Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentWhisperMessageNotification;

        /// <summary>
        /// Event to handle Customer Disconnect Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> CustomerDisconnectNotification;

        /// <summary>
        /// Event to handle Customer Connection Status Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> CustomerConnectionStatusNotification;

        /// <summary>
        /// Event to handle Error Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> ErrorNotification;

        /// <summary>
        /// Event to handle Get Destinations Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> GetDestinationsNotification;

        /// <summary>
        /// Event to handle Is Typing Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> IsTypingNotification;

        /// <summary>
        /// Event to handle New Chat Message Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> NewMessageNotification;

        /// <summary>
        /// Event to handle New Welcome Message Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> NewWelcomeMessageNotification;
        /// <summary>
        /// Event to handle New Comfort Message Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> NewComfortMessageNotification;
        /// <summary>
        /// Event to handle New Whisper Message Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> NewWhisperMessageNotification;
        /// <summary>
        /// Event to handle Read Customer History Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> ReadCustomerHistoryNotification;
        /// <summary>
        /// Event to handle Agent Data Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> AgentDataNotification;

        /// <summary>
        /// Event to handle New Page Push Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> NewPagePushNotification;

        /// <summary>
        /// Event to handle Supervisor Barge Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> SupervisorBargeNotification;

        /// <summary>
        /// Event to handle Screenpops Notifications
        /// </summary>
        public static event EventHandler<NotificationEvent> ScreenPopsNotification;

        private void PopulateEventsDictionary()
        {
            Events = new Dictionary<string, Action<NotificationEvent>>(){
                { "AcknowledgementNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AcknowledgementNotification) },
                { "AgentActiveNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentActiveNotification) },
                { "AgentChangeConferenceOwnerNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentChangeConferenceOwnerNotification) },
                { "AgentCompleteConferenceNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentCompleteConferenceNotification) },
                { "AgentExitRoomNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentExitRoomNotification) },
                { "AgentGetObservedIntrinsicsNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentGetObservedIntrinsicsNotification) },
                { "AgentJoinRoomNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentJoinRoomNotification) },
                { "AgentLoginNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentLoginNotification) },
                { "AgentOperationNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentOperationNotification) },
                { "AgentQuitChatNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentQuitChatNotification) },
                { "AgentSetClosedReasonNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentSetClosedReasonNotification) },
                { "AgentWhisperMessageNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentWhisperMessageNotification) },
                { "CustomerDisconnectNotification", (NotificationEvent NE) => CreateSingleEvent(NE, CustomerDisconnectNotification) },
                { "CustomerConnectionStatusNotification", (NotificationEvent NE) => CreateSingleEvent(NE, CustomerConnectionStatusNotification) },
                { "ErrorNotification", (NotificationEvent NE) => CreateSingleEvent(NE, ErrorNotification) },
                { "GetDestinationsNotification", (NotificationEvent NE) => CreateSingleEvent(NE, GetDestinationsNotification) },
                { "IsTypingNotification", (NotificationEvent NE) => CreateSingleEvent(NE, IsTypingNotification) },
                { "NewMessageNotification", (NotificationEvent NE) => CreateSingleEvent(NE, NewMessageNotification) },
                { "ReadCustomerHistoryNotification", (NotificationEvent NE) => CreateSingleEvent(NE, ReadCustomerHistoryNotification) },
                { "AgentDataNotification", (NotificationEvent NE) => CreateSingleEvent(NE, AgentDataNotification) },
                { "NewPagePushNotification", (NotificationEvent NE) => CreateSingleEvent(NE, NewPagePushNotification) },
                { "SupervisorBargeNotification", (NotificationEvent NE) => CreateSingleEvent(NE, SupervisorBargeNotification) },
                { "ScreenPopsNotification", (NotificationEvent NE) => CreateSingleEvent(NE, ScreenPopsNotification) }
            };
        }

        #endregion EVENTS

        /// <summary>
        /// Used to fire an event of the relevant type
        /// </summary>
        /// <param name="notificationEvent"></param>
        public void CreateEvent(NotificationEvent notificationEvent)
        {
            //Search dictionary for event to fire
            Events[notificationEvent.Notification.GetType().Name](notificationEvent);
        }

        public void TryReconnect()
        {
            if (Reconnect != null)
            {
                Wsock.isReconnecting = true;
                Reconnect(true);
            }
        }

        /// <summary>
        /// Fires the correct event
        /// </summary>
        /// <param name="notificationEvent"></param>
        /// <param name="eventName"></param>
        private void CreateSingleEvent(NotificationEvent notificationEvent, EventHandler<NotificationEvent> eventName)
        {

            if (notificationEvent.Notification.GetType().Name.Equals("NewMessageNotification"))
            {
                NewMessageNotification notification = notificationEvent.Notification as NewMessageNotification;
                ProcessNewMessageEvents(notification, notificationEvent);
            }
            else
            {
                EventHandler<NotificationEvent> handler = eventName;
                if (null != handler)
                {
                    if (!notificationEvent.Notification.GetType().Name.Equals("AgentDataNotification"))
                    {
                        log.Info(" *** Firing new event: " + notificationEvent.Notification.GetType().Name);
                    }
                    handler(this, notificationEvent);
                }
            }

        }

        private void ProcessNewMessageEvents(NewMessageNotification notification, NotificationEvent notificationEvent)
        {
            switch (notification.body.messageType.ToUpper())
            {
                case "WELCOMEMESSAGE":
                    ProcessSpecificNewMessage(notificationEvent, NewWelcomeMessageNotification);
                    break;
                case "COMFORTMESSAGE":
                    ProcessSpecificNewMessage(notificationEvent, NewComfortMessageNotification);
                    break;
                case "WHISPER":
                    ProcessSpecificNewMessage(notificationEvent, NewWhisperMessageNotification);
                    break;
                case "NORMAL":
                    ProcessSpecificNewMessage(notificationEvent, NewMessageNotification);
                    break;
                default:
                    log.Error("Unknown new message type");
                    break;
            }
        }

        private void ProcessSpecificNewMessage(NotificationEvent notificationEvent, EventHandler<NotificationEvent> notificationEventHandler)
        {
            EventHandler<NotificationEvent> nmhandler = notificationEventHandler;
            if (null != nmhandler)
            {
                log.Info(" *** Firing new event: " + notificationEventHandler.ToString());
                nmhandler(this, notificationEvent);
            }
        }
    }
}