﻿//////////////////////////////////////////////////////////////////////////////
// LoginControl.xaml.cs
//
// Copyright © 2008-2014 Avaya Inc. All rights reserved.
// See: www.avaya.com
//////////////////////////////////////////////////////////////////////////////
using EncoderDecoder;
using EncoderDecoder.Notifications;
using EncoderDecoder.Requests;
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Threading;
using WebSocketSharp;

namespace CustomDesktop
{
    /// <summary>
    /// Handles gathering the login details of an agent
    /// </summary>
    public partial class LoginControl : UserControl
    {
        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);
        private ICCTConnector cctConnection;
        public ICCTConnector CCTConnection { get { return cctConnection; } set { cctConnection = value; } }
        public delegate void LoggingInEvent();
        public event LoggingInEvent LoggingInEventHandler;
        public delegate void LogInCompleted(ResultCode result);
        public event LogInCompleted LogInCompletedEventHandler;


        private AgentLoginRequest reconnection;
        private bool IsReconnection = false;
        public delegate void mmConnectionState(bool state);
        public event mmConnectionState MMConnectionStateChanged;


        /// <summary>
        /// Default Constructor.
        /// </summary>
        public LoginControl()
        {
            InitializeComponent();
            this.tbDomain.Text = Globals.DefaultCCTDomain;
            this.tbServerName.Text = Globals.DefaultCCTServer;
            this.tbUserID.Text = Globals.DefaultCCTUser;

            //WS Stuff
            WsockEvent.AgentLoginNotification += WsAgentLoginHandler;
            WsockEvent.ErrorNotification += WsErrorHandler;
            WsockEvent.Reconnect += WsReconnect;
        }

        

        public string GetUsername()
        {
            return tbUserID.Text;
        }

        public string GetDomain()
        {
            return tbDomain.Text;
        }


        public void LogIn(object sender, DoWorkEventArgs e)
        {
            Wsock.loggedIn = false;

            ResultCode result = new ResultCode();
            try
            {
                result = cctConnection.Connect((LoginDetails)e.Argument);
                if (result.ResultType == ResultCode.ResultCodeType.CCT_SUCCESS)
                {
                    log.Info("CCT login was successful, logging in to Multimedia WS-" + Globals.DefaultWSURL);

                    //Create/Open WebSocket 
                    Wsock ws = new Wsock(Globals.DefaultWSURL);
                    Wsock.agentId = Globals.agentID.ToString();
                    ResultCode r2 = new ResultCode();

                    //Check websocket is open before sending login details
                    Stopwatch s = new Stopwatch();
                    s.Start();
                    while (s.Elapsed < TimeSpan.FromSeconds(10) && s.IsRunning)
                    {
                        if (Wsock.isOpen)
                        {
                            log.Info("WebSocket has been opened");

                            //Log into email
                            EmailManager em = new EmailManager();
                            em.EmailLogon(Globals.agentID.ToString(), Globals.CCMMpassword);

                            //Encode the login request to JSON and send down websocket
                            AgentLoginRequest agentLoginRequest = CreateMMLoginObject(Globals.agentID.ToString(), Globals.CCMMpassword, AgentLoginRequest.JSclientType.custom);
                            reconnection = agentLoginRequest;
                            Wsock.Send(agentLoginRequest);
                            ws.StartWebsocketStatusTimedWork();
                            s.Stop();
                        }
                    }
                    s.Stop();

                    //Check if the websocket is not yet open and close out of CCT
                    if (!Wsock.isOpen)
                    {
                        
                        r2.ResultType = ResultCode.ResultCodeType.MM_COULD_NOT_OPEN_WEBSOCKET;
                        result = r2;
                        log.Error("WebSocket could not be opened");

                    }

                    //Check outcome of MM login
                    s.Start();
                    while (s.Elapsed < TimeSpan.FromSeconds(10) && s.IsRunning)
                    {
                        if (Wsock.loggedIn)
                        {
                            log.Info("Logged into MM");

                            //temp changes
                            r2.ResultType = ResultCode.ResultCodeType.SMF_SUCCESS;
                            result = r2;
                            s.Stop();
                        }

                    }
                    s.Stop();

                    //Check if the MM login was not a success and close out of CCT
                    if (Wsock.isOpen && !Wsock.loggedIn)
                    {
                        r2.ResultType = ResultCode.ResultCodeType.MM_LOGIN_ERROR;
                        result = r2;
                        Wsock.CloseWebSocket(CloseStatusCode.IncorrectData, "Login to MM failed");
                        log.Error("Could not log in to MM");

                    }
                }
            }
            catch (Exception ex)
            {
                log.Error("Exception occured logging in", ex);
                result.ResultType = ResultCode.ResultCodeType.GENERAL_ERROR;
                result.Message = ex.ToString();
            }
            e.Result = result;
        }

        private void WsErrorHandler(object s, NotificationEvent e)
        {

            var messageEvent = e.Notification as ErrorNotification;
            if (messageEvent != null)
            {
                try
                {

                    Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                    {
                        MessageBox.Show(messageEvent.body.code + " " + messageEvent.body.errorMessage, "Error", MessageBoxButton.OK, MessageBoxImage.Error);
                    }));

                }

                catch (Exception ex)
                {
                    log.Error("Could not decode new login notification " + messageEvent.ToString(), ex);
                }
            }
        }

        private void WsReconnect(bool firstPassThrough)
        {

            try
            {

                log.Info("Attempting to reconnect");

                //Update GUI
                MMConnectionStateChanged(false);

                if (firstPassThrough)
                {
                    Dispatcher.BeginInvoke(DispatcherPriority.Normal, (Action)(() =>
                    {
                        MessageBox.Show("WebSocket went down - Attempting to reconnect", "Error", MessageBoxButton.OK, MessageBoxImage.Error);
                    }));
                }

                //Disable sending
                IsReconnection = true;
                Wsock.isOpen = false;

                bool stillReconnecting = false;
                int numOfRetries = 1;
                while (!stillReconnecting)
                {
                    log.Info("Retry num - " + numOfRetries);
                    if (numOfRetries < 50)
                    {
                        stillReconnecting = DoReconnection();
                        numOfRetries++;
                    }
                    else if (MessageBox.Show("Websocket could now be reconnected - Do you want to try again?", "Error", MessageBoxButton.YesNo) == MessageBoxResult.Yes)
                    {
                        log.Info("--Retrying due to user selection dialog");
                        WsReconnect(false);
                    }
                    else
                    {
                        log.Info("--Not Retrying - Exiting");
                        //Exit the program
                        cctConnection.Disconnect();
                        Environment.Exit(0);
                    }

                }

                log.Info("Finished reconnection");
            }

            catch (Exception ex)
            {
                log.Error("Could not reconnect", ex);
            }

        }

        private bool DoReconnection()
        {
            //Open new wsock
            log.Info("Creating new websocket to - " + Globals.DefaultWSURL);
            Wsock ws = new Wsock(Globals.DefaultWSURL);

            //Check websocket is open before sending login details
            Stopwatch s = new Stopwatch();
            s.Start();
            while (s.Elapsed < TimeSpan.FromSeconds(5) && s.IsRunning)
            {
                if (Wsock.isOpen)
                {
                    log.Info("Websocket has been opened in DoReconnection");

                    //Send reconnect
                    reconnection.body.reconnect = true;
                    Wsock.Send(reconnection);
                    s.Stop();
                    ws.StartWebsocketStatusTimedWork();
                    return true;
                }
            }
            s.Stop();
            log.Info("Websocket has not been opened in DoReconnection");
            Wsock.isOpen = false;
            ws.StopWebsocketStatusTimedWork();
            ws = null;
            return false;
        }

        private void WsAgentLoginHandler(object s, NotificationEvent e)
        {

            var messageEvent = e.Notification as AgentLoginNotification;
                if (messageEvent != null)
                {
                    try
                    {
                        if (IsReconnection)
                        {
                            log.Info("Setting reconnecting flag back to false");
                            Wsock.isReconnecting = false;

                            if (messageEvent.body.loginSuccess)
                            {
                                Wsock.loggedIn = true;
                                log.Info("Received reconnection login notification back and was a success");
                                IsReconnection = false;
                                MMConnectionStateChanged(true);
                            }
                            else
                            {
                                Wsock.loggedIn = false;
                                log.Error("Received reconnection login notification back and was a failure");
                            }
                        }
                        else
                        {

                            if (messageEvent.body.loginSuccess)
                            {
                                Wsock.loggedIn = true;
                                log.Info("Received login notification back and was a success");
                            }
                            else
                            {
                                Wsock.loggedIn = false;
                                log.Error("Received login notification back and was a failure");
                            }
                        }

                    }

                    catch (Exception ex)
                    {
                        log.Error("Could not decode new login notification " + messageEvent.ToString(), ex);
                    }
                }
        }

        /// <summary>
        /// Handles the completion of the CCT login worker thread.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        public void OnLogIn(object sender, RunWorkerCompletedEventArgs e)
        {
            if (LogInCompletedEventHandler != null && e.Result != null)
            {
                if (((ResultCode)e.Result).ResultType == ResultCode.ResultCodeType.SMF_SUCCESS)
                {
                    //Remove the password and object as has reference to password
                    tbPassword.Password = "";
                    tbPasswordCCMM.Password = "";
                }
                LogInCompletedEventHandler((ResultCode)e.Result);
            }
            else
            {
                //nothing listening, so disconnect unused session
                cctConnection.Disconnect();
            }
        }

        #region GUI Event handlers

        /// <summary>
        /// Event handler for the submit button. Carries out basic validation and then attempts login.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtnSubmitClick(object sender, RoutedEventArgs e)
        {
            LoginAsync();
        }

        private bool CheckLoginDetails()
        {
            if (String.IsNullOrEmpty(tbDomain.Text))
            {
                MessageBox.Show("Please enter a domain");
                return false;
            }
            else if (String.IsNullOrEmpty(tbPassword.Password))
            {
                MessageBox.Show("Please enter a password");
                return false;
            }
            else if (String.IsNullOrEmpty(tbUserID.Text))
            {
                MessageBox.Show("Please enter a user ID");
                return false;
            }
            else if (String.IsNullOrEmpty(tbPasswordCCMM.Password) && tbPasswordCCMM.IsEnabled)
            {
                MessageBox.Show("Please enter a CCMM password");
                return false;
            }

            return true;
        }

        private bool CheckServerDetails()
        {
            if (String.IsNullOrEmpty(tbPort.Text))
            {
                MessageBox.Show("Please enter a port");
                return false;
            }
            else if (String.IsNullOrEmpty(tbServerName.Text))
            {
                MessageBox.Show("Please enter a server name or IP");
                return false;
            }
            return true;
        }

        public void LoginAsync()
        {
            LoginDetails credentials = new LoginDetails();
            //check server connection details
            if (!CheckServerDetails())
            {
                return;
            }

            //check credentials
            //If Use current user is checked, no need to validate other settings.
            if (!cbCurrUser.IsChecked.Value)
            {
                if (!CheckLoginDetails())
                {
                    return;
                }
            }

            //Set the credentials in to the login credentials object
            credentials.server = tbServerName.Text;
            credentials.port = tbPort.Text;
            credentials.useCurrentUser = cbCurrUser.IsChecked.Value;
            credentials.user = tbUserID.Text;
            credentials.password = tbPassword.Password;
            credentials.domain = tbDomain.Text;
            if (tbPasswordCCMM.IsEnabled)
            {
                Globals.CCMMpassword = tbPasswordCCMM.Password;
            }
            else if (Globals.CCMMpassword != null)
            {
                Globals.CCMMpassword = Globals.agentID.ToString();
            }
            //Log in to cct asynchronously as it can take some time.
            BackgroundWorker backgroundWorker = new BackgroundWorker();

            backgroundWorker.DoWork += LogIn;
            backgroundWorker.RunWorkerCompleted += OnLogIn;
            backgroundWorker.RunWorkerAsync(credentials);

            if (LoggingInEventHandler != null)
            {
                LoggingInEventHandler();
            }
            //Hide this control while logging in. If the login fails, then the control will be changed back to visible later.
            this.Visibility = Visibility.Hidden;
            log.Info(string.Format("Logging in to CCT with {0}", credentials));
        }

        // Create the MM login object to send to Agent Controller
        private AgentLoginRequest CreateMMLoginObject(string agentUsername, string agentPass, AgentLoginRequest.JSclientType type)
        {
            try
            {
                AgentLoginRequest request = new AgentLoginRequest();
                request.apiVersion = AgentLoginRequest.JSapiVersion.one_point_zero;
                request.body.agentId = agentUsername;
                request.body.password = agentPass;
                request.body.clientType = type;
                return request;
            }
            catch(Exception ex)
            {
                log.Error("Exception occured creating AgentLoginRequest", ex);
                return null;
            }
        }

        /// <summary>
        /// Event handler for the use current user checkbox
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void CbCurrUserChecked(object sender, RoutedEventArgs e)
        {
            tbUserID.IsEnabled = false;
            tbDomain.IsEnabled = false;
            tbPassword.IsEnabled = false;
        }

        /// <summary>
        /// Event handler for the CCMM password checkbox
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void CbCurrUserUnchecked(object sender, RoutedEventArgs e)
        {
            tbUserID.IsEnabled = true;
            tbDomain.IsEnabled = true;
            tbPassword.IsEnabled = true;
        }

        /// <summary>
        /// Event handler for the CCMM password checkbox
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void CbDefaultCCMMChecked(object sender, RoutedEventArgs e)
        {
            tbPasswordCCMM.IsEnabled = false;
            tbPasswordCCMM.Password = "";
        }

        /// <summary>
        /// Event handler for the use current user checkbox
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void CbDefaultCCMMUnchecked(object sender, RoutedEventArgs e)
        {
            tbPasswordCCMM.IsEnabled = true;
        }

        /// <summary>
        /// Submits the form if a return key is detected in the password box
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnKeyDownHandler(object sender, KeyEventArgs e)
        {
            if (e.Key == Key.Return)
            {
                BtnSubmitClick(sender, new RoutedEventArgs());
            }
        }
        #endregion

        /// <summary>
        /// Simple struct for holding login info
        /// </summary>
        public struct LoginDetails
        {
            public String server;
            public String port;
            public bool useCurrentUser;
            public String user;
            public String password;
            public String domain;

            public override String ToString()
            {
                if (useCurrentUser)
                {
                    return string.Format("Credentials for currently logged in user on {0}:{1}", server, port);
                }
                else
                {
                    return string.Format("Credentials for {2}\\{3} on {0}:{1}", server, port, domain, user);
                }
            }
        }
    }
}
