﻿//////////////////////////////////////////////////////////////////////////////
// Wsock.cs
//
// Copyright 2015 Avaya Inc. All Rights Reserved.
//
// Usage of this source is bound to the terms described in
// Licenses/License.txt
//
// Avaya - Confidential & Proprietary. Use pursuant to your signed agreement or
// Avaya Policy
//////////////////////////////////////////////////////////////////////////////


using EncoderDecoder.Requests;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System;
using System.Timers;
using WebSocketSharp;

namespace EncoderDecoder
{
    public class Wsock
    {
        // Create a logger for use in this class
        private static readonly log4net.ILog log = log4net.LogManager.GetLogger(System.Reflection.MethodBase.GetCurrentMethod().DeclaringType);

        static WebSocket ws;
        JDecoder jdecode;
        public static bool isOpen;
        public static bool loggedIn;
        public static bool isReconnecting;
        WsockEvent wsockEvent;
        public static string agentId;

        public delegate void mmConnectionState(bool state);
        public static event mmConnectionState MMConnectionStateChanged;
        
        public Wsock(Uri URL)
        {
            jdecode = new JDecoder();

            ws = new WebSocket(URL.ToString());
            ws.SslConfiguration.EnabledSslProtocols = System.Security.Authentication.SslProtocols.Tls11 | System.Security.Authentication.SslProtocols.Tls12;
            ws.Log.Level = LogLevel.Info;
            ws.Log.File = @"C:\Logs\ws.log";
            ws.OnOpen += OnOpen;
            ws.OnMessage += OnMessage;
            ws.OnError += OnError;
            ws.OnClose += OnClose;
            ws.ConnectAsync();

            TimeSpan ts = new TimeSpan(0, 0, 2);
            ws.WaitTime = ts;

            wsockEvent = new WsockEvent();

        }


        public void OnOpen(Object sender, EventArgs e)
        {
            isOpen = true;

            log.Info("Websocket Connection Opened");
        }

        
        public void OnClose(Object sender, CloseEventArgs e)
        {
            log.Info("Websocket Connection Closed:");
            log.Info(" > Reason: " + e.Reason);
            log.Info(" > Code: " + e.Code);
            log.Info(" > WasClean: " + e.WasClean);

            StopWebsocketStatusTimedWork();

            if (e.Reason.Contains("An exception has occurred while connecting"))
            {
                log.Debug("Reconnecting error - An exception has occurred while connecting");
                return;
            }
            if (e.Code != 1000 && loggedIn)
            {
                log.Info("Reconnecting due to non clean close and previously logged in");
                if (!isReconnecting)
                {
                    log.Info("Firing reconnecting event");
                    wsockEvent.TryReconnect();
                }
                else
                {
                    log.Info("Not firing event as already reconnecting");
                }

            }
            if (e.Code != 1000 && !loggedIn)
            {
                log.Info("Not reconnecting due to non clean close but previously not logged in");
            }
        }

        public void OnError(Object sender, ErrorEventArgs e)
        {
            StopWebsocketStatusTimedWork();

            if (e.Message.Contains("An exception has occurred while connecting"))
            {
                log.Debug("Reconnecting error - An exception has occurred while connecting");
                return;
            }

            log.Info("Websocket Connection Errored:");
            log.Info(" > Message: " + e.Message);
            log.Info(" > Exception: " + e.Exception);
        }

        public void OnMessage(Object sender, MessageEventArgs e)
        {
            //TEMP
            JObject json = JObject.Parse(e.Data);
            string formatted = json.ToString(Formatting.Indented);
            
            NotificationEvent ne = new NotificationEvent();
            ne.Notification = jdecode.Decode(e.Data);
            if (!ne.Notification.GetType().Name.Equals("AgentDataNotification"))
            {
                ws.Log.Info(" <<< WebSocket Message Received >>> " + agentId);
                ws.Log.Info(formatted);
                log.Info(" <<< WebSocket Message Received:" + e.Data);
                log.Info(" *** Type is:" + ne.Notification.GetType().Name);
            }
            

            //FireEvent - look up map for classname and then call correct event handler
            wsockEvent.CreateEvent(ne);

        }

        private static void Send(String msg)
        {
            ws.Send(msg);

            //TEMP
            JObject json = JObject.Parse(msg);
            string formatted = json.ToString(Formatting.Indented);
            if(msg.Contains("agentData")){
                return;
            }
            ws.Log.Info(" <<< WebSocket Message Sent >>> " + agentId);
            ws.Log.Info(formatted);

            log.Info(" >>> WebSocket Message Sent:" + msg);
        }

        public static void Send(IBaseRequest requestObject)
        {
            JEncoder jencode = new JEncoder();
            String json = jencode.Encode(requestObject);
            Send(json);
        }

        public static void SendSensitiveData(String msg, IBaseRequest objectToClean, String replaceWith)
        {
            //Send the data
            ws.Send(msg);
            String sensitive = "";

            AgentLoginRequest request = objectToClean as AgentLoginRequest;

            //Clean before logging
            if (request!=null)
            {
                request.body.password = replaceWith;
                //Encode
                JEncoder jencode = new JEncoder();
                sensitive = jencode.Encode(objectToClean);
            }

            //Log
            log.Info(" >>> WebSocket Message Sent:" + sensitive);
        }

        public static void CloseWebSocket(CloseStatusCode code, String reason)
        {
            isOpen = false;
            loggedIn = false;
            ws.Close(code, reason);
            log.Info("Websocket Connection Closed Expicitly:");
            log.Info(" > Reason: " + reason);
            log.Info(" > Code: " + code);
        }

        public static bool CheckConnectionAlive()
        {
            bool result = false;
            result = ws.Ping();
            return result;
        }

        private Timer websocketStatusTimer;

        public void StartWebsocketStatusTimedWork()
        {
            log.Info("Starting subscription to websocket status check");
            try
            {

                    websocketStatusTimer = new Timer(3000);
                    websocketStatusTimer.Enabled = true;
                    websocketStatusTimer.Elapsed += new ElapsedEventHandler(WebsocketStatusTimerTick);
                    websocketStatusTimer.Start();
                    log.Info("Finished subscription to websocket status check");
                
            }
            catch (Exception ex)
            {
                log.Error("Exception in StartWebsocketStatusTimedWork",ex);
            }
        }

        public void StopWebsocketStatusTimedWork()
        {
            
            if (websocketStatusTimer != null && websocketStatusTimer.Enabled)
            {
                log.Info("Stopping subscription to websocket status check");
                websocketStatusTimer.Stop();
            }
            
        }

        private void WebsocketStatusTimerTick(object sender, ElapsedEventArgs e)
        {
            try
            {
                lock (websocketStatusTimer)
                {
                    if (this.websocketStatusTimer.Enabled)
                    {
                        this.websocketStatusTimer.Stop();
                        CheckConnectionAlive();
                        this.websocketStatusTimer.Start();
                    }
                }
            }
            catch (Exception ex)
            {
                log.Error("An exception occured in WebsocketStatusTimerTick", ex);
            }

        }
    }
}
