﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.Security.Authentication;
using System.Net.Security;
using System.Collections;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Xml;
using System.Xml.Linq;

namespace LocationAPIExampleApp
{
    public enum FrameType { Continuation = 0x00, Text = 0x01, Binary = 0x02, Close = 0x08, Ping = 0x09, Pong = 0x0A }
    public enum ErrorCodes { Normal = 1000, Shutdown = 1001, ProtocolError = 1002, DataError = 1003, FrameTooLarge = 1004, NoStatus = 1005, CloseError = 1006, UTF8Error = 1007 }

    public partial class FrmMainForm : Form
    {
        WebSocket fConnection = null;
        X509Certificate2 certificate = null;

        public FrmMainForm()
        {
            InitializeComponent();
        }

        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            FrmAboutForm about = new FrmAboutForm();
            about.ShowDialog(this);
        }
        
        private void ConnectBtn_Click(object sender, EventArgs e)
        {
            if (IsValidIPAddress(TextBoxIPAddress.Text))
            {
                errorProvider1.Clear();
                string username = TextBoxUsername.Text.ToString();
                string password = TextBoxPassword.Text.ToString();

                fConnection = new WebSocket();
                fConnection.Log += Log;
                fConnection.CloseEv += CloseConnection;

                if (fConnection != null)
                {
                    if (!fConnection.Connect(TextBoxIPAddress.Text, "443", "/locationapi/", true, "-", "locationapi", "-", "-", 1, username, password, certificate))
                    {
                        Log("INFO: Unable to connect to " + TextBoxIPAddress.Text);
                    }
                    else
                    {
                        Log("INFO: Connected to " + TextBoxIPAddress.Text);
                    }
                }
            }
            else
            {
                errorProvider1.SetError(BtnConnect, "Please enter a valid IP Address");
            }
        }

        private void quitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        private void TextBoxIPAddress_Validating(object sender, CancelEventArgs e)
        {
            if (!IsValidIPAddress(TextBoxIPAddress.Text))
            {
                TextBoxIPAddress.Text = "";
                errorProvider1.SetError(TextBoxIPAddress, "Invalid IP Address");
            }
            else
            {
                errorProvider1.Clear();
            }
        }
  
        private static bool IsValidIPAddress(string ipAddress)
        {
            try
            {
                IPAddress address = IPAddress.Parse(ipAddress);
            }
            catch (Exception e)
            {
                Console.WriteLine("Exception: " + e.ToString());
                return false;
            }
            return true;
        }

        private void BtnClearLog_Click(object sender, EventArgs e)
        {
            TextBoxOutput.Clear();
        }

        private void BtnSaveLog_Click(object sender, EventArgs e)
        {
            string text = TextBoxOutput.Text;
            Stream saveStream;
            SaveFileDialog saveFileDialog1 = new SaveFileDialog();
            saveFileDialog1.Filter = "txt files (*.txt)|*.txt|All files (*.*)|*.*";
            saveFileDialog1.FilterIndex = 2;
            saveFileDialog1.RestoreDirectory = true;
            if (saveFileDialog1.ShowDialog() == DialogResult.OK)
            {
                if ((saveStream = saveFileDialog1.OpenFile()) != null)
                {
                    byte[] bytes = Encoding.UTF8.GetBytes(text);
                    saveStream.Write(bytes, 0, bytes.Length);
                    saveStream.Close();
                }
            }
        }

        public void CloseConnection()
        {
            if (this.InvokeRequired)
            {
                this.Invoke(new WebSocket.CloseEvent(CloseConnection));
            }
            else
            {
                if (fConnection != null)
                {
                    fConnection.Log -= Log;
                    fConnection.CloseEv -= CloseConnection;
                    fConnection.Close((int)ErrorCodes.Normal, "");
                    fConnection = null;
                }
            }
        }

        private void FrmMainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            CloseConnection();
        }

        private void Log(String aData)
        {
            try
            {
                if (this.InvokeRequired)
                {
                    this.Invoke(new WebSocket.LogEvent(Log), new Object[] { aData });
                }
                else
                {
                    TextBoxOutput.AppendText(aData);
                    TextBoxOutput.AppendText("\r\n");
                }
            }
            catch (ObjectDisposedException ex)
            {
                Console.WriteLine("ObjectDisposedException: " + ex.ToString());
            }
        }

        private void BtnDisconnect_Click(object sender, EventArgs e)
        {
            SendCommand("Close Link");
        }

        private void BtnPing_Click(object sender, EventArgs e)
        {
            SendCommand("Test Location API");
        }

        private void ButGetPBXs_Click(object sender, EventArgs e)
        {
            SendCommand("Get PBXs");
        }

        private void BtnGetExtensions_Click(object sender, EventArgs e)
        {
            SendCommand("Get Extensions");
        }

        private void BtnGetLocations_Click(object sender, EventArgs e)
        {
            SendCommand("Get Locations");
        }

        private void BtnSubscribe_Click(object sender, EventArgs e)
        {
            SendCommand("Extn Registration Subscribe");
        }

        private void BtnUnsubscribe_Click(object sender, EventArgs e)
        {
            SendCommand("Extn Registration Unsubscribe");
        }

        private void BtnSetDynamicLoc_Click(object sender, EventArgs e)
        {
            string extnNum = TextBoxExtnNumber.Text.ToString();
            string locationId = TextBoxLocationId.Text.ToString();
            if (string.IsNullOrEmpty(extnNum) == false && string.IsNullOrEmpty(locationId) == false)
            {
                SendCommand("Set Dynamic Location " + extnNum + " " + locationId);
            }
            else
            {
                Log("INFO: Unable to execute command, extn or location field is empty");
            }
        }

        private void BtnNG911_Click(object sender, EventArgs e)
        {
            SendCommand("NG 911");
        }

        private void BtnNG911Subs_Click(object sender, EventArgs e)
        {
            SendCommand("NG 911 Subscribe");
        }

        public void SendCommand(string command)
        {
            if (fConnection != null && !fConnection.Closed())
            {
                fConnection.SendText(command);
            }
            else
            {
                Log("INFO: Unable to execute command, connection is closed");
            }
        }

        private void BtnSelectCert_Click(object sender, EventArgs e)
        {
            X509Certificate2 certSelected = null;
            X509Store x509Store = new X509Store(StoreName.My, StoreLocation.CurrentUser);
            x509Store.Open(OpenFlags.ReadOnly);

            X509Certificate2Collection col = x509Store.Certificates;
            X509Certificate2Collection sel = X509Certificate2UI.SelectFromCollection(col, "Select Certificate", "Please select a certificate", X509SelectionFlag.SingleSelection);

            if (sel.Count > 0)
            {
                X509Certificate2Enumerator en = sel.GetEnumerator();
                en.MoveNext();
                certSelected = en.Current;
            }

            x509Store.Close();

            if (certSelected != null && certSelected.HasPrivateKey)
            {
                certificate = certSelected;
            }
            else
            {
                MessageBox.Show("Error: invalid certificate, reverting to self signed", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                certificate = null;
            }
        }

        private void FrmMainForm_Load(object sender, EventArgs e)
        {

        }

        private void BtnBadCommand_Click(object sender, EventArgs e)
        {
            SendCommand("Bad Dummy Command");
        }

        private void TextBoxIPAddress_TextChanged(object sender, EventArgs e)
        {

        }
    }
    
    public class WebSocket
    {
        public delegate void LogEvent(string text);
        public delegate void CloseEvent();
        public event LogEvent Log;
        public event CloseEvent CloseEv;
        public string cachedString = String.Empty;
        public string receivedData = String.Empty;
        public MemoryStream cachedBinary = new MemoryStream();
   
        protected TcpClient tcpClient = null;
        protected string cookie = "";
        protected string extension = "";
        protected string origin = "";
        protected string protocol = "";
        protected string host = "";
        protected string port = "";
        protected string resName = "";
        protected int version = 0;
        protected Headers headers;
        protected bool fSsl;
        protected SslStream sslStream;
        protected bool requireMask = false;
        protected bool masking = true;
        protected bool closedByMe = false;
        protected bool closedByPeer = false;

        public WebSocket()
        {

        }
        
        public bool Closed()
        {
            return (closedByMe && closedByPeer) || (tcpClient == null) || !tcpClient.Connected;
        }

        public void Close(int closeCode, string closeReason)
        {
            byte[] bytes;
            MemoryStream ms = new MemoryStream();
            string s = closeReason;
            if (!Closed())
            {
                closedByMe = true;
                if (!closedByPeer)
                {
                    bytes = new byte[2];
                    bytes[0] = (byte)((int)closeCode / 256);
                    bytes[1] = (byte)((int)closeCode % 256);
                    ms.Write(bytes, 0, 2);
                    bytes = Encoding.UTF8.GetBytes(s);
                    while (bytes.Length > 123)
                    {
                        s = s.Substring(0, s.Length - 1);
                        bytes = Encoding.UTF8.GetBytes(s);
                    }
                    ms.Write(bytes, 0, bytes.Length);
                    SendData(true, false, false, false, (int)FrameType.Close, ms);
                }
                Close();
                if (Log != null) Log("INFO: ConnectionClose called for " + host);
                if (CloseEv != null) CloseEv();
            }
        }

        public bool Ping(string aData)
        {
            return SendData(true, false, false, false, (int)FrameType.Ping, aData);
        }

        public bool Pong(string aData)
        {
            return SendData(true, false, false, false, (int)FrameType.Pong, aData);
        }

        public bool SendText(String aString)
        {
            return SendData(true, false, false, false, (int)FrameType.Text, aString);
        }

        protected virtual void Close()
        {
            if (tcpClient.Connected) { tcpClient.Close(); }
        }
        
        protected virtual void ProcessStream(bool aReadFinal, bool aRes1, bool aRes2, bool aRes3, MemoryStream aStream)
        {
            cachedBinary.Write(aStream.ToArray(), 0, (int)aStream.Length);
        }

        protected virtual void ProcessStreamContinuation(bool aReadFinal, bool aRes1, bool aRes2, bool aRes3, MemoryStream aStream)
        {
            cachedBinary.Write(aStream.ToArray(), 0, (int)aStream.Length);
            if (aReadFinal)
            {
                cachedBinary.SetLength(0);
            }
        }

        protected virtual void ProcessText(bool aReadFinal, bool aRes1, bool aRes2, bool aRes3, string aString)
        {
            cachedString = aString;
        }

        protected virtual void ProcessTextContinuation(bool aReadFinal, bool aRes1, bool aRes2, bool aRes3, string aString)
        {
            cachedString += aString;
            if (aReadFinal)
            {
                cachedString = String.Empty;
            }
        }

        protected static byte[] Reverse(byte[] array)
        {
            byte temp;
            int highCtr = array.Length - 1;

            for (int ctr = 0; ctr < array.Length / 2; ctr++)
            {
                temp = array[ctr];
                array[ctr] = array[highCtr];
                array[highCtr] = temp;
                highCtr -= 1;
            }
            return array;
        }

        protected virtual bool SendData(bool aWriteFinal, bool aRes1, bool aRes2, bool aRes3, int aWriteCode, MemoryStream aStream)
        {
            bool result = !Closed() && ((aWriteCode == (int)FrameType.Close) || !closedByMe);
            int bt = 0;
            int sendLen = 0;
            int i;
            long len = 0;
            Stream stream;
            byte[] bytes;
            byte[] masks = new byte[4];
            byte[] send = new byte[65536];
            Random rand = new Random();
            if (result)
            {
                try
                {
                    stream = getStream(tcpClient);

                    
                    bt = (aWriteFinal ? 1 : 0) * 0x80;
                    bt += (aRes1 ? 1 : 0) * 0x40;
                    bt += (aRes2 ? 1 : 0) * 0x20;
                    bt += (aRes3 ? 1 : 0) * 0x10;
                    bt += aWriteCode;
                    stream.WriteByte((byte)bt);

                    len = (masking ? 1 : 0) * 0x80;
                    if (aStream.Length < 126) len += aStream.Length;
                    else if (aStream.Length < 65536) len += 126;
                    else len += 127;
                    stream.WriteByte((byte)len);

                    if (aStream.Length >= 126)
                    {
                        if (aStream.Length < 65536)
                        {
                            bytes = System.BitConverter.GetBytes((ushort)aStream.Length);
                        }
                        else
                        {
                            bytes = System.BitConverter.GetBytes((ulong)aStream.Length);
                        }
                        if (BitConverter.IsLittleEndian) bytes = Reverse(bytes);
                        stream.Write(bytes, 0, bytes.Length);
                    }

                    if (masking)
                    {
                        masks[0] = (byte)rand.Next(256);
                        masks[1] = (byte)rand.Next(256);
                        masks[2] = (byte)rand.Next(256);
                        masks[3] = (byte)rand.Next(256);
                        stream.Write(masks, 0, masks.Length);
                    }

                    aStream.Position = 0;
                    while ((sendLen = aStream.Read(send, 0, send.Length)) > 0)
                    {
                        if (masking)
                        {
                            for (i = 0; i < send.Length; i++)
                            {
                                send[i] = (byte)(send[i] ^ masks[i % 4]);
                            }
                        }
                        stream.Write(send, 0, sendLen);
                    }
                    aStream.Position = 0;
                    var sr = new StreamReader(aStream, Encoding.UTF8 );
                    if (aWriteCode != (int)FrameType.Ping && aWriteCode != (int)FrameType.Pong)
                    {
                        if (Log != null) Log("INFO: ConnectionWrite: " + sr.ReadToEnd());
                    }
                    /*
                    else if (aWriteCode == (int)FrameType.Ping)
                    {
                        if (Log != null) Log("INFO: ConnectionWrite: Sending Ping to " + host);
                    }
                    else if (aWriteCode == (int)FrameType.Pong)
                    {
                        if (Log != null) Log("INFO: ConnectionWrite: Sending Ping response to " + host);
                    }
                    */
                }
                catch
                {
                    result = false;
                }
            }
            return result;
        }

        protected bool SendData(bool aWriteFinal, bool aRes1, bool aRes2, bool aRes3, int aWriteCode, String aData)
        {
            MemoryStream ms = new MemoryStream();
            Encoding outputEnc = new UTF8Encoding(false); // do not include BOM
            StreamWriter sw = new StreamWriter(ms, outputEnc);
            sw.Write(aData);
            sw.Flush();
            return SendData(aWriteFinal, aRes1, aRes2, aRes3, aWriteCode, ms);
        }



        protected void SocketThreadFunc()
        {
            bool final, res1, res2, res3;
            int code, closeCode;
            byte[] closeReasonB;
            string closeReason;
            bool readRes;
            int lastCode = -1;
            int errorCode = -1;
            MemoryStream ms = new MemoryStream();

            if (Log != null) Log("INFO: ConnectionOpen event for " + host);
            SendText("Hello");

            while (true)
            {
                readRes = ReadData(out final, out res1, out res2, out res3, out code, ms);
                if (readRes)
                {
                    ms.Position = 0;
                    errorCode = -1;
                    FrameType fEnum = (FrameType)code;
                    switch (fEnum)
                    {
                        case FrameType.Continuation:
                            if (lastCode == (int)FrameType.Text)
                            {
                                ProcessTextContinuation(final, res1, res2, res3, Encoding.UTF8.GetString(ms.ToArray()));
                            }
                            else if (lastCode == (int)FrameType.Binary)
                            {
                                ProcessStreamContinuation(final, res1, res2, res3, ms);
                            }
                            else
                            {
                                errorCode = (int)ErrorCodes.ProtocolError;
                            }
                            if (final) lastCode = -1;
                            break;
                        case FrameType.Text:
                            ProcessText(final, res1, res2, res3, Encoding.UTF8.GetString(ms.ToArray()));
                            if (!final) lastCode = code;
                            else lastCode = -1;
                            break;
                        case FrameType.Binary:
                            ProcessStream(final, res1, res2, res3, ms);
                            if (!final) lastCode = code;
                            else lastCode = -1;
                            break;
                        case FrameType.Ping:
                            //if (Log != null) Log("INFO: ConnectionRead: Received Ping from " + host);
                            Pong("");
                            break;
                        case FrameType.Pong:
                            break;
                        case FrameType.Close:
                            closeCode = (int)ErrorCodes.NoStatus;
                            closeReason = String.Empty;
                            if (ms.Length > 1)
                            {
                                closeCode = ms.ReadByte() * 256 + ms.ReadByte();
                                if (ms.Length > 2)
                                {
                                    closeReasonB = new byte[ms.Length - 2];
                                    ms.Read(closeReasonB, 0, closeReasonB.Length);
                                    closeReason = Encoding.UTF8.GetString(closeReasonB);
                                }
                            }
                            closedByPeer = true;
                            if (Log != null) Log("INFO: Connection closed by peer, host: " + host);
                            if ((closeCode == (int)ErrorCodes.Normal) && (!closedByMe))
                            {
                                Close((int)ErrorCodes.Normal, String.Empty);
                            }
                            else
                            {
                                Close();
                            }
                            break;
                        default:
                            errorCode = (int)ErrorCodes.DataError;
                            break;
                    }
                    if (errorCode == -1)
                    {
                        ms.Position = 0;
                        var sr = new StreamReader(ms, System.Text.Encoding.UTF8);
                        if (code != (int)FrameType.Ping && code != (int)FrameType.Pong)
                        {
                            if (Log != null)
                            {
                                receivedData += sr.ReadToEnd();
                                if (receivedData.Contains("</response>"))
                                {
                                    try
                                    {
                                        XDocument xdoc = XDocument.Parse(receivedData);
                                        XmlWriterSettings xws = new XmlWriterSettings();
                                        xws.Encoding = System.Text.Encoding.UTF8;
                                        xws.OmitXmlDeclaration = false;
                                        xws.Indent = true;
                                        xws.IndentChars = "    "; // 4 spaces
                                        MemoryStream output = new MemoryStream();
                                        XmlWriter xw = XmlWriter.Create(output, xws);
                                        xdoc.Save(xw);
                                        xw.Flush();
                                        output.Flush();
                                        output.Position = 0;
                                        var sr2 = new StreamReader(output, Encoding.UTF8);
                                        Log("INFO: ConnectionRead: \r\n" + sr2.ReadToEnd());
                                    }
                                    catch (Exception e)
                                    {
                                        Log("INFO: ConnectionRead exception parsing xml for data: \r\n" + receivedData);
                                    }
                                    receivedData = String.Empty;
                                }
                            }
                        }
                        /*
                        else if (code == (int)FrameType.Pong)
                        {
                            if (Log != null) Log("INFO: ConnectionRead: Received Ping response from " + host);
                        }
                        */
                        ProcessData(ms);
                    }
                    else
                    {
                        break;
                    }
                }
                else
                {
                    errorCode = (int)ErrorCodes.DataError;
                    break;
                }
            }
            if (errorCode != -1)
            {
                Close(errorCode, String.Empty);
            }
            ms.Dispose();
        }


        protected bool ReadByte(Stream aStream, out int aByte)
        {
            aByte = aStream.ReadByte();
            return aByte > -1;
        }

        protected bool ReadData(out bool aReadFinal, out bool aRes1, out bool aRes2, out bool aRes3, out int aReadCode, MemoryStream aStream)
        {
            bool result = true;
            bool mask = false;
            int bt, j, k;
            long len, i;
            int[] masks = new int[4];
            byte[] buffer;
            Stream ns;

            aReadFinal = false;
            aRes1 = false;
            aRes2 = false;
            aRes3 = false;
            aReadCode = -1;
            result = !Closed();
            if (result)
            {
                ns = getStream(tcpClient);

                ns.ReadTimeout = Timeout.Infinite;
                try
                {
                    result = ReadByte(ns, out bt);
                    ns.ReadTimeout = 10 * 1000;
                    if (result)
                    {
                        
                        aReadFinal = (bt & 0x80) == 0x80;
                        aRes1 = (bt & 0x40) == 0x40;
                        aRes2 = (bt & 0x20) == 0x20;
                        aRes3 = (bt & 0x10) == 0x10;
                        aReadCode = (bt & 0x0f);

                        result = ReadByte(ns, out bt);
                        if (result)
                        {
                            mask = (bt & 0x80) == 0x80;
                            len = (bt & 0x7F);
                            if (len == 126)
                            {
                                result = ReadByte(ns, out bt);
                                if (result)
                                {
                                    len = bt * 0x100;
                                    result = ReadByte(ns, out bt);
                                    if (result)
                                    {
                                        len = len + bt;
                                    }
                                }
                            }
                            else if (len == 127)
                            {
                                result = ReadByte(ns, out bt);
                                if (result)
                                {
                                    len = bt * 0x100000000000000;
                                    result = ReadByte(ns, out bt);
                                    if (result)
                                    {
                                        len = len + bt * 0x1000000000000;
                                        result = ReadByte(ns, out bt);
                                    }
                                    if (result)
                                    {
                                        len = len + bt * 0x10000000000;
                                        result = ReadByte(ns, out bt);
                                    }
                                    if (result)
                                    {
                                        len = len + bt * 0x100000000;
                                        result = ReadByte(ns, out bt);
                                    }
                                    if (result)
                                    {
                                        len = len + bt * 0x1000000;
                                        result = ReadByte(ns, out bt);
                                    }
                                    if (result)
                                    {
                                        len = len + bt * 0x10000;
                                        result = ReadByte(ns, out bt);
                                    }
                                    if (result)
                                    {
                                        len = len + bt * 0x100;
                                        result = ReadByte(ns, out bt);
                                    }
                                    if (result)
                                    {
                                        len = len + bt;
                                    }
                                }
                            }

                            if ((result) && (requireMask) && (!mask))
                            {
                                Close((int)ErrorCodes.ProtocolError, String.Empty);
                                result = false;
                            }

                            if (result && mask)
                            {
                                result = ReadByte(ns, out masks[0]);
                                if (result) result = ReadByte(ns, out masks[1]);
                                if (result) result = ReadByte(ns, out masks[2]);
                                if (result) result = ReadByte(ns, out masks[3]);
                            }

                            if (result)
                            {
                                aStream.SetLength(0);
                                aStream.Position = 0;
                                ns.ReadTimeout = 1000 * 60 * 60 * 2;
                                buffer = new byte[len];
                                j = 0;
                                while (len > 0)
                                {
                                    k = ns.Read(buffer, j, (int)Math.Min(len, (long)System.Int32.MaxValue));
                                    j += k;
                                    len -= k;
                                }
                                if (mask)
                                {
                                    for (i = 0; i < buffer.Length; i++)
                                    {
                                        buffer[i] = (byte)(buffer[i] ^ masks[i % 4]);
                                    }
                                }
                                aStream.Write(buffer, 0, buffer.Length);
                                aStream.Position = 0;
                            }
                        }
                    }
                }
                catch
                {
                    result = false;
                }
            }

            return result;
        }

        public bool Connect(string aHost, string aPort, string aResourceName, bool aSsl,
          string aOrigin, string aProtocol, string aExtension,
          string aCookie, int aVersion, string username, string password, X509Certificate2 certificate)
        {
            host = aHost;
            port = aPort;
            resName = aResourceName;
            fSsl = aSsl;
            origin = aOrigin;
            protocol = aProtocol;
            extension = aExtension;
            cookie = aCookie;
            version = aVersion;
            headers = new Headers();

            try
            {
                tcpClient = new TcpClient(aHost, int.Parse(aPort));
	
                if (fSsl)
                {
                    sslStream = new SslStream(tcpClient.GetStream(), false, new RemoteCertificateValidationCallback(validateServerCertificate), null);
                    if (certificate != null)
                    {
                        X509CertificateCollection collection1 = new X509CertificateCollection();
                        collection1.Add(certificate);
                        sslStream.AuthenticateAsClient(host, collection1, SslProtocols.Tls, false);
                    }
                    else
                    {
                        sslStream.AuthenticateAsClient(host);
                    }
                }

                Stream stream = getStream(tcpClient);
                StreamReader sr = new StreamReader(stream);
                StreamWriter sw = new StreamWriter(stream);
                string key = "";
                Random rand = new Random();
                String get;
                String line;
                Char[] separator = new char[] { ':' };
                Char[] separator2 = new char[] { ' ' };
                string[] parts;
                SHA1 sha = new SHA1CryptoServiceProvider();

                string authString = username + ":" + password;
                var plainTextBytes = System.Text.Encoding.UTF8.GetBytes(authString);
                string base64authString = System.Convert.ToBase64String(plainTextBytes);
                sw.Write(String.Format("GET {0} HTTP/1.1\r\n", resName));
                sw.Write(String.Format("Connection: Upgrade\r\n"));
                sw.Write(String.Format("Authorization: Basic {0}\r\n", base64authString));
                sw.Write(String.Format("User-Agent: Avaya-IPO-Dynamic-Location\r\n"));
                sw.Write(String.Format("Host: {0}:{1}\r\n", host, port));
                sw.Write(String.Format("Upgrade: websocket\r\n"));
                while (key.Length < 16) key += (char)(rand.Next(85) + 32);
                key = Convert.ToBase64String(Encoding.ASCII.GetBytes(key));
                sw.Write(String.Format("Sec-WebSocket-Key: {0}\r\n", key));
                sw.Write(String.Format("Sec-WebSocket-Version: {0}\r\n", version));
                if (protocol != "")
                    sw.Write(String.Format("Sec-WebSocket-Protocol: {0}\r\n", protocol));
                if (origin != "")
                    sw.Write(String.Format("Sec-WebSocket-Origin: {0}\r\n", origin));
                if (extension != "")
                    sw.Write(String.Format("Sec-WebSocket-Extensions: {0}\r\n", extension));
                if (cookie != "")
                    sw.Write(String.Format("Cookie: {0}\r\n", cookie));
                sw.Write("\r\n");
                sw.Flush();
                
                get = sr.ReadLine();
                if (get.ToLower().IndexOf("http/1.1 101") > -1)
                {
                    do
                    {
                        line = sr.ReadLine();
                        if (!String.IsNullOrEmpty(line))
                        {
                            parts = line.Split(separator, 2);
                            headers.Append(parts[0].ToLower(), parts.Length == 2 ? parts[1] : "");
                        }
                    } while (!String.IsNullOrEmpty(line));

                    if ((headers.Contains("upgrade")) && (headers["upgrade"].Trim().ToLower() == "websocket".ToLower()) && (headers.Contains("connection")) && (headers["connection"].Trim().ToLower().IndexOf("upgrade") > -1))
                    {
                        protocol = "";
                        extension = "";
                        if (headers.Contains("sec-websocket-protocol")) protocol = headers["sec-websocket-protocol"].Trim();
                        if (headers.Contains("sec-websocket-extensions")) extension = headers["sec-websocket-extensions"].Trim();
                        if (headers.Contains("sec-websocket-accept"))
                        {
                            get = headers["sec-websocket-accept"].Trim();
                            key = Convert.ToBase64String(sha.ComputeHash(Encoding.ASCII.GetBytes(key + "258EAFA5-E914-47DA-95CA-C5AB0DC85B11")));
                            if (get == key)
                            {
                                StartRead();
                                return true;
                            }
                        }
                    }
                }
            }
            catch
            {
            }
            try { tcpClient.Close(); }
            catch { }
            return false;
        }
        
        protected virtual bool validateServerCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return true;
        }
        
        protected internal void StartRead()
        {
            var t = new Thread(SocketThreadFunc);
            t.Start();
        }

        protected Stream getStream(TcpClient aClient)
        {
            if (fSsl) return sslStream;
            else return aClient.GetStream();
        }

        protected void ProcessData(MemoryStream ms)
        {
            ms.Position = 0;
            var sr = new StreamReader(ms, System.Text.Encoding.UTF8);
            string data = sr.ReadToEnd();
            if (data.Contains("Test Location API"))
            {
                SendText("Location API Operational");
            }
            else if (data.Contains("Close Link"))
            {
                if (CloseEv != null) CloseEv();
            }
        }

        public class Headers : DictionaryBase
        {
            public String this[String key] { get { return ((String)Dictionary[key]); } set { Dictionary[key] = value; } }
            public ICollection Keys { get { return (Dictionary.Keys); } }
            public ICollection Values { get { return (Dictionary.Values); } }
            public void Add(String key, String value) { Dictionary.Add(key, value); }
            public void Append(String key, String value) { if (this.Contains(key)) this[key] += ',' + value; else this[key] = value; }
            public bool Contains(String key) { return (Dictionary.Contains(key)); }
            public void Remove(String key) { Dictionary.Remove(key); }

            public string ToHeaders()
            {
                string result = String.Empty;
                foreach (DictionaryEntry entry in this)
                {
                    result += entry.Key + ": " + entry.Value + "\r\n";
                }
                if (result != String.Empty)
                    result += "\r\n";
                return result;
            }
        
            protected override void OnInsert(Object key, Object value)
            {
                if (key.GetType() != typeof(System.String))
                    throw new ArgumentException("key must be of type String.", "key");
                if (value.GetType() != typeof(System.String))
                    throw new ArgumentException("value must be of type String.", "value");
            }
        
            protected override void OnRemove(Object key, Object value)
            {
                if (key.GetType() != typeof(System.String))
                    throw new ArgumentException("key must be of type String.", "key");
            }
        
            protected override void OnSet(Object key, Object oldValue, Object newValue)
            {
                if (key.GetType() != typeof(System.String))
                    throw new ArgumentException("key must be of type String.", "key");
        
                if (newValue.GetType() != typeof(System.String))
                    throw new ArgumentException("newValue must be of type String.", "newValue");
            }
        
            protected override void OnValidate(Object key, Object value)
            {
                if (key.GetType() != typeof(System.String))
                    throw new ArgumentException("key must be of type String.", "key");
        
                if (value.GetType() != typeof(System.String))
                    throw new ArgumentException("value must be of type String.", "value");
            }
        }
    }
}
