﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Net;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Serialization;
using System.Windows.Forms.DataVisualization.Charting;

namespace ResourceMonitor
{
    public partial class TrunkStatus : Form
    {
        private TrunkGroup trunkGroup = null;
        private VoIpGroup voipGroup = null;
        private String moduleInfo = "";

        public TrunkStatus(TrunkGroup tg)
        {
            InitializeComponent();
            
            trunkGroup = tg;

            if (tg.CompressionPreferences != null)
            {
                // This should be a VoIP group, if data not embedded retrieve from parent
                if (tg.VoIpGroup == null)
                {
                    String voipCid = trunkGroup.CID.TrimEnd('6');
                    voipCid = voipCid + "4";
                    tg.VoIpGroup = MainForm.Instance.RetrieveData(voipCid, false, false, 100) as VoIpGroup;
                }
            }
            voipGroup = tg.VoIpGroup;

            if (voipGroup == null)
            {
                // Digital trunk group, get the module info
                String[] cid = tg.CID.Split('.');
                if (cid.Length == 6)
                {
                    String moduleCid = cid[0] + "." + cid[1] + "." + cid[2] + ".2";
                    InternalModule im = MainForm.Instance.RetrieveData(moduleCid, false, false, 100) as InternalModule;
                    if (im != null)
                    {
                        moduleInfo = "Slot: " + im.InternalSlotNumber + " Port: " + cid[4];
                    }
                }
            }

            if (trunkGroup.PerformanceHistory != null)
            {
                // Add columns to table for relevant properties
                lvPerformance.Columns.Clear();
                lvPerformance.Columns.Add("Interval Start Time");
                int startItem = 0;
                int endItem = 0;
                int currentItem = 0;
                if (trunkGroup.CircuitType < 4 || trunkGroup.CircuitType > 7)
                {
                    // E1 types
                    startItem = 0;
                    endItem = 7;
                    if (trunkGroup.CircuitType != 1)
                    {
                        endItem = 6;
                    }
                }
                else
                {
                    // T1 types
                    startItem = 8;
                    endItem = 20;
                }

                ErrorRow er = new ErrorRow();
                PropertyInfo[] objInfo = er.GetType().GetProperties();
                if (objInfo.Length > 0)
                {
                    foreach (PropertyInfo pi in objInfo)
                    {
                        if (!pi.PropertyType.IsClass)
                        {
                            if (currentItem++ < startItem)
                            {
                                continue;
                            }
                            lvPerformance.Columns.Add(pi.Name, pi.Name);
                            if (currentItem > endItem)
                            {
                                break;
                            }
                        }
                    }
                }
            }
            else
            {
                tabControl1.TabPages.Remove(tabPage4);
            }
            RefreshView();
        }

        private void RefreshView()
        {
            // Status
            if (voipGroup != null)
            {
                lbLineTypeText.Visible = false;
                lbLineTypeValue.Visible = false;
                lbAddressText.Visible = true;
                lbAddressValue.Visible = true;

                if (voipGroup.IpGroupType == 4)
                {
                    lbSummaryTitle.Text = "H323 Trunk Summary";
                    lbAddressValue.Text = voipGroup.TrunkGatewayAddress;
                }
                else if (voipGroup.IpGroupType == 12)
                {
                    lbSummaryTitle.Text = "IP Office Trunk Summary";
                    lbAddressValue.Text = voipGroup.TrunkGatewayAddress;
                    lbCompressionText.Visible = true;
                    lbCompressionValue.Visible = true;
                    lbCompressionValue.Text = "";
                    foreach (TrunkCodecInfo codec in trunkGroup.CompressionPreferences.TrunkCodecInfo)
                    {
                        if (lbCompressionValue.Text != "")
                        {
                            lbCompressionValue.Text += ", ";
                        }
                        lbCompressionValue.Text += MainForm.Instance.ConvertToEnumerationValue("TrunkCodecInfo::Codec", codec.Codec);
                    }
                    lbFastStartText.Visible = true;
                    lbFastStartValue.Visible = true;
                    lbFastStartValue.Text = trunkGroup.EnableFastStart ? "On" : "Off";
                    lbSilenceSupText.Visible = true;
                    lbSilenceSupValue.Visible = true;
                    lbSilenceSupValue.Text = trunkGroup.SilenceSuppression ? "On" : "Off";
                    lbMediaStreamText.Visible = true;
                    lbMediaStreamValue.Visible = true;
                    lbMediaStreamValue.Text = MainForm.Instance.ConvertToEnumerationValue("VoIpGroup::MediaStream", voipGroup.MediaStream);
                    lbProtocolText.Visible = true;
                    lbProtocolValue.Visible = true;
                    lbProtocolValue.Text = MainForm.Instance.ConvertToEnumerationValue("VoIpGroup::TransportProtocol", voipGroup.TransportProtocol);
                }
                else if (voipGroup.IpGroupType == 5)
                {
                    lbSummaryTitle.Text = "SIP Trunk Summary";
                    lbAddressValue.Text = voipGroup.ResolvedPeerAddress;
                    lbDomainNameText.Visible = true;
                    lbDomainNameValue.Visible = true;
                    lbDomainNameValue.Text = voipGroup.DomainName;
                    lbCompressionText.Visible = true;
                    lbCompressionValue.Visible = true;
                    lbCompressionValue.Text = "";
                    foreach (TrunkCodecInfo codec in trunkGroup.CompressionPreferences.TrunkCodecInfo)
                    {
                        if (lbCompressionValue.Text != "")
                        {
                            lbCompressionValue.Text += ", ";
                        }
                        lbCompressionValue.Text += MainForm.Instance.ConvertToEnumerationValue("TrunkCodecInfo::Codec", codec.Codec);
                    }
                    lbFastStartText.Visible = true;
                    lbFastStartValue.Visible = true;
                    lbFastStartValue.Text = trunkGroup.EnableFastStart ? "On" : "Off";
                    lbSilenceSupText.Visible = true;
                    lbSilenceSupValue.Visible = true;
                    lbSilenceSupValue.Text = trunkGroup.SilenceSuppression ? "On" : "Off";
                    lbMediaStreamText.Visible = true;
                    lbMediaStreamValue.Visible = true;
                    lbMediaStreamValue.Text = MainForm.Instance.ConvertToEnumerationValue("VoIpGroup::MediaStream", voipGroup.MediaStream);
                    lbProtocolText.Visible = true;
                    lbProtocolValue.Visible = true;
                    lbProtocolValue.Text = MainForm.Instance.ConvertToEnumerationValue("VoIpGroup::TransportProtocol", voipGroup.TransportProtocol);
                }
                else if (voipGroup.IpGroupType == 9)
                {
                    lbSummaryTitle.Text = "SM Trunk Summary";
                    lbAddressValue.Text = voipGroup.ResolvedPeerAddress;
                    lbDomainNameText.Visible = true;
                    lbDomainNameValue.Visible = true;
                }
            }
            else
            {
                lbLineTypeText.Visible = true;
                lbLineTypeValue.Visible = true;
                lbLineTypeValue.Text = MainForm.Instance.ConvertToEnumerationValue("TrunkGroup::CircuitType", trunkGroup.CircuitType);
            }
            lbLineNumberValue.Text = trunkGroup.TrunkNumber.ToString() + " " + moduleInfo;
            lbNumberOfChannelsValue.Text = trunkGroup.NumberOfChannels.ToString();
            lbAdministeredChannelsValue.Text = trunkGroup.AdministeredChannels.ToString();
            lbChannelsInUseValue.Text = trunkGroup.ChannelsInUse.ToString();
            lbAdminStateValue.Text = MainForm.Instance.ConvertToEnumerationValue("TrunkGroup::AdminState", trunkGroup.AdminState);

            // Utilization Summary
            if (trunkGroup.CallsByCategory != null)
            {
                lbTitle.Text = "Utilization Summary For Line: " + trunkGroup.TrunkNumber + " " + moduleInfo;
                DateTime st = MainForm.Instance.SwitchTimeToDateTime(trunkGroup.CallsByCategory.CountStartTime);
                lbCounterStartTime.Text = st.ToString();

                // Get the clock time from the formatted start time string
                long startTime = MainForm.Instance.SwitchTimeToClockTime(trunkGroup.CallsByCategory.CountStartTime);

                // Calculate time in milliseconds from start of count
                long timeSinceStartOfCount = 0;

                listView1.Items.Clear();

                TrunkUtilizationCache cache = trunkGroup.CallsByCategory.UtilizationCache;
                if (cache != null)
                {
                    try
                    {
                        // Get the clock time from the formatted start time string
                        long cacheTime = MainForm.Instance.SwitchTimeToClockTime(cache.LogTime);
                        timeSinceStartOfCount = (cacheTime - startTime) / 1000;

                        label2.Visible = true;
                        lbViewAtTime.Visible = true;
                        DateTime dt = MainForm.Instance.SwitchTimeToDateTime(cache.LogTime);
                        lbViewAtTime.Text = dt.ToString();
                    }
                    catch
                    {
                    }

                    // Call chart
                    chart1.Series["Data"].Points.Clear();

                    long total = cache.OutgoingConnectedCalls + cache.IncomingConnectedCalls + cache.IncomingAbandonedCalls;

                    int category = 1;
                    int ptIdx = chart1.Series["Data"].Points.AddXY(category, cache.OutgoingConnectedCalls);
                    DataPoint pt = chart1.Series["Data"].Points[ptIdx];
                    pt.LegendText = CallCategory2Text(category, true) + ": #VALY";

                    category = 2;
                    ptIdx = chart1.Series["Data"].Points.AddXY(category, cache.IncomingConnectedCalls);
                    pt = chart1.Series["Data"].Points[ptIdx];
                    pt.LegendText = CallCategory2Text(category, true) + ": #VALY";

                    category = 3;
                    ptIdx = chart1.Series["Data"].Points.AddXY(category, cache.IncomingAbandonedCalls);
                    pt = chart1.Series["Data"].Points[ptIdx];
                    pt.LegendText = CallCategory2Text(category, true) + ": #VALY";

                    // If there are no calls ensure the chart is still drawn with single color
                    if (total == 0)
                    {
                        ptIdx = chart1.Series["Data"].Points.AddXY(1, 100);
                        pt = chart1.Series["Data"].Points[ptIdx];
                        pt.IsVisibleInLegend = false;
                        pt.Color = Color.Aqua;
                    }

                    chart1.ChartAreas[0].Area3DStyle.Enable3D = true;
                    chart1.ChartAreas[0].Area3DStyle.Inclination = 45;
                    chart1.ChartAreas[0].Area3DStyle.LightStyle = LightStyle.Realistic;
                    chart1.ChartAreas[0].Area3DStyle.WallWidth = 12;
                    chart1.Series["Data"]["PieLabelStyle"] = "Disabled";
                    chart1.Series.Invalidate();

                    // Duration chart
                    chart2.Series["Data"].Points.Clear();

                    // The sum of any durations below 1% will be only 1% of the total
                    byte percentsUsed = 0;
                    bool subOnePercentPresent = false;

                    long totalDuration = cache.OutgoingCallDuration + cache.IncomingRingingCallDuration + cache.IncomingConnectedCallDuration + cache.AbandonedCallDuration;

                    long[] durations = new long[3];
                    durations[0] = cache.OutgoingCallDuration;
                    durations[1] = cache.IncomingRingingCallDuration + cache.IncomingConnectedCallDuration;
                    durations[2] = cache.AbandonedCallDuration;

                    for (int i = 0; i < 3; i++)
                    {
                        category = i + 1;
                        long duration = durations[i];
                        double val = duration > 0 ? ((duration / (double)timeSinceStartOfCount) / trunkGroup.NumberOfChannels) * 100 : 0;
                        if (duration == 0 || val >= 1)
                        {
                            percentsUsed += (byte)Math.Round(val);
                            ptIdx = chart2.Series["Data"].Points.AddXY(category, Math.Round(val) * 2); // Multiply by two to visually distinguish 1 % from < 1 %
                            pt = chart2.Series["Data"].Points[ptIdx];
                            pt.LegendText = CallCategory2Text(category, true) + " call time: " + Math.Round(val) + " %";
                        }
                        else
                        {
                            // Only need a single percentage point to represent all slices below 1
                            if (!subOnePercentPresent)
                            {
                                subOnePercentPresent = true;
                                percentsUsed++;
                            }
                            ptIdx = chart2.Series["Data"].Points.AddXY(category, 1);
                            pt = chart2.Series["Data"].Points[ptIdx];
                            pt.LegendText = CallCategory2Text(category, true) + " call time" + ": < 1 %";
                        }
                    }

                    // Add the idle time point
                    ptIdx = chart2.Series["Data"].Points.AddXY(4, (100 - percentsUsed) * 2);
                    pt = chart2.Series["Data"].Points[ptIdx];
                    pt.LegendText = "Idle time: " + (100 - percentsUsed) + " %";
                    pt.Color = Color.Aqua;

                    chart2.ChartAreas[0].Area3DStyle.Enable3D = true;
                    chart2.ChartAreas[0].Area3DStyle.Inclination = 45;
                    chart2.ChartAreas[0].Area3DStyle.LightStyle = LightStyle.Realistic;
                    chart2.ChartAreas[0].Area3DStyle.WallWidth = 12;
                    chart2.Series["Data"]["PieLabelStyle"] = "Disabled";
                    chart2.Series.Invalidate();

                    chart2.Legends["Legend1"].TextWrapThreshold = 30;
                    chart2.Legends.Invalidate();

                    listView1.SuspendLayout();
                    listView1.Items.Clear();

                    ListViewItem item = new ListViewItem(CallCategory2Text(1, false));
                    item.SubItems.Add(cache.OutgoingConnectedCalls.ToString());
                    TimeSpan ts = TimeSpan.FromSeconds(cache.OutgoingCallDuration);
                    item.SubItems.Add(String.Format("{0:0#}:{1:0#}:{2:0#}", ts.Hours, ts.Minutes, ts.Seconds));
                    item.SubItems.Add("");
                    listView1.Items.Add(item);

                    item = new ListViewItem(CallCategory2Text(2, false));
                    item.SubItems.Add(cache.IncomingConnectedCalls.ToString());
                    ts = TimeSpan.FromSeconds(cache.IncomingConnectedCallDuration);
                    item.SubItems.Add(String.Format("{0:0#}:{1:0#}:{2:0#}", ts.Hours, ts.Minutes, ts.Seconds));
                    ts = TimeSpan.FromSeconds(cache.IncomingRingingCallDuration);
                    item.SubItems.Add(String.Format("{0:0#}:{1:0#}:{2:0#}", ts.Hours, ts.Minutes, ts.Seconds));
                    listView1.Items.Add(item);

                    item = new ListViewItem(CallCategory2Text(3, false));
                    item.SubItems.Add(cache.IncomingAbandonedCalls.ToString());
                    item.SubItems.Add("");
                    ts = TimeSpan.FromSeconds(cache.AbandonedCallDuration);
                    item.SubItems.Add(String.Format("{0:0#}:{1:0#}:{2:0#}", ts.Hours, ts.Minutes, ts.Seconds));
                    listView1.Items.Add(item);
                }

                listView1.AutoResizeColumn(0, ColumnHeaderAutoResizeStyle.HeaderSize);
                listView1.AutoResizeColumn(1, ColumnHeaderAutoResizeStyle.HeaderSize);
                listView1.AutoResizeColumn(2, ColumnHeaderAutoResizeStyle.HeaderSize);
                listView1.AutoResizeColumn(3, ColumnHeaderAutoResizeStyle.HeaderSize);
                listView1.ResumeLayout();
            }
            else
            {
                tabControl1.TabPages.Remove(tabPage2);
            }

            // Alarms
            if (trunkGroup.Alarms != null)
            {
                lvAlarms.Items.Clear();
                label8.Text = "Alarms for Line: " + trunkGroup.TrunkNumber + " " + moduleInfo;

                // Trunk Group alarms first
                if (trunkGroup.Alarms.TrunkGroupAlarm != null)
                {
                    foreach (TrunkGroupAlarm entry in trunkGroup.Alarms.TrunkGroupAlarm)
                    {
                        DateTime dt = MainForm.Instance.SwitchTimeToDateTime(entry.GenericInfo.LastOccurrenceTime);
                        ListViewItem item = new ListViewItem(dt.ToString());
                        item.SubItems.Add(entry.GenericInfo.NumberOfOccurrences.ToString());
                        item.SubItems.Add(MainForm.Instance.ConvertToEnumerationValue("TrunkGroupAlarm::AlarmType", entry.AlarmType));
                        if (entry.GenericInfo.Current)
                        {
                            item.ForeColor = Color.Red;
                        }
                        lvAlarms.Items.Add(item);
                    }
                }

                // Trunk Channel alarms (Channels property for non-analog only exists if alarms present)
                if (trunkGroup.Channels != null && trunkGroup.Channels.ChannelInfoEntry != null)
                {
                    foreach (ChannelInfoEntry ch in trunkGroup.Channels.ChannelInfoEntry)
                    {
                        foreach (TrunkChannelAlarm entry in ch.ChannelInfo.Alarms.TrunkChannelAlarm)
                        {
                            ListViewItem item = new ListViewItem(entry.GenericInfo.LastOccurrenceTime);
                            item.SubItems.Add(entry.GenericInfo.NumberOfOccurrences.ToString());
                            item.SubItems.Add(MainForm.Instance.ConvertToEnumerationValue("TrunkChannelAlarm::AlarmType", entry.AlarmType));
                            if (entry.GenericInfo.Current)
                            {
                                item.ForeColor = Color.Red;
                            }
                            lvAlarms.Items.Add(item);
                        }
                    }
                }

                if (lvAlarms.Items.Count > 0)
                {
                    lvAlarms.AutoResizeColumn(0, ColumnHeaderAutoResizeStyle.ColumnContent);
                    lvAlarms.AutoResizeColumn(1, ColumnHeaderAutoResizeStyle.HeaderSize);
                    lvAlarms.AutoResizeColumn(2, ColumnHeaderAutoResizeStyle.HeaderSize);
                }
                else
                {
                    lvAlarms.AutoResizeColumn(0, ColumnHeaderAutoResizeStyle.HeaderSize);
                    lvAlarms.AutoResizeColumn(1, ColumnHeaderAutoResizeStyle.HeaderSize);
                    lvAlarms.AutoResizeColumn(2, ColumnHeaderAutoResizeStyle.HeaderSize);
                }
            }
            else
            {
                tabControl1.TabPages.Remove(tabPage3);
            }

            // Performance
            if (trunkGroup.PerformanceHistory != null)
            {
                lvPerformance.Items.Clear();
                DateTime dt = MainForm.Instance.SwitchTimeToDateTime(trunkGroup.PerformanceHistory.LastRowCreationTime);
                long interval = trunkGroup.PerformanceHistory.RowCreationInterval;
                long modifier = 0;

                if (trunkGroup.PerformanceHistory.ErrorRows != null)
                {
                    // Newest row is at end of list so work backwards
                    for (int i = trunkGroup.PerformanceHistory.ErrorRows.ErrorRow.Length - 1; i >= 0; i--)
                    {
                        ErrorRow err = trunkGroup.PerformanceHistory.ErrorRows.ErrorRow[i];

                        // Creation time determined by position in list and interval between row creation
                        long seconds = interval * modifier++;
                        TimeSpan value = TimeSpan.FromSeconds(seconds);
                        DateTime time = dt.Subtract(value);
                        ListViewItem item = new ListViewItem(time.ToShortTimeString());

                        // Add items and subitems for each relevant property
                        PropertyInfo[] objInfo = err.GetType().GetProperties();
                        if (objInfo.Length > 0)
                        {
                            foreach (PropertyInfo pi in objInfo)
                            {
                                if (!pi.PropertyType.IsClass)
                                {
                                    if (lvPerformance.Columns.ContainsKey(pi.Name))
                                    {
                                        String s = pi.GetValue(err).ToString();
                                        if (s == "0" || s == "-1")
                                        {
                                            s = ""; // As per SSA, blank if zero (or uninitialized)
                                        }
                                        item.SubItems.Add(s);
                                    }
                                }
                            }
                        }
                        lvPerformance.Items.Add(item);
                    }
                }

                for (int i = 0; i < lvPerformance.Columns.Count; i++)
                {
                    lvPerformance.AutoResizeColumn(i, ColumnHeaderAutoResizeStyle.HeaderSize);
                }
            }
        }

        private String CallCategory2Text(long category, bool shortForm)
        {
            String callType = category.ToString();
            switch (category)
            {
                case 1:
                    return "Outgoing";
                case 2:
                    return "Incoming";
                case 3:
                    if (shortForm)
                    {
                        return "Abandoned";
                    }
                    return "Incoming Abandoned";
            }
            return callType;
        }

        private void button1_Click(object sender, EventArgs e)
        {
            // For refresh use the dataset request with trunk number
            String cid = "trunks/" + trunkGroup.TrunkNumber;
            SSIObject obj = MainForm.Instance.RetrieveData(cid, false, false, 100);
            if (obj is TrunkGroup)
            {
                trunkGroup = obj as TrunkGroup;
                RefreshView();
            }
        }
    }
}
