/* Copyright Avaya LLC. */

#import <Foundation/Foundation.h>
#import "CSCustomizedButton.h"

@protocol CSButtonModuleServiceDelegate;

/**
 * The CSButtonModuleService object is accessible from the CSUser object,
 * providing access to the button customization feature.
 * The CSButtonModuleService object provides a set of APIs that allow
 * application developers to implement button customization features.
 *
 * The client application is notified about list updates on customized buttons
 * reported by the button module service through handlers that the
 * application has registered on the CSButtonModuleService object.
 * These updates may be the result of local operations on the
 * CSButtonModuleService object as well as remote changes initiated by either
 * the service/server or other end users.
 */
@interface CSButtonModuleService : NSObject

/**
 * The delegate responsible for handling button module service events.
 */
@property (nonatomic, weak) id<CSButtonModuleServiceDelegate> delegate;

/**
 * A Boolean value indicating whether the button module service is available.
 *
 * @return YES if the button module service is available, NO otherwise.
 */
@property (nonatomic, readonly) BOOL isServiceAvailable;

/**
 * Returns the list of available customized buttons.
 *
 * @return An NSArray of CSCustomizedButton objects.
 */
- (NSArray *)availableCustomizedButtons;

/**
 * Returns a consolidated button list consisting of customized buttons and
 * non-customized but configured features.
 *
 * The customized location for non-customized but configured features will be
 * the same as their configured location.
 * If there is a customized button at the configured location, then
 * the configured feature's customized location will be the next available
 * free button module location.
 *
 * @return An NSArray of CSCustomizedButton objects.
 */
- (NSArray *)availableConsolidatedButtons;

/**
 * Adds a customized button at location given by user.
 *
 * @param button Informational parameters for the button to be added.
 * @param configuredLocation Location of the button as configured on SMGR.
 * @param customizedLocation Location on the button module where the user wants to add the customized button.
 * @param handler A block to be called when the operation is completed.
 *        @li @c error If the operation was successful this parameter is nil. If an error occurred it
 *        contains an object describing the error.
 */
- (void)addButton:(CSCustomizedButton *)button configuredLocation:(NSInteger)configuredLocation customizedLocation:(NSInteger)customizedLocation completionHandler:(void (^)(NSError *error))handler;

/**
 * Removes the customized button from the button module.
 *
 * @param button Informational parameters for the button to be removed.
 * @param handler A block to be called when the operation is completed.
 *        @li @c error If the operation was successful this parameter is nil. If an error occurred it
 *        contains an object describing the error.
 */
- (void)removeButton:(CSCustomizedButton *)button completionHandler:(void (^)(NSError *error))handler;

/**
 * Moves a customized button to a location given by user.
 *
 * @param button Informational parameters for the button to be moved.
 * @param customizedLocation Location on the button module where the user wants to move the customized button.
 * @param handler A block to be called when the operation is completed.
 *        @li @c error If the operation was successful this parameter is nil. If an error occurred it
 *        contains an object describing the error.
 */
- (void)moveButton:(CSCustomizedButton *)button customizedLocation:(NSInteger)customizedLocation completionHandler:(void (^)(NSError *error))handler;

/**
 * Updates the customized button with personalized label given by user.
 * Existing personalized label can be reset to its default value by passing an empty label.
 *
 * @param label Personalized label for the customized button.
 * @param button Informational parameters for the button to be personalized.
 * @param handler A block to be called when the operation is completed.
 *        @li @c error If the operation was successful this parameter is nil. If an error occurred it
 *        contains an object describing the error.
 */
- (void)setButtonLabel:(NSString *)label button:(CSCustomizedButton *)button completionHandler:(void (^)(NSError *error))handler;

@end

@protocol CSButtonModuleServiceDelegate <NSObject>
@optional

/**
 * Sent to indicate that button module service is available.
 *
 * @param buttonModuleService The button module service instance sending the message.
 */
- (void)buttonModuleServiceDidBecomeAvailable:(CSButtonModuleService *)buttonModuleService;

/**
 * Sent to indicate that button module service is unavailable.
 *
 * @param buttonModuleService The button module service instance sending the message.
 */
- (void)buttonModuleServiceDidBecomeUnavailable:(CSButtonModuleService *)buttonModuleService;

/**
 * Sent when the customized button list is updated.
 *
 * @param buttonModuleService The button module service instance sending the message.
 */
- (void)buttonModuleServiceDidUpdateCustomizedButtonList:(CSButtonModuleService *)buttonModuleService;

/**
 * Sent when the consolidated button list is updated.
 * It is a list consisting of customized buttons and non-customized but configured features.
 *
 * @param buttonModuleService The button module service instance sending the message.
 */
- (void)buttonModuleServiceDidUpdateConsolidatedButtonList:(CSButtonModuleService *)buttonModuleService;

@end
