/**
 * LoginHandler.m
 * Copyright 2014-2015 Avaya Inc.
 * All rights reserved. Usage of this source is bound to the terms described in
 * the file MOBILE_VIDEO_SDK_LICENSE_AGREEMENT.txt, included in this SDK.
 * Avaya – Confidential & Proprietary. Use pursuant to your signed agreement or Avaya Policy.
 */


#import <Foundation/Foundation.h>
#import "AAWGTokenRequest.h"
#import "AppSettings.h"
#import "Logging.h"

#define DATA_BUFFER_SIZE 3000
#define HTTP_OK 200
#define HTTP_NO_CONTENT 204

@interface  AAWGTokenRequest()<NSURLSessionDelegate>


@property (nonatomic, strong) id<AAWGTokenRequestDelegate> delegate;
@property (nonatomic, strong) NSHTTPURLResponse *httpResponse;
@property (nonatomic, strong) NSData *jsonData;
@property (nonatomic, strong) NSMutableData *data;

@end

@implementation AAWGTokenRequest

NSDictionary *jsonPayload;
NSString* loginUrlFormatString = @"%@://%@:%i/%@";

- (void)getAuthenticationToken:(NSString*)server displayName:(NSString*)displayName fromAddress:(NSString*)fromAddress andNotify:(id<AAWGTokenRequestDelegate>)delegate {
    //Reset the data buffer and set the delegate


    self.data = [NSMutableData dataWithCapacity:DATA_BUFFER_SIZE];
    self.delegate = delegate;
    
    NSString* port = [AppSettings getTokenServicePort];
    NSString* protocol = ([AppSettings isTokenServiceSecure]) ? @"https" : @"http";
    NSString* urlPath = [AppSettings getTokenServiceUrlPath];
    NSString*url = [NSString stringWithFormat:loginUrlFormatString, protocol, server, [port intValue], urlPath];
    
    [self buildRequestPayload:displayName andFromAddress:fromAddress];
    
    self.jsonData = [NSJSONSerialization dataWithJSONObject:jsonPayload options:NSJSONWritingPrettyPrinted error:nil];
    
    NSMutableURLRequest *request = [[NSMutableURLRequest alloc] initWithURL:[NSURL URLWithString:url]];
    [request setTimeoutInterval:5.0]; // Test app, so timeout quickly...
    [request setValue:@"application/vnd.avaya.csa.tokens.v1+json" forHTTPHeaderField:@"Content-Type"];
    [request setValue:@"application/vnd.avaya.csa.tokens.v1+json" forHTTPHeaderField:@"Accept"];
    request.HTTPBody = self.jsonData;
    request.HTTPMethod = @"POST";
    
    [Logging logDebug:@"%s Sending AAWGToken request to url %@ with payload: %@,", __PRETTY_FUNCTION__, url, jsonPayload];
    
    NSURLSession *session = [NSURLSession sharedSession];
    NSURLSessionDataTask *task = [session dataTaskWithRequest:request completionHandler:^(NSData * _Nullable data, NSURLResponse * _Nullable response, NSError * _Nullable error) {
        
        if (response) {
            [self parseResponse:data andResponse:response];
        } else if (error) {
            [Logging logError:@"%s Error: %@ %@", __PRETTY_FUNCTION__,
             [error localizedDescription],
             [[error userInfo] objectForKey:NSURLErrorFailingURLStringErrorKey]];
            [self.delegate getAAWGTokenFailed:error];
        }
    }];
    [task resume];
}

- (void)parseResponse:(NSData *)data andResponse:(NSURLResponse *)response
{
    NSHTTPURLResponse *httpResponse = (NSHTTPURLResponse *)response;
    self.httpResponse = httpResponse;
    [self.data setLength:0];
    [self.data appendData:data];
    
    long statusCode = [self.httpResponse statusCode];
    if (statusCode == HTTP_OK) {
        NSMutableDictionary *tokenDict = [NSJSONSerialization JSONObjectWithData:self.data options:0 error:nil];
        NSString *token = [tokenDict objectForKey:@"encryptedToken"];
        [self.delegate getAAWGTokenSuccessful:token];
    } else {
        //Notify the delegate of the error
        NSDictionary *errorDictionary = @{ NSLocalizedDescriptionKey : [NSString stringWithFormat:@"Unexpected HTTP response code: %li", statusCode]};
        
        NSError *error = [[NSError alloc] initWithDomain:@"AvayaOceanaWebRTC Token Request"
                                                    code:1 userInfo:errorDictionary];
        [self.delegate getAAWGTokenFailed:error];
    }
}

- (NSDictionary*)buildRequestPayload:(NSString*)displayName andFromAddress:(NSString*)fromAddress
{
    NSMutableDictionary *requestPayload = [[NSMutableDictionary alloc]init];
    
    [requestPayload setObject:@"csaGuest" forKey:@"use"];
    [requestPayload setObject:displayName forKey:@"displayName"];
    [requestPayload setObject:fromAddress forKey:@"callingNumber"];
    [requestPayload setObject:@"120000" forKey:@"expiration"];
    
    jsonPayload = [NSDictionary dictionaryWithDictionary:requestPayload];
    
    return jsonPayload;
}

@end
